;******************************************************************************
;* file papcoadd_step_mag
;*      papcoadd_step_mag
;* end
;*
;* this files contains all the collected routines which have to be added in
;* for step_mag. 
;*
;* papco is written such that if the files starting with "papcoadd" exist
;* in the papco_MODULES directory
;* then the routines contained therin are utilised to add that plot type
;* into the papco package. This ensures easy extendability of papco; also makes
;* papco "configurable" by adding or leaving out specific Plot types.
;*
;* THIS ROUTINE AND THE AUXILLIARY READ AND PLOT ROUTINES MUST BE INCLUDED
;* IN THE FILE papco_add_init.pro IN papco_MODULES SO THAT THEY CAN BE
;* COMPILED AT STARTUP.
;*
;* If the plot type is in itself a submenu containing more step_mags, then
;* these must be listed at the top of this file, as it is read by papco.
;* the entries for each sub-plotype must come after the line 
;* "file papcoadd_step_mag" and before the "end" line, one entry per line.
;* Each of the routines here must then be dublicated for each sub-plotype
;* used: see papcoadd_crres_plots for an example.
;*
;* Apart from editing this template the user has to write a panel editor 
;* widget for  step_mag. Use the file papco_panel_template.pro in $papco.
;* If slice functions are to be supported a slice program needs to be
;* written. Use template papco_slice_template in $papco.
;*
;* Finally, a data read and data plot routine are required aswell. The data
;* read will be specific for each problem. An example for a standard
;* line plot is given in the file p_step_mag.pro in $papco. This can also be
;* used as a basis for more complex plots like color spectrograms etc.
;*
;* The Way these add-in routines work:
;* ===================================
;*
;* At the startup of papco, the directory $papco_MODULES is searched for all
;* files beginning with "papcoadd_". From the names of these files the
;* step_mag is stripped off and used to construct the procedure name
;* "addplottype_step_mag" - this procedure must be contained in the file
;* papcoadd_step_mag and is used to add the button for this step_mag to
;* the papco main window. Since "addplottype_step_mag" simply returns
;* widget ID of the button the button itself can contain further submenus
;* for plot sub-types (e.g. the button for SWE can contain a menu of buttons
;* for a range of SWE plots, ie meoments, spectra, etc)
;* To inform papco of all these sub-plot types (and to avoid having to
;* write a seperate file for each sub plot type) all the types have to
;* be listed in the header of the file papcoadd_step_mag.pro. These entries
;* have to finish off with the line ";* end" - see papcoadd_crres_plots.pro
;* for an example. All step_mag names are stored in the string array
;* step_mags and are placed in a common block:
;* COMMON ADDINS, step_mags
;*
;* If there are no sub plot types, then the step_mag as used in the filename
;* for papcoadd_step_mag.pro is used to construct the calls to the procedures
;* in that file. The convention for naming these procedures is
;* "procedure_name_step_mag" , so that papco can always construct the right
;* procedure call when needed. 
;*
;* When there are sub plot types, their names get used to construct the
;* required calls. There are 6 procedured that need to be called for each
;* step_mag, so that a set of these 6 procedures has to be written for each
;* sub plot type. In the same way, you need to write a panel editor for each
;* plot type (sometimes several plot sub types may share the same editor,
;* e.g. multiple line plots or color spectra, see papcoadd_crres_plots).
;* 
;* In papco, whenever an action for a step_mag is taken, the information
;* stored in the structure widgetData is queried. In this structure there
;* is an array of PlotInfo structures, which contain all the relevant 
;* information for a chosen panel. The field plotInfo.panelkind is loaded
;* with the name of the step_mag. This is set by the user in this file in
;* the routine addpanel_step_mag ( "panelKind='step_mag'" ). This MUST
;* be set correctly as all the routine names used in this add-in file are
;* constructed using this infromation!
;*
;* Even if the papcoadd files are in the $papco_MODULES directory and are
;* successfully parsed by papco, they can only be called if the
;* corresponding add-in files have also been included in the
;* initialisation file papco_add_init.pro. If not, papco won't crash, but
;* that plot type will not be available.
;*
;* Currently there are two ways in which step_mags are included in papco.
;* The preferred way is thorugh an add-in file such as this one. However,
;* originally the first step_mags were "hardwired" into papco. This is
;* in the process of being untangled into modules. 
;* In the meantime, whenever papco has to undertake some action for a
;* step_mag, the field plotInfo.panelkind is compared to the entries in the
;* (COMMON ADDINS) step_mags array. If a match is found, the required
;* procedure name is constructed and called, if not, control passes on to
;* the built in case statement which handles the built-in step_mags.
;*
;*
;*    Those procedures are contained:
;*
;*    PRO addplottype_step_mag,plottypes_base,pb_plottype
;*        ->  adds the button for the new plot type to the main window.
;*
;*    PRO plottable_step_mag, plottable
;*        -> Sets a variable to indicate plottable without orbit.
;*
;*    PRO descriptor_step_mag, plotInfo, descriptStr
;*        -> Constructs the string used to label plots in the panels window
;*
;*    PRO addpanel_step_mag
;*        -> Calls the panel editor for this plottype and adds the plot to the
;*            panel window.
;*
;*    PRO editor_step_mag, plotNumber, newPlotData
;*        -> This procedure calls the panel editor when called from 
;*	     papco_EditPlot (papco_paneledit.pro).
;*
;*    PRO  draw_step_mag, panelVector, currPlotInfo, get_data_call, IsRead
;*        -> This procedure does the actual data read call and the plot call
;*           for this plottype
;******************************************************************************

;******************************************************************************
;* PROCEDURE:     
;*      PRO addplottype_step_mag,plottypes_base,buttonGeometry,pb_plottype
;* 
;* DESCRIPTION:  
;*	This procedure adds a button for this plottype to the plottypes_base.
;*      The UVALUE declared here is critical, since it is used to construct
;*      the procedure names of all procedures contained in this add-in
;*	papckage, so that they can be automatically called when needed.
;*
;* INPUTS:       
;*	plottypes_base	the widget ID of the parent widget to contain the
;*			plottype buttons.
;*
;* OUTPUTS: 
;*   	pb_plottype	the widget ID of the button created here.
;*      buttonGeometry  size of the button. Only used in simple
;*                      WIDGET_BUTTON to make them the same size as the
;*                      compound CW_PDMENU, pulldown for subtypes.
;*
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	addplottype_step_mag,plottypes_base,buttonGeometry,pb_plottype
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
pro addplottype_step_mag,plottypes_base,buttonGeometry,pb_plottype

  MenuDesc = [ $
               { CW_PDMENU_S, 3, strcenter('STEP 6.4',10) }, $ 
               { CW_PDMENU_S, 0, 'configure STEP 6.4' }, $
               { CW_PDMENU_S, 2, 'STEP 6.4 MAG' } $
             ]  

  pdm_step_mag = CW_PDMENU(plottypes_base, MenuDesc, $
                            UVALUE='pdm_step_mag')

END



;******************************************************************************
;* PROCEDURE:     
;*      PRO addpanel_step_mag
;*
;* DESCRIPTION:  
;*	Calls the panel editor for this plottype and adds the plot to the
;*      panel window.
;*      using an instance of the papco_PLOTINFO structure called "defaults"
;*      the default settings of the panel editor can be controlled.
;*      panelEditor must be set to the name of the panel editor routine
;*
;* INPUTS:       
;*	event		event value of papco main window. This is
;*                      only used if there are plot sub_types, 
;*                      for example see papcoadd_crres_plots
;*
;* OUTPUTS: 
;*   	none
;*
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	addpanel_step_mag
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
pro addpanel_step_mag,event

  common get_error, get_err_no, get_err_msg

  panelKind='step_mag'

  r_module_config, panelKind, config_data, msg
  if (get_err_no ne 0) and (event.value ne 1) then begin
    message,'You need to configure the module first', /cont
    papco_module_configure, panelKind
    return
  endif  

  CASE event.value of
    1: BEGIN
      papco_module_configure, panelKind
    end
    2: BEGIN
      panelEditor='step_mag_panel'
      defaults=papco_getPlotInfoStruct()
      defaults.typeVector=[0,0,0,0]
      papco_Add_Plot, panelEditor, panelKind, initialPlotInfo=defaults
    end
    else: begin
      message,'Nothing defined for this submenu entry',  /cont
      return
    end  
  endcase

END

;******************************************************************************
;* 
;* PROCEDURE:     
;*      PRO plottable_step_mag, plottable
;*
;* DESCRIPTION:  
;*	Sets a variable to indicate plottable without orbit.
;*
;* INPUTS:       
;*	none
;*
;* OUTPUTS: 
;*   	plottable	a boolean 0 if plottable only by CRRES orbit
;*                                1 if plottable by time also.
;*
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	plottable_step_mag, plottable
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
pro plottable_step_mag, plottable

   plottable=1

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO descriptor_step_mag, plotInfo, descriptStr
;*
;* DESCRIPTION:  
;*	Constructs the string used to label plots in the panel window
;*
;* INPUTS:       
;*	plotInfo        structure containing all data for this plot.
;*                      (structure type: papco_PLOTINFO, 
;*                                             see papco_variables.text) 
;*
;* OUTPUTS: 
;*   	descriptStr	a string containing the label
;*
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	descriptor_step_mag, plotInfo, descriptStr
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
pro descriptor_step_mag, plotInfo, descriptStr

   COMMON STEP_INFO, step_regions, step_observers

   descriptStr=''
   typeNames=['H component ', 'Z component ', 'X component ', $
           'Y component ', 'F component ', 'Declination ']
   plotType=strtrim(typeNames(plotInfo.typeVector(2)), 2)
   reg=plotInfo.typeVector(0)
   obs=plotInfo.typeVector(1)
   
   step_read_observers
   descriptStr=descriptStr+plotType + ', '+ step_regions(reg).region+'/'+$
     step_observers(obs).id + ' - '+step_observers(obs).description

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO editor_step_mag, plotNumber, newPlotData
;* 
;* DESCRIPTION:  
;*	This procedure calls the panel editor when called from 
;*	papco_EditPlot (papco_paneledit.pro).
;*
;* INPUTS:       
;*	plotNumber	anInteger. 0<=plotNumber<=widgetData.numberOfPlots
;*			the index of the plot in widgetData.plotInfos.
;*		
;* OUTPUTS: 
;*   	newPlotData     the new structure returned by the panel editor
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	editor_step_mag, plotNumber, newPlotData
;*
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
PRO editor_step_mag, plotNumber, newPlotData
  
  COMMON PLOT_COMPOSER, widgetData

   plotInfo=widgetData.plotInfos(plotNumber)
   newPlotData=step_mag_panel(plotInfo, $
        ACTION='Update '+plotInfo.panelKind, $
	GROUP=widgetData.main_base)

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO draw_step_mag, panelVector, currPlotInfo, orbitNo $
;*                          get_data_call, slice_type, IsRead
;* DESCRIPTION:  
;*	This procedure performs the actual data read call and the plot call
;*      for this plottype.
;*      
;*
;* INPUTS:       
;*	panelVector	position of the plot (see papco_conventions.text, 6)
;*	currPlotInfo	structure containing all data for this plot.
;*                      (structure type: papco_PLOTINFO, 
;*                                             see papco_variables.text) 
;*
;* OUTPUTS: 
;*   	get_data_call	a string containing the data call. Used by other
;*                      routines that need to know how to get the data
;*                      in a panel, e.g. to draw spectra, distribution functions
;*      IsRead		a boolean which is set when the the data is read the
;*                      first time. Useful if a given data read puts data
;*                      into commons which can be plotted in many ways.
;*                      When set, data does not have to be read again.
;*      slice_type      Selects which slices are supported and for which
;*                      slice routines have been written.
;*                      This is a byte logical with 8 swittches. Currently
;*                      Three are supported (this may be extended later)
;*                       1 = mouse zooms (default, always possible)
;*                       2 = mouse slices
;*                       4 = mouse calls level zero
;*                       8, 16, 32, 64, 128 = not supported yet
;*                      you can support more than one type of slice
;*                      by arythmetically adding the options, eg
;*                      slice_type=5  support zoom and level zero
;*
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	draw_step_mag, panelVector, currPlotInfo, orbitNo, $
;*                                 get_data_call, slice_type, IsRead
;* MODIFICATION HISTORY:       
;*     written November 1995, Reiner Friedel
;******************************************************************************
pro draw_step_mag, panelVector, currPlotInfo, orbitNo, $
                   get_data_call, slice_type, IsRead
  
  COMMON STEP_INFO, step_regions, step_observers
  
  step_read_observers            
  region=STRLOWCASE(step_regions(currPlotInfo.typeVector(0)).region)
  observer=STRLOWCASE(step_observers(currPlotInfo.typeVector(1)).id)
  
  get_data_call="r_step_mag2, '" + region + "', '" + observer + "'"
  
  print,IsRead,size(IsRead) 
  IF NOT IsRead THEN BEGIN
    r_step_mag2, region, observer 
    
    IF papco_Draw_CheckResult(currPlotInfo.panelKind, $
                              panelVector, orbitNr, $
                              currPlotInfo.typeVector, $
                              currPlotInfo.channel) $
      THEN BEGIN
      slice_Type=1
      IsRead=0
      p_step_mag2, panelVector, currPlotInfo.typeVector(2)+1, $
        _EXTRA=extra_PlotInfo
    ENDIF
  ENDIF ELSE BEGIN
    slice_Type=1
    p_step_mag2, panelVector, currPlotInfo.typeVector(2)+1, $
      _EXTRA=extra_PlotInfo
  ENDELSE
  
END

