;******************************************************************************
;******************************************************************************
;* FILE:
;*    $papco_PATH/crres_panel_template.pro
;* 
;*    This file belongs to the papco-project. See $papco_PATH/papco.pro for
;*    further information.
;*
;* DESCRIPTION: 
;*    This file contains the panel-editor for CRRES-plots - that is, 
;*    CRRES-flux, CRRES-spec, CRRES-pitch
;*
;* USAGE OF THE FIELDS IN THE papco_PLOTINFO-STRUCTURE ('aPlotInfo'):
;*    copied from papco_variables.text
;*
;*		    For 'STEP' - plots, three elements of typeVector are used.
;*
;*		    Step-plots need a region-name (e.g. 'ML' for middle
;*		    latitude) and an observer's abbreviation (e.g. 'BOU' for
;*		    the station in Boulder).
;*
;*		    As I was to lazy to type them all in by myself, I took the
;*		    file 'dirinfo.txt' from the STEP-CD, edited it, renamed it
;*		    to 'lepa.info' and put it into $papco_PATH.
;*
;*		    The descriptions are read using the Procedure
;*		    'step_read_observers', which defines the COMMON-block
;*		        COMMON STEP_INFO, step_regions, step_observers 
;*		    The common-variable step_regions is an array of
;*		    region-names (stored in STEP_REGION-structure-instances)
;*		    and the common-variable step_observers is an array of
;*		    observer-names (stored in STEP_OBSERVER-structure-
;*		    instances).  For info about those structures read on in
;*		    this file, $papco_PATH/papco_variables.text, and for the
;*		    'step_read_observers' - procedure read the file
;*		    $papco_PATH/step_mag_panel.pro.
;*
;*		    typeVector(0) this is the index of the region's name.
;*				  The name of the region is:
;*				  step_regions(aPlotInfo.typeVector(0)).region
;*
;*		    typeVector(1) this is the index of the observer's name.
;*				  The name of the observer is:
;*				  step_observers(aPlotInfo.typeVector(1)).id
;*
;*		    typeVector(2) this field stores the type of plot:
;* 				   0 = H component 
;*				   1 = Z component 
;*				   2 = X component 
;*				   3 = Y component 
;*				   4 = F component 
;*				   5 = Declination
;*
;*
;* FILES YOU MAY WANT TO READ
;*       $papco_PATH/papco_conventions.text     which conventions were used
;*						when developing ?
;*	 $papco_PATH/papco_variables.text	what structures and global
;*						variables are used by 
;*						which file ?
;*
;* COMMON-BLOCKS:
;*    DEPLO_PANEL	contains the window-information
;*
;* STRUCTURES:
;*    This module creates one concrete structure, that inherits from the 
;*    papco_PANEL_EDITOR-structure (see $papco_PATH/papco_panel.pro for a
;*    description). 
;*       {DEPLO_PANEL,...
;*	     ; imagine the papco_PANEL_EDITOR-structure was inserted...
;*      
;*    the additional fields are :
;*           {helpFile:'crres_panel.help', $
;*                    typeVector:aPlotInfo.typeVector, $
;*                    channel:aPlotInfo.channel, $
;*                    slider_displaysEnergy:0, $
;*                    lbl_energy_Pitch:lbl_energy_Pitch, $
;*                    sl_energy_pitch:sl_energy_Pitch, $
;*                    spb_integral:spb_integral, $
;*                    spb_differential:spb_differential, $
;*                    spb_energy:spb_energy, $
;*                    sl_channel:sl_channel,$
;*                    lbl_channel:lbl_channel}
;*  
;*
;* OVERVIEW:
;*    This function creates the editor-window:
;*       FUNCTION  crres_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;*    You might try this:
;*	 print, deplo_panel(papco_getPlotInfoStruct())
;*
;* MODIFICATION HISTORY:       
;*     august 1995, written by A.Keese
;******************************************************************************
;******************************************************************************

;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  step_mag_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor for a STEP-panelplot.
;*
;* INPUTS:       
;*	aPlotInfo	a structure of the papco_PLOTINFO-type
;*			this structure is edited in the window.
;*     
;* KEYWORDS:
;* 	GROUP		a WIDGET_BASE
;*			when specified, this is used as GROUP_LEADER
;*	ACTION		a string
;*			when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      For an enhanced description read $papco_PATH/papco_variables.text
;*
;* CALLING SEQUENCE:
;*	aPlotInfoStruct=Step_Panel(papco_getPlotInfoStruct())
;*      print, step_mag_panel(aPlotInfoStruct)
;*
;* MODIFICATION HISTORY:       
;*     written august 1995, Andreas Keese
;******************************************************************************
FUNCTION  step_mag_panel, aPlotInfo, GROUP=group, ACTION=ACTION
  
  COMMON STEP_PANEL_DATA, widgetData
  COMMON STEP_INFO, step_regions, step_observers
   common get_error, get_err_no, get_err_msg

  IF NOT KEYWORD_SET(GROUP) THEN GROUP=0
  IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

  step_read_observers
  IF get_err_no NE 0 THEN return, {CANCELED:1}

; --- validate the info in 'aPlotInfo -----------------------------------------
  tv=aPlotInfo.typeVector
  tv=tv > 0
  tv(0)=tv(0) < N_ELEMENTS(step_regions)-1
  tv(1)=tv(1) < N_ELEMENTS(step_observers)-1
  tv(2)=tv(2) < 5

; create the base-window   
  additionalWData={canceled:1}
  panel_base=papco_WIDGET_BASE(title=aPlotInfo.panelKind, column=1, $
                                GROUP_LEADER=group, /MODAL)

;--- the first part of the window will consist in the widgets -----------------
;    that control the height of that plot
  widgets=papco_panel_AddHeightWidgets(panel_base, aPlotInfo)
  additionalWData=create_struct(widgets, additionalWData)

; --- Add the list-widgets ----------------------------------------------------
  outerList_Base=widget_base(panel_base, COLUMN=1, /FRAME)
  lbl=WIDGET_LABEL(outerList_base, VALUE='   '+ $
                   strformat('Region', /LEFT, 33)+'Observer', $
                   /ALIGN_LEFT)
  list_base=WIDGET_BASE(outerList_Base, ROW=1)

  lb_regions=WIDGET_LIST(list_base, xsize=24, $
                         ysize=N_ELEMENTS(step_regions), $
                         value=step_regions.region+' = '+$
                         step_regions.description, $
                         uvalue='lb_regions')
  lb_observers=WIDGET_LIST(list_base,  xsize=48, ysize=20, $
                           uvalue='lb_observers')

; --- Choose-Plottype widgets -------------------------------------------------
   outer_Plot_Type_Base=WIDGET_BASE(panel_base, COLUMN=1, /FRAME)
   lbl=WIDGET_LABEL(outer_Plot_Type_base, VALUE='Select plot type')
   plotType_Base=WIDGET_BASE(outer_Plot_Type_Base, ROW=2, /EXCLUSIVE)
   typeNames=['H component               ', $
              'Z component              ',  $
              'X component',                $
              'Y component               ', $
              'F component              ',  $
              'Declination']
   spb_plotTypes=lonarr(N_ELEMENTS(typeNames))
   FOR i=0, N_ELEMENTS(typeNames)-1 DO $
      spb_plotTypes(i)=WIDGET_BUTTON(plotType_base, $
			VALUE=typeNames(i), $
			UVALUE='spb_plotTypes'+strtrim(string(i), 2))
   WIDGET_CONTROL, spb_plotTypes(aPlotInfo.typeVector(2)), /SET_BUTTON

;--- add the widgets for choosing the plot-style ------------------------------
   widgets=papco_panel_add_PlotStyleButtons(panel_base, aPlotInfo, 0)
   additionalWData=create_struct(widgets, additionalWData)
   
;--- now, add the widgets for choosing the y-axis-ranges ----------------------
   widgets=papco_PANEL_AddYScale_Widgets(panel_base, aPlotInfo)
   additionalWData=create_struct(widgets, additionalWData)
   
;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window.
   widgets=papco_panel_AddActionButtons(panel_base, Action, aPlotInfo,/CONFIG)
   additionalWData=create_struct(widgets, additionalWData)
   obs_ID=step_observers(aPlotInfo.typeVector(1)).id ; this is the id of the 
                                                     ; chosen observer

   thisWidgetsData={region_Index:aPlotInfo.typeVector(0), $
                    plotType:aPlotInfo.typeVector(2), $
                    observer_ID:obs_ID, $
                    helpfile:'step_mag_panel.help', $
                    lb_regions:lb_regions, $
                    lb_observers:lb_observers}

   widgetData=papco_PANEL_CreateWidgetData(thisWidgetsData, additionalWData)

;--- realize the window
   papco_Panel_Realize, panel_base, widgetData
   step_mag_panel_refresh

   papco_xmanager, 'step_mag_panel', panel_base

;--- return Data --------------------------------------------------------------
   observer_Index=WHERE(step_observers.id EQ widgetData.observer_ID)
   typeVector=[widgetData.region_Index, $
               observer_Index(0), $
               widgetData.plotType, $
               aPlotInfo.typeVector(3)]

   returnData={typeVector:typeVector}

   additionalRData=papco_panel_ReturnData(widgetData)

   return, create_struct(ReturnData, additionalRData)
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO step_mag_panel_refresh,LB_REGIONS_Event=lb_Regions_Event, $
;*                     SET_OBSERVER_INDEX=set_observer_index
;* 
;* DESCRIPTION:  
;*      update the display of the editor-widgets
;* 
;* INPUTS:       
;*      none
;* 
;* KEYWORDS:
;*      only thge widgets corresponding to the set keyword will be updated
;*
;* CALLING SEQUENCE:
;*      have a look at step_mag_panel_event
;*
;* MODIFICATION HISTORY:       
;*      september 1995, written by A.Keese 
;******************************************************************************
PRO step_mag_panel_refresh,LB_REGIONS_Event=lb_Regions_Event, $
                       SET_OBSERVER_INDEX=set_observer_index
   
   COMMON STEP_PANEL_DATA, widgetData
   COMMON STEP_INFO, step_regions, step_observers
   
   IF NOT KEYWORD_SET(lb_regions_event)  THEN $
     WIDGET_CONTROL, widgetData.lb_regions, $
                     SET_LIST_SELECT=widgetData.region_Index

   id_region=step_regions(widgetData.region_Index).region
   tmp_obs=step_observers(WHERE(step_observers.region EQ id_region))
   str_list=strarr(N_ELEMENTS(tmp_obs))
   FOR i=0, N_ELEMENTS(tmp_obs)-1 DO BEGIN
      IF tmp_obs(i).appears_on_CD eq 1 THEN cd_string=' CD 1'
      IF tmp_obs(i).appears_on_CD eq 2 THEN cd_string=' CD   2'
      IF tmp_obs(i).appears_on_CD eq 3 THEN cd_string=' CD 1,2'
      
      str_list(i)=tmp_obs(i).id + ' - ' + $
        strformat(tmp_obs(i).description, 16)+ $ 
        string(tmp_obs(i).lat,tmp_obs(i).long, $
               FORMAT='("lat:", F6.1, " long:",F6.1)') + cd_string
   ENDFOR


   IF N_ELEMENTS(SET_OBSERVER_INDEX) GT 0 THEN BEGIN
      widgetData.observer_ID=tmp_obs(set_observer_index).id
   ENDIF ELSE BEGIN
      WIDGET_CONTROL, widgetData.lb_observers, $
        SET_VALUE=str_list

      observer_Index=WHERE(tmp_obs.id EQ widgetData.observer_ID, count)
      IF count EQ 0 THEN BEGIN
         observer_Index=0
         widgetData.observer_ID=tmp_obs(0).id 
      ENDIF ELSE BEGIN
         observer_Index=observer_Index(0)
      ENDELSE
      WIDGET_CONTROL, widgetData.lb_observers, SET_LIST_SELECT=observer_Index
   ENDELSE

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO step_mag_panel_event, event
;* 
;* DESCRIPTION:  
;*	This processes events from the XMANAGER.
;*
;* INPUTS:       
;*	event		an XManager event
;*			
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	called by XMANAGER
;*
;* MODIFICATION HISTORY:       
;*     written august 1995, Andreas Keese
;******************************************************************************
PRO step_mag_panel_event, event
   COMMON STEP_PANEL_DATA, widgetData
   WIDGET_CONTROL,GET_UVALUE=uval, event.id

   IF strleft(uval, 13) EQ 'spb_plotTypes' THEN BEGIN
      widgetData.plotType=fix(strfrom(uval, 13))
      return
   ENDIF

   CASE uval OF
      'lb_regions' : BEGIN
         widgetData.region_Index=event.index
         widgetData.observer_ID=''
         step_mag_panel_refresh, /LB_Regions_Event
      END

      'lb_observers': BEGIN
         step_mag_panel_refresh, SET_OBSERVER_INDEX=event.index
         IF event.clicks GT 1 THEN BEGIN
            event.id=widgetData.pb_done ; fake a pressed 'Done'-Button
            widgetData=papco_panel_Event(event, widgetData)
         ENDIF
      END

      ELSE: $
        widgetData=papco_panel_Event(event, widgetData)
   ENDCASE 
END
