;******************************************************************************
;******************************************************************************
;* FILE:	 
;*    step_read_observers.pro
;* 
;* DESCRIPTION:  
;*    This file contains a function for reading STEP-observer-descriptions
;*    These observer-descriptions consist of an abbreviation for an 
;*    an observer and its description
;*
;*    The descriptions are read from the file $papco_PATH/step_info.txt
;*
;* FILE-FORMAT of step_info.txt:
;*      this file is the file 'dirinfo.txt' from the main-directory of the
;*      STEP-CDROM. It has been  edited, so that it has this format:
;*
;*        - the first n lines are the region - names. They have this format:
;*          regionID = description
;*        - the regions are followed by the line '*****'
;*        - now, the observers on CD 1 are listed in this format:
;*          region    description    id     lat    long
;*                ^^^            ^^^    ^^^     ^^^ 
;*                at least three blanks between the fields !
;*        - the observers on CD 2 are seperated from the observers on
;*          CD 1 by a line of stars: '*****'. 
;*          the observer-descriptions for CD 2 have the same format as
;*          observer-descriptions above that '*****'-line
;*
;*      CAUTION ! The file is expected to contain no Tab-stops. You can
;*      achieve this by loading the file into the emacs, then marking all of
;*      the file and then pressing META-x and executing the command 'untabify'.
;* 
;* COMMON-BLOCKS:
;*      COMMON STEP_INFO, step_regions, step_observers
;*
;*	step_regions    is an array of instances of the STEP_REGION-structure.
;*                      Here, abbreviations for regions and their
;*			descriptions are stored.
;*
;*      step_observers  is an array of STEP_OBSERVER-instances.
;*			it stores a list of observers and their descriptions
;*			
;* STRUCTURES:
;* 	see the file 'papco_variables' for a description
;*
;*      tmp={STEP_REGION,  $
;*           region:'', $
;*           description:''}
;*      step_regions=replicate(tmp, 20)
;*
;*      tmp={STEP_OBSERVER, $
;*           region:'', $
;*           appears_on_CD:1, $  ; this field: 1 -> appears on CD 1
;*                               ;             2 -> appears on CD 2
;*                               ;             3 -> appears on both CDs
;*           description:'', $
;*           id:'', $
;*           lat:0.0, $
;*           long:0.0}
;*      step_observers=replicate(tmp, 200)
;* OVERVIEW:
;*    This procedures are contained:
;* 
;*      PRO STEP_Read_Observers, REREAD=REREAD
;*    
;* MODIFICATION HISTORY:       
;*     august,september 1995, written by A.Keese 
;***************************************************************************
;***************************************************************************

;***************************************************************************
;* 
;* PROCEDURE:     
;*      PRO STEP_Read_Observers, REREAD=REREAD
;* 
;* DESCRIPTION:  
;*      read the file $papco_PATH/step_info.txt
;*      the descriptions of this file are read into the common-block
;*
;* INPUTS:       
;*      none
;* 
;* KEYWORDS:
;*      REREAD       read the file - even if it was already read
;*
;* CALLING SEQUENCE:
;*      STEP_Read_Observers, /REREAD
;*      STEP_Read_Observers
;*
;* MODIFICATION HISTORY:       
;*      september 1995, written by A.Keese 
;*
;***************************************************************************
PRO STEP_Read_Observers, REREAD=REREAD

  common step_info, step_regions, step_observers
  common get_error, get_err_no, get_err_msg

  get_err_no = 0 & get_err_msg = 'STEP INFO read succesfully'

  IF N_ELEMENTS(step_regions) GT 0 AND NOT KEYWORD_SET(REREAD) THEN RETURN

  ;open the file 'step.info' in step module directory
  ; check env. variable pointing to module directory
  if not (papco_check_data_env('PAPCO_MODULES', PATH=path)) then return  
  fln = path+'step_mag/plot_read/step.info'

  OPENR, f_Unit, fln, ERROR=err, /GET_LUN
  IF err NE 0 THEN BEGIN
    get_err_no = 2 &  get_err_msg = 'could not read step-info'
    message, get_err_msg, /cont
    return
  ENDIF

  sectionOfFile=0              ; this is incremented, when the first line of 
                                ; stars ('*****') is found.
                                ; if 0 : now reading regions-descriptions
                                ; if 1 : now reading observers-descriptions

   now_reading_CD=1             ; incremented, when the second line of stars
                                ; is found. It indicates,
                                ; observer-descriptions for which CD
                                ; are read.

   numberOfObservers=0
   numberOfRegions=0

   tmp={STEP_REGION,  $
        region:'', $
        description:''}
   step_regions=replicate(tmp, 20)
   tmp={STEP_OBSERVER, $
        region:'', $
        appears_on_CD:1, $      ; this field: 1 -> appears on CD 1
                                ;             2 -> appears on CD 2
                                ;             3 -> appears on both CDs
        description:'', $
        id:'', $
        lat:0.0, $
        long:0.0}
   step_observers=replicate(tmp, 200)


   WHILE NOT EOF(f_Unit) DO BEGIN
      dummy=''
      readf, f_unit, dummy
      line=strtrim(dummy, 2)
      IF strleft(line, 3) EQ '***' THEN BEGIN
        IF sectionOfFile EQ 1 THEN now_reading_CD=2
        sectionOfFile=1

      ENDIF ELSE BEGIN
         IF sectionOfFile EQ 0 THEN BEGIN ; read a region-description
            ind1=strpos(line, '=')
            region=strtrim(strleft(line, ind1), 2)
            desc=strtrim(strfrom(line, ind1+1), 2)

            step_regions(numberOfRegions).region=region
            step_regions(numberOfRegions).description=desc
            numberOfRegions=numberOfRegions+1
         ENDIF ELSE BEGIN       ; read an observer-description
            ind1=strpos(line, '   ') ; 1st part of line: region
            region=strtrim(strleft(line, ind1), 2)
            line=strtrim(strfrom(line, ind1+1), 2)

            ind1=strpos(line, '   ') ; 2nd part: observer-descriptio
            desc=strtrim(strleft(line, ind1), 2)
            line=strtrim(strfrom(line, ind1+1), 2)

            ind1=strpos(line, '   ') ; 3rd part : observer - id
            obs=strtrim(strleft(line, ind1), 2)
            line=strtrim(strfrom(line, ind1+1), 2)

            ind1=strpos(line, '   ') ; 4th part : latitude (float)
             lat=strtrim(strleft(line, ind1), 2)
                                     ; 5th part : longitude  
            lon=strtrim(strfrom(line, ind1+1), 2)

            step_observers(numberOfObservers).region=region
            step_observers(numberOfObservers).description=desc
            step_observers(numberOfObservers).id=obs
            step_observers(numberOfObservers).lat=FLOAT(lat)
            step_observers(numberOfObservers).long=FLOAT(lon)
            step_observers(numberOfObservers).appears_on_CD=now_reading_CD
            numberOfObservers=numberOfObservers+1
         ENDELSE
      ENDELSE 
   ENDWHILE ; end of EOF(f_Unit)
   
   step_regions=step_regions(0:numberOfRegions-1)
   step_observers=step_observers(0:numberOfObservers-1)


; now, duplicate ids in 'observers' are removed and the CD-fields of
; the duplicate ids are changed.
   obs_tmp=step_observers(sort(step_observers.id))
   n_unique=0                   ; number of unique observer-ids
   lastID=''                    
   both_cds=0                   ; set to one, if an observer appears on 
                                ; both CDs. 
   FOR i=0, numberOfObservers-1 DO BEGIN
      step_observers(n_unique)=obs_tmp(i)

      IF both_cds NE 0 THEN $
         step_observers(n_unique).appears_on_CD=3

      if obs_tmp(i).id EQ lastID THEN BEGIN ; as obs_tmp are the sorted
                                ; observers, duplicate observers
                                ; follow each other. In this case, the
                                ; n_unique-counter isn't increased,
                                ; and this will overwrite duplicates
         both_cds=1
      ENDIF ELSE BEGIN
        n_unique=n_unique+1      
        both_cds=0
      ENDELSE
      lastID=obs_tmp(i).id
   ENDFOR
   step_observers=step_observers(0:n_unique-1) ; erase the unused part
   CLOSE, f_Unit
   FREE_LUN, f_Unit
END
