;******************************************************************************
;* FILE: tdrs_panel.pro
;*
;* DESCRIPTION: 
;*    This is a demonstration on how to implement a papco-panel-editor.
;*    It makes use of all the features provided by $papco_XX/papco_panel.pro
;*    (a collection of standard papco widgets). It constructs an editor for a
;*    very simple papco-plot, that has all the default-functionality.
;*
;* COMMON-BLOCKS:
;*    tdrs_panel_data   contains the window-information
;*
;* STRUCTURES:
;*    This widget inherits a PlotInfo structure as an argument. This
;*    structure is then modified by the user according to his/her needs:
;*    Either by including standard PAPCO widgets from papco_panel.pro,
;*    which extend this structure, or by modifying its tags. The modified
;*    structure is then handed back to PAPCO.
;* 
;*    The PlotInfo structure is created by a call to papco_getPlotInfoStruct
;*    and contains a series of tags which are used to describe a given panel.
;*    These tags come in three categories:
;*       1. Tags used by PAPCO internally - don't touch!          
;*       2. Tags added by including standard PAPCO widgets. You can edit 
;*          or preset these tags if you wish.
;*       3. User Tags for control of user-specified features. There
;*          are enough user tags specified to handle most needs. 
;*    The main limitation of the User Tags is that you're stuck with
;*    what has been defined in PAPCO. If you REALLY need additional tags,
;*    this entails changing the PAPCO core, and should be done by
;*    whoever maintains the core. You can do this yourself too, by editing
;*    pro papco_getPlotInfoStruct contained in $papco_XX/papco/papco.pro
;*    but your changes will then not propagate to the next PAPCO version.
;*
;*    The PAPCO 09 set of user tags is:
;*
;*       typeVector:intarr(4), $ ; user keyword (old, for compatibility)
;*       swittch:lonarr(3),     $ ; user keyword (old, for compatibility)
;*       options:fltarr(20),   $ ; user keyword (float options)
;*       ioptions:lonarr(16),  $ ; user keyword (long options)
;*       soptions:strarr(16)   $ ; user keyword (string options)
;*
;*    NB! If you need more variables to commincate between your panel 
;*    editor and your plot routine then you can use your own common block
;*
;*    For a more complete description of the other PlotInfo tags and the
;*    tags used by PAPCO supplied widget refer to
;*    $papco_doc/papco_variables.text
;*
;* OVERVIEW:
;*    This function creates the editor-window:
;*       FUNCTION  tdrs_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;*    To test you might try this to get a stand-alone widget:
;*       a=papco_getPlotInfoStruct()   which creates the structure
;*                                     containing the plot information 
;*	 a=tdrs_panel(a)          which calls up the panel window.
;*    Interrogating a with help,a,/str shows the contents. 
;*
;* MODIFICATION HISTORY:       
;*     august 1995, written by A.Keese
;*     december 1995, modified by R.Friedel at Goddard
;*     april 1996, modified by R.Friedel
;*     march 1997, modified by R.Friedel
;*     August 1998, modified by R.Friedel
;******************************************************************************

;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  tdrs_panel, aPlotInfo, GROUP=group, ACTION=ACTION
;* 
;* DESCRIPTION:  
;*	Open an editor widget for a panel, using the methods defined in the 
;*	file $papco_XX/papco_panel.pro .
;*	When the user presses the 'Done'- or the 'Cancel'-Button, the values 
;*	changed by the event-methods of papco_panel.pro are returned.
;*	The returned value is a structure containing those values and a flag,
;*	'canceled', which the caller may read out to check, whether the 
;*	window was closed using the 'Cancel' or the 'Done'-Button.
;*
;* INPUTS:       
;*      aPlotInfo       a structure of the papco_plotinfo-type
;*                      this structure is edited in the panel editor window.
;* KEYWORDS:
;*      GROUP           a WIDGET_BASE
;*	                when specified, this is used as GROUP_LEADER
;*      ACTION          a string
;*	                when specified, the 'Done'-Button is labeled with it
;*		        if not specified, the 'Done'-Button is labeled 'Done'
;*
;* OUTPUT:	 
;*      a default-structure of type papco_panel_editor_return. Though this
;*      type is nowhere defined, it is a virtually existing data-type. 
;*      All the functions in $papco_XX/papco.pro that edit plots or add new
;*      ones, expect the opened editor-windows to return a structure containing
;*      at least some fields. Typically this structure contains fields
;*      used by the papco_panel widgets, and some fields "inherited"
;*      from aplotinfo but modified by the user.
;*
;* CALLING SEQUENCE:
;*      newPlotData=papco_PANEL_TEMPLATE(aplotInfo, ACTION=, GROUP=)
;******************************************************************************
FUNCTION  tdrs_panel, aPlotInfo, GROUP=group, ACTION=ACTION

  COMMON tdrs_panel_data, widgetData

;--- if GROUP is not specified, the group-leader will be the desktop
  IF NOT KEYWORD_SET(GROUP) THEN GROUP=0
;--- set the label for the action-button
  IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

;--- the routines used by papco_panel.pro define tags for a structure. Those
;    tags will be stored in the structure 'additionalWData'. When you realize 
;    the window, you have to add your own window-specific data to that  
;    structure and afterwards assign it to widgetData.
  additionalWData={canceled:1}

;--- create the base-window ---------------------------------------------------
;    since this can be a large widget, make a left and right side.
  panel_base=papco_WIDGET_BASE(title=aPlotInfo.panelKind, row=1, $
                               GROUP_LEADER=group, /MODAL)
  panel_left=widget_base(panel_base,column=1)
  panel_right=widget_base(panel_base,column=1)

;--- All papco_panel_add routines are papco panel widgets that can be included
;    in the panel editor or not. They provide functionality that papco 
;    supports. They dynamically extend the panel editors widget structure, and
;    all the events they generate are supported by papco_panel_event, 
;    transparent to the user. 

;--- the first part of the window will consist of the widget that -------------
;    controls the height of that panel. This is needed for all panel editors.
  widgets=papco_panel_AddHeightWidgets(panel_left, aPlotInfo)
  additionalWData=create_struct(widgets, additionalWData)
   
;
;---  <<<<<<<<<<  insert your widgets below here!  >>>>>>>>>
;
  type_base=WIDGET_LABEL(panel_left, $
  VALUE='***                                               ***')
  type_base=WIDGET_LABEL(panel_left, $
  VALUE='***           PLACE OWN WIDGETS HERE              ***')

;--- this example uses a simple slider to set channel number. The channel 
;    number is passed out in the  typevector(0) tag of structure aplotInfo.
;    channel numbers go from 1 to 10, if 0 is chosen the data is plotted in the
;    form of a color spectrogram.

  channel_base=WIDGET_BASE(panel_left, /FRAME, ROW=1)
  sl_channel=WIDGET_SLIDER(channel_base, XSIZE=450, /DRAG, $
                           TITLE=strcenter('channel number',75), $
                           VALUE=aPlotInfo.typevector(0) > 0 < 10, $
                           UVALUE='sl_channel',$
                           MINIMUM=0, MAXIMUM=10)

  type_base=WIDGET_LABEL(panel_left, $
  VALUE='*** 1 - 10 chooses Channel, O a color spectrogram ***')
  type_base=WIDGET_LABEL(panel_left, $
  VALUE='***              END OWN WIDGETS                  ***')
  type_base=WIDGET_LABEL(panel_left, $
  VALUE='***                                               ***')
;
;---  <<<<<<<<<<  insert your widgets above here!  >>>>>>>>>
;   
   
;--- for examples of standard buttons or sliders look at other panel editors.
;    some standard buttons which have been recycled before can be found in
;    papco_XX/papco_lib/more_panel_widgets.

;--- As the same panel is called when you edit a plot, interrogate the tags in 
;    the plotInfoStruct (aplotInfo) passed in and set your widgets accordingly.
;    In this way the widget is always reinitialized to its last state.
   
;--- The next five papco panel widgets will be added only if you want to
;    support that functionality, and if they make sense for your
;    tdrs. If you do not want to use them, just comment them out.

;--- some panels might need to read/input to text files (event, text ----------
;    module). So a generic papco widget is supplied for this. 
  widgets=papco_panel_AddFile_Widgets(panel_left, aPlotInfo)
  additionalWData=create_struct(widgets, additionalWData)   
;--- add the papco panel widget for choosing the color bar for color plots ----
  widgets=papco_panel_Addcolorbar_Widgets(panel_left, aPlotInfo)
  additionalWData=create_struct(widgets, additionalWData)
;--- add the  papco panel widget for choosing the plot-style for line plots ---
  widgets=papco_panel_add_PlotStyleButtons(panel_left, aPlotInfo, 0)
  additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widget for choosing the y-axis-ranges ----------------
  widgets=papco_panel_AddYScale_Widgets(panel_right, aPlotInfo,/VERTICAL)
  additionalWData=create_struct(widgets, additionalWData)
;--- add the papco panel widgets for choosing the z-axis-ranges ---------------
  widgets=papco_panel_AddZScale_Widgets(panel_right, aPlotInfo,/VERTICAL)
  additionalWData=create_struct(widgets, additionalWData)
   
;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window. This is needed for all panel editors. If you want to 
;    make use of the configure feature, then add the keyword CONFIG
  widgets=papco_panel_AddActionButtons(panel_right, Action, aPlotInfo, $
                                       /CONFIG, /VERTICAL)
  additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You always
;    need to specify the help-file, and your structure must contain the
;    'additionalWData'. You should create the helpfile in the papco_interface
;    directory of your module (a template is in the tdrs example module)
;    Also add the ID's of your widgets so that you can perform widget_control
;    actions in the events routine or in a refresh routine if needed.
  thisWidgetsData={helpfile:'tdrs_panel.help', $
                   typevector:aPlotInfo.typevector}
  widgetData=papco_panel_CreateWidgetData(thisWidgetsData, additionalWData)

;--- realize the window -------------------------------------------------------
  papco_Panel_Realize, panel_base, widgetData
  papco_xmanager, 'tdrs_panel', panel_base

;--- the returned data consists of the standard-data inherited from
;    papco_panel.pro and your new data. Stick to the tags provided in
;    aPlotInfo, there are ample variables for the user
  returnData={typevector:widgetData.typevector}
  additionalRData=papco_panel_ReturnData(widgetData)

  return, create_struct(ReturnData, additionalRData)
   
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO tdrs_panel_event, event
;* 
;* DESCRIPTION:  
;*      This processes events from the XMANAGER.
;*      As this example does not extend the basic functionality of plot-
;*      editors, it directly calls the default-handler, papco_PANEL_Event
;*
;* INPUTS:       
;*      event            an XManager event
;*                  
;* KEYWORDS:
;*      none
;* 
;* CALLING SEQUENCE:
;*      called by XMANAGER
;******************************************************************************
PRO tdrs_panel_event, event
  
  COMMON tdrs_panel_data, widgetData

  WIDGET_CONTROL,GET_UVALUE=uval, event.id

; in this editor, we only need to process the channel slider events. events
; other than those from your own widgets need to be passed on to 
; 'papco_panel_Event', which handles the events generated
; by the papco-provided panel widgets. Your own added panel widgets events
; need to be processed here:  you should check 'event' for your events and call
; 'papco_panel_Event', if event was not generated by your own panel widgets.
; The best way to learn is to look at some of the other module's panel editors.

  if uval eq 'sl_channel' then begin
    widgetData.typevector(0)=event.value
  endif else begin
    widgetData=papco_panel_Event(event, widgetData)
  endelse
    
END
