;******************************************************************************
;*  NAME:        r_tdrs
;*
;*  DESCRIPTION: Routine to read tdrs files. Adheres to
;*               papco conventions: Error common block
;*                                  Uses MJDT time
;*                                  papco read philosophy
;*
;*  INPUT:       none (uses common block time from papco)
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*
;*  DATE:        June 1996
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES:     February 1997. Used MJDT time now, and is a fully
;*               running example module on it's own
;*               October 1997. Rewritten to be more general and fully
;*               adhering to papco philosophy. Added remore Get feature, and
;*               the example data file is no longer part of the 
;*               module - it needs to be gotten remotely first!
;******************************************************************************
pro r_tdrs, VERBOSE=VERBOSE

COMMON tdrs_data, input_header, input_data
COMMON mjdt, mjdt_start, mjdt_end
COMMON get_error, get_err_no, get_err_msg
   
; common get_error is used by papco to check if the data read was successful.
; it is up to the user to set this correctly and to make sure that the read- 
; routine does not crash on an I/O error - otherwise papco will crash at that
; point, too. papco draws an empty panel when get_err_no in not equal to zero
; and prints get_err_msg in that panel for user information. papco will try
; to invoce the plot routine for this data if get_err_no = 0 , otherwise an
; emplty panel with the error message get_err_msg is plotted.

get_err_no=0  &  get_err_msg='tdrs read succesfully'
   
; check env. variable pointing to data directory
if not (papco_check_data_env('TDRS_EPH_DATA', PATH=path)) then return
   
; check VERBOSE keyword and set
if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
   
; clear data arrays for reuse
input_header=0  &  input_data=0   
   
;for TDRS read ALL data into one file!
r_tdrs_onefile, path, VERBOSE=VERBOSE

end

;******************************************************************************
;*  NAME:        r_tdrs_onefile,fln
;*
;*  DESCRIPTION: reads one file of type plot_file. Assumes data has been
;*               stored in IDL saveset using structures input_header and
;*               input_data
;*
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_tdrs_onefile, path, VERBOSE=VERBOSE
  
common tdrs_data, input_header, input_data
common get_error, get_err_no, get_err_msg
   
get_err_no=0  &  get_err_msg='tdrs onefile read succesfully'
   
; clear data arrays for reuse
input_header=0  &  input_data=0   
   
on_ioerror, read_original
fln = path+'tdrs_orbit.idl'
restore,fln, VERBOSE=VERBOSE
dealloc_lun,/quiet   ;frees logical units used by save/restore
message,'read '+fln, /cont
return

stop

read_original:       ;if the .idl file is not found, try to read the original
r_tdrs_onefile_original, path, VERBOSE=VERBOSE
      
end   
  
;******************************************************************************
;*  NAME:        r_tdrs_onefile_original,fln
;*
;*  DESCRIPTION: reads one file of type plot_file_original. Under the
;*               papco read philosophy, files should be read the first
;*               time in whatever format they might have - ascii,
;*               fortran binary, pascal binary, etc. The files might also
;*               be in some compressed format, so they need to be uncompressed
;*               first. 
;*               
;*               You might also have to change time formats. 
;*               A lot of files are stored per day, and have seconds of
;*               that day as the time. This is what is assumed here and
;*               is papco standard. You need to convert the time
;*               to papco standard if you intend to use the plotroutines 
;*               supplied with papco (use the routines in
;*               $papco_BASE/papco_Lib/time_convert.pro). 
;*               If you use your own plot routines this step is not needed.
;*               
;*               The first read might be slow, esp if the format is ascii.
;*               Use the IDL "save" command to make a binary copy of the
;*               file which can then be read quickly using "restore". The
;*               next time only the IDL saveset is read, which is much
;*               faster.
;*               
;*               In order to be able to concatonate the arrays easily,
;*               it makes sense to have the data in an array of time-ordered
;*               structures, which makes adding together days of data
;*               trivial using the IDL concatonation operator "[]".
;*               
;*  INPUT:       fln  - string of filename
;*
;*  OUTPUT:      none (places data into common blocks)
;*
;*  DATE:        October 1997
;*
;*  AUTOR:       R. Friedel
;******************************************************************************
pro r_tdrs_onefile_original, path, VERBOSE=VERBOSE
  
common tdrs_data, input_header, input_data
common get_error, get_err_no, get_err_msg  
   
get_err_no=0  &  get_err_msg='tdrs_onefile_orignal read succesfully'
   
; clear data arrays for reuse
input_header=0  &  input_data=0   

;------------------------------------------------------------------------------
; read the original data files - THIS NEEDS TO BE REWRITTEN BY USER!!!!
  
; define the header and data structures. Edit to suit your data!
max_no_rec=5000              ;the maximum number time records in your data
ext='.txt'                   ;file extention of your dat files
input_header={tdrs_header,tag1:0,tag2:0}
dummy={tdrs_data, tai:0.0d, lat:0.0, long:0.0, rad:0.0}
data = fltarr(6)
input_data=replicate(dummy, max_no_rec)
c1 = 0l & c2 = 0l & in_str = ''

files = findfile(path+'*.txt', count = c)

FOR i = 0, c-1 DO BEGIN
    ;on_ioerror, out  
    openr,u,files(i),/get_lun
    message, files(i), /cont
    FOR j = 0, 1 DO readf, u, in_str
    while not eof(u) do begin   ;read in data and store in input_data
        readf, u, in_str

        t_str = strupcase(strmid(in_str, 0, 22))
        r = strsplit(t_str, ' ', /extract)
        t_str =  r(0)+'-'+r(1)+'-'+r(2)+'T'+r(3)
        utc = STR2UTC(t_str) 
        tai = UTC2TAI(utc)

        data_str = strmid(in_str, 24,  strlen(in_str)-24)
        reads, data_str, data

        input_data(c1).tai = tai
        input_data(c1).lat = data(0)
        input_data(c1).long = data(1)
        input_data(c1).rad = data(2)+6371.0 ;altitude + Earth radius
   
        print,  input_data(c1)

        c1 = c1+1 & c2 = c2+1  
        IF c2 EQ max_no_rec THEN BEGIN
            input_data=[input_data, replicate(dummy, max_no_rec)]
            c2 = 0l
        ENDIF     

    ENDWHILE
    close, u & free_lun, u

ENDFOR
input_data = input_data(0:c1-1)

; end of user written section
;------------------------------------------------------------------------------

fln = path+'tdrs_orbit.idl'
message, fln, /cont

; save read data with correct time. Rename data to *.idl
save,filename=fln, input_header,input_data, /compress, VERBOSE=VERBOSE
dealloc_lun,/quiet              ;frees logical units used by save/restore
return
  
out:
; if there was an error, report it. 
get_err_no=!ERROR & get_err_msg='!C '+!ERR_STRING
   
END

;******************************************************************************
;make tdrs ephemeris files, calculate magnetic coordinates
PRO m_tdrs_eph, FORCE=FORCE, INTMOD=INTMOD, EXTMOD=EXTMOD, PITCH=PITCH, $
                VERBOSE=VERBOSE, NOREAD = NOREAD,  SAVE = SAVE

COMMON tdrs_data, input_header, input_data
common mjdt, mjdt_start, mjdt_end
common get_error, get_err_no, get_err_msg

; check keywords and set
if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0     
if keyword_set(INTMOD) then kint=long(INTMOD) else kint=0l
if keyword_set(EXTMOD) then kext=long(EXTMOD) else kext=2l
if keyword_set(PITCH) then alpha=long(PITCH) else alpha=90.0d

;make new mag field coord structure
new_struct_name='MODEL_'+varprt(kint)+'_'+varprt(kext)
new_struct=create_struct(name='UNILB', 'L',0.0,'MLT',0.0,'MLAT',0.0)  
  
;see if ephemeris file exists. if not, make it.
IF NOT keyword_set(NOREAD) THEN r_tdrs, VERBOSE=VERBOSE

;check for existing tags, work out what needs to be done    
tags=tag_names(input_data(0))
; check if required mag coords exist in file.
index=where(tags eq new_struct_name,c)
if c eq 0 then BEGIN
    need_to_do=1
    dummy2=create_struct(input_data(0),new_struct_name,new_struct)
    dummy3=replicate(dummy2,n_elements(input_data)) 
    ;fill new array with copies of existing array
    for j=0l, n_elements(input_data)-1 do $ 
        for m=0,n_elements(tags)-1 do dummy3(j).(m)=input_data(j).(m)
    input_data=temporary(dummy3)
endif else need_to_do=0

;get the index of the new mag coords tag
tags=tag_names(input_data(0))
idx=where(tags eq new_struct_name) & new_idx=idx(0)    

;calculate new magnetic coords from UNILB library
IF need_to_DO OR keyword_set(FORCE) THEN BEGIN
    message,' calculating L, MLT, MLAT for '+new_struct_name, /cont

    ;setup unilib shareab;e object location
    lib_dir = '/n/toaster/u/friedel/idl/fortran/UNILIB/ver_2.06/source'
    lib_name = lib_dir+'/unilib.so'
    entry_name = 'uxidl_'

    ;initialize UNILIB library
    ;MOD by RF to UT990: set kinit to 533 to disable magnetopause checking
    kunit = 533l & kinit = 1l & iver = 0l
    result = call_external(lib_name, entry_name,'UT990', $
                           kunit, kinit, iver)

    ;select a internal geomagnetic field model
    t = TAI2UTC(input_data(0).TAI, /EXTERNAL )
    year = double(t.year) & lbint = string('',format="(a32)") & ifail = 0l
    result = call_external(lib_name, entry_name,'UM510', $
                           kint, year, lbint, kunit, ifail)

    FOR i = 0l, n_elements(input_data)-1 DO BEGIN

        ;compute modified Julian date as required - idl roputines.
        ;NOTE: tmjd is modified Julian date: JD - 240000.5
        ;amjd is modified Julian Day based on the 1st of January 1950
        ;tmjd = TAI2UTC(input_data(i).TAI)
        ;tmjd2amjd = doy2utc(1,1950)
        ;amjd = double(tmjd.mjd-tmjd2amjd.mjd)+double(tmjd.time)/86400000.0d

        ;compute modified Julian date as required
        t    = TAI2UTC(input_data(i).TAI, /EXTERNAL )

        mdate = {iyear:long(t.year), imonth:long(t.month), iday:long(t.day), $
                 ihour:long(t.hour), imin:long(t.minute), $
                 secs:double(t.second), amjd:0.0d}
        result = call_external(lib_name, entry_name,'UT540', mdate)

        ;select an external magnetic field model
        param = dblarr(10) & lbext = string('',format="(a32)")

        result = call_external(lib_name, entry_name,'UM520', $
                               kext, mdate.amjd, param, lbext, kunit, ifail)
         
        ;convert the location from geodetic to geocentric coordinates
        mgde = {radius:double(input_data(i).RAD), $
                colat:90.0d - double(input_data(i).LAT), $
                elong:double(input_data(i).LONG)}
        mgeo = {radius:0.0d, colat:0.0d, elong:0.0d}
        result = call_external(lib_name, entry_name,'UM536', $
                               mgde, mgeo)

        ;evaluate the magnetic field. This also updates 
        ;the Sun position and the SM coordinates (done by UM522 and UM524)
        mb = {dnrm:0.0d, rho:0.0d, theta:0.0d, phi:0.0d}
        result = call_external(lib_name, entry_name,'UM530', $
                               mgeo, mb, ifail)   

        ;evaluate the (B, L) coordinates
        nfbm = 1l
        fbm = 0.0d & flm = 0.0d & fkm = 0.0d & fsm = 0.0
        fbeq = 0.0d & fs = 0.0d
        result = call_external(lib_name, entry_name,'UL220', $ 
                               mgeo,alpha,nfbm,fbm,flm,fkm,fsm,fbeq,fs,ifail)

        ;evaluate the magnetic local time and latitude
        xmlt = 0.0d &  xlat = 0.0d
        result = call_external(lib_name, entry_name,'UM538', $
                               mgeo, mdate.amjd, xmlt, xlat, ifail)
        
        input_data(i).(new_idx).L=flm  
        input_data(i).(new_idx).mlt=xmlt 
        input_data(i).(new_idx).mlat=xlat

        IF verbose EQ 2 THEN begin
            print, TAI2UTC(input_data(i).TAI,  /ECS), i
            print, 'L value, MLT, MLAT: ', input_data(i).(new_idx).L, $
              input_data(i).(new_idx).mlt, input_data(i).(new_idx).mlat
            print, ''
        ENDIF 

    ENDFOR

    IF keyword_set(SAVE) THEN BEGIN
        ; check env. variable pointing to data directory
        if not (papco_check_data_env('TDRS_EPH_DATA', PATH=path)) then return
        fln = path+'tdrs_orbit.idl'
        save, filename = fln, input_header, input_data, $
          /compress, verbose = verbose
    ENDIF 

stop

ENDIF ELSE BEGIN
    message,  'Mag coords exist for: '+new_struct_name, /cont
ENDELSE 

close, /all

END
