PRO gen_bfield, CDF, cdfcal=cdfcal, rec_start, rec_count_in, $
                start_time=timein, end_time=timeout, $
                hydtime=hydtim, mfetime=mfetim, Hz50=Hz50, Hz54=Hz54, $       
                fixed_payload=fixed_payload, $
                nocorrect=nocorrect, $  
                nobmag=nobmag, $
                nofilter=nofilter, $
                noclampbmag=noclampbmag, $
                nogains= nogains, $
                nomatrix= nomatrix, $
                xyz=xyz, thetaphi=thetaphi, $
                btheta=btheta, bphi=bphi, bmag=bmag, $
                bx=bx, by=by, bz=bz, $
                time=t, bvalid=bvalid, $
                bflags=bflags, $
                verbose=verbose, $
                lag_corrected=lag_corrected, $
                raw_angle_error=raw_angle_error, $
                berrx=bxerr, berry=byerr, berrz=bzerr, $
                dbxs=dbxs, dbys=dbys, dbzs=dbzs, $
                offset_raw_in= offset_raw_in, $
                offset_in= offset_in, $
                matrix_in= matrix_in, $
                offset_phi_in= offset_phi_in, $
                gains_in= gains_in, $                
                version=version, $
                phase_out=phase_out, $
                matrix_out=matrix_out, $
                gains_out= gains_out, $
                fill_in= fill_in, $
                old_spin_rate=old_spin_rate, $
                science_mode=science_mode
                             
;+
;
;PRO gen_bfield, CDF, cdfcal=cdfcal, rec_start, rec_count_in, $
;                start_time=timein, end_time=timeout, $
;                hydtim=hydtim, mfetim=mfetim, Hz50=Hz50, Hz54=Hz54, $
;                fixed_payload=fixed_payload, $
;                nocorrect=nocorrect, $  
;                nofilter=nofilter, $
;                noclampbmag=noclampbmag, $
;                nogains= nogains, $
;                nomatrix= nomatrix, $
;                xyz=xyz, thetaphi=thetaphi, $
;                btheta=btheta, bphi=bphi, bmag=bmag, $
;                bx=bx, by=by, bz=bz, $
;                time=t, bvalid=bvalid, $
;                bflags=bflags, $
;                verbose=verbose, $
;                lag_corrected=lag_corrected, $
;                raw_angle_error=raw_angle_error, $
;                berrx=bxerr, berry=byerr, berrz=bzerr, $
;                old_spin_rate=old_spin_rate
;
; Input
;   rec_start, rec_count  fix    define the hydra level1 cdf records for 
;                                which corresponding bfield data is to 
;                                be returned.
;
;   CDF                   long   CDF handle that is input data source
;                         string  open cdf file for this day.
;   cdfcal                long   CDF handle for the calibration file
;
;   start_time=st         float  seconds.  Time is translated to
;                                preceeding record number=>rec_start
;   end_time=et           float  seconds.  Time is translated to last
;                                inclusive block.
;
;   /hydtim                      Return data interpolated to hydra
;                                time tags, that is 27Hz.  Direction
;                                information comes from HYDRA
;                                telemetry at 27Hz.  Bmag is
;                                interpolated from 8Hz data.
;
;   /mfetim (default)            Data is returned at MFE resolution, 8Hz.
;
;   /Hz54                        data is returned at 54Hz.
;
;   /nocorrect                   No corrections are performed, and raw
;                                data is returned in rotating payload coord.
;
;   /lag_corrected               27/54 Hz data magnitude (from 8 Hz) is 
;                                corrected for on-board MFE time-filter.
;
;   version=v                    returns the snapshot version of gen_bfield.
;
;   /old_spin_rate               Use the "old" pre - 3/17/02 spin_rate
;                                calculation
;
;   /fill_in                     fill in gaps in the data.  (fill_in=1
;                                by default, so set fill_in=0 to turn off)
;
;   /science_mode                keep a watchful eye on the
;                                sensibility of data for scientific
;                                analysis.  For instance, put up a
;                                warning if offsets are equal to zero.
;
; Output
;   /xyz                 calculate bx, by, and bz
;   /thetaphi            calculate thetab, phib, bmag
;   btheta, bphi, bmag   float in nT
;   bx, by, bz           float in nT
;   t                    double, seconds since midnight
;   bvalid               1=valid, 0=invalid
;   bflags               and 1 = 1  Hydra filter
;                        and 2 = 2  Negative descriminant in ClampBmag
;                        and 4 = 4  All components interpolated from 8Hz
;                        and 8 = 8  data dropout has been interpolated across
;   raw_angle_error      for 27/54 Hz - difference in degrees between 
;                        calc-via no_lag angles and hydra angles.
;   berrx/y/z/           Error in component based on high-freq (> 4Hz) content
;-

; defines
   i4fill = long(-2147483648)
   i2fill = -32768
   rfill = -1.0e-31
   rfill_new = -1.0e31


; BEGIN CODE

   version_str= '$Name: v1_43 $'
   version= version_str


   if getenv('verbose') eq '1' then verbose=1

; open cdf files if necessary.
   if (size( cdf ))(1) eq 7 then begin
       init_level1, cdf, bm, mode_info
       cdf= mode_info.l1stat.rawid
       cdfcal= mode_info.l1stat.calid
   endif

; process keywords and parameters

   hydtim= keyword_set( hydtim )
   Hz50= keyword_set( Hz50 )
   if keyword_set( hz54 ) then hz50=1

   mfetim= keyword_set( mfetim ) or ( hydtim eq 0 and Hz50 eq 0 )

   xyz= keyword_set( xyz )
   thetaphi= keyword_set( thetaphi )
   if n_elements( cdfcal ) eq 0 then cdfcal=cdf

   noclampbmag= keyword_set( noclampbmag )
   nofilter= keyword_set( nofilter )
   nocorrect= keyword_set( nocorrect )
   if nocorrect then begin
       message, 'not correcting -> no clamp bmag is on', /cont
       noclampbmag=1
   endif
   nobmag= keyword_set(nobmag)
   if nobmag and ( nocorrect eq 0 ) then begin
       message, 'keyword nobmag only works with nocorrect.', /cont
       stop, ''
   endif

   if n_elements(fill_in) eq 0 then begin
       fill_in=1
   endif

;
;  Correct for lag by default if environment variable set
;
   if (getenv('LAG_CORRECT_BFIELD') eq '1') then lag_corrected=1

   if n_elements( lag_corrected ) eq 0 then begin
       message, 'lag_correct not specified, performing lag_correction '+$
         'by default', /cont       
       lag_corrected=1
   endif else begin
       noclampbmag=1
   endelse

   if lag_corrected eq 0 then begin
       message, 'lag_correct keyword is not set in gen_bfield call.',/cont
       message, 'this needs to be corrected.', /cont
       wait, 1
   endif   

   hyd_cdf_varget, CDF, 'MAX_BLOCK_NUMBER', maxrec
   
   if n_params() eq 1 then begin ; rec_start, rec_count come from start_time
       hyd_cdf_varget, CDF, 'BLOCK_TIME', dumtime, rec_start=0, rec_count=maxrec
       dumtime = dumtime MOD 86400000. / 1000.
       
       istart = where(dumtime GE timein)
       rec_start = ( istart(0)-1)>0
       istop = where(dumtime LE timeout)
       if istop(0) eq -1 then istop=32765
       rec_stop= (istop(n_elements(istop)-1)+1) < (maxrec-1)
       rec_count = ( rec_stop-rec_start+1 ) > 1
   endif else begin
       rec_count = rec_count_in
   endelse

   rec_count= rec_count>2

   hyd_cdf_varget, CDF, 'BLOCK_MODE', block_mode, $
     rec_start=rec_start, rec_count=rec_count

; read in input data based on output resolution   
   read_raw_mfe, CDF, rec_start, rec_count, bx0, by0, bz0, t, $
     hydtim=hydtim, Hz50=Hz50, bvalid=bvalid, nofilter=keyword_set(nofilter), $
     cdfcal=cdfcal, bflags=bflags, lag_corrected=lag_corrected, $
     angle_error=raw_angle_error, bxerr=bxerr, byerr=byerr, bzerr=bzerr, $
     dbxs=dbxs, dbys=dbys, dbzs=dbzs, nobmag=nobmag

   if Hz50 then begin
       
;      hrec says for any measurement, which hydra block it came from.
       hrec= make_array( 768, value=1, /int ) # indgen( rec_count )
       hrec= hrec(*)

   endif else if hydtim then begin
       
;      hrec says for any measurement, which hydra block it came from.
       hrec= make_array( 384, value=1, /int ) # indgen( rec_count )
       hrec= hrec(*)

   endif else begin
       
;      get hydra time tags and create map to hydra records
       hyd_cdf_varget, CDFcal, 'BLOCK_TIME', block_time, $
         rec_start=rec_start, rec_count=(rec_count+1)<maxrec
       block_time1= block_time mod 86400000 / 1000.

;      trim to corresponding hydra time range.
       rin= where( t ge block_time1(0) and t lt $
                   block_time1(n_elements(block_time1)-1) )
       if rin(0) ne -1 then begin
           t= t(rin)
           bvalid= bvalid(rin)
           bflags= bflags(rin)
           bx0=bx0(rin)
           by0=by0(rin)
           bz0=bz0(rin)
       endif else begin
           message, 'No MFE data corresponds to specified Hydra times.' + $
             '  Passing all MFE data.', /cont
       endelse

       hrec= long( interpol( lindgen(n_elements(block_time1)), $
                             block_time1, $
                             t ) )
       hrec= hrec > 0 < (n_elements(block_time1)-2) ;just in case
       
   endelse

   if not hydtim then begin
       hyd_cdf_varget, CDFcal, 'BLOCK_TIME', block_time, $
         rec_start=rec_start, rec_count=rec_count
       block_time1= block_time mod 86400000 / 1000.
       htim= reform( rebin( block_time(*), 384L*rec_count, /sample ), $
                     384, rec_count )       
       hyd_cdf_varget, CDF, 'TIME_OFFSET', time_offset, $
         rec_start=block_mode(0)-1
       time_offset=time_offset*1000 ; sec -> millisec
       htim= htim + time_offset # make_array( rec_count, value=1 )

   endif

   hyd_cdf_varget, cdfcal, 'BLOCK_TIME', midn, $
     rec_start=rec_start, rec_count=1

   midn= midn - ( midn mod 86400000 )
   rectimes= midn + (t*1000)
   
   cdf_epoch, mfefixx, 1996, 4, 2, 17, 30, /compute 
   
   rflipx= where( rectimes lt mfefixx )
   if rflipx(0) ne -1 then begin
       bx0( rflipx )= -bx0( rflipx )
       message, 'phib: x-axis value flipped.', /cont
   ENDIF 


;  Okay, that's the input.  Now, perform the correction.

; ------------------------------------------------
; CORRECTIONS: offsets, gains, fixed_payload, etc.
; ------------------------------------------------

; get offset-corrected mfe data and interpolate to hydra times for clampbmag
   if ( hydtim or hz50 ) and not noclampbmag then begin
       gen_bfield, cdf, cdfcal=cdfcal, rec_start, rec_count, $
         /mfetim, /nogains, /nomatrix, nofilter=keyword_set(nofilter), $
         /xyz, bx=mfebx, by=mfeby, bz=mfebz, time=mfetime, bvalid=bvalidmfe,$
         lag_correct=0
       r= check_math( 0,1 )     ; inhibit math messages
       mfebmag= sqrt( mfebx^2 + mfeby^2 + mfebz^2 )           
       r= check_math( 0,0 )
       r= where( bvalidmfe eq 0 )
       if r(0) ne -1 then mfebmag(r)=1e31
       bmag_interp1= interpol( mfebmag, mfetime, t )
       rv= where( bvalidmfe ne 0 )
       if rv(0) ne -1 then begin
           max= max( mfebmag( rv ) )
           rnv= where( bmag_interp1 gt max or bmag_interp1 lt 0.) 
           if rnv(0) ne -1 then bvalid(rnv)=0
       endif else begin
           bvalid(*)=0
       endelse
       r= check_math( 0,1 )     ; inhibit err messages from squaring fill.
       bmag_interp1_sq= temporary( bmag_interp1 ) ^ 2
       r= check_math( 0,0 )     ; reset message printing.
   endif
   
; calculate sc_phase(time) for the payload -> fixed payload xformation.
   hyd_cdf_varget, CDF, 'PHASE_START', phase_start, $
     rec_start=rec_start, rec_count=rec_count
   phase_start= reform( rebin( phase_start(*), 384L*rec_count, /sample ), $
                        384, rec_count )

   if block_mode(0) ne block_mode(n_elements(block_mode)-1) then $
     message, 'Warning: Block modes change, but phi_offset assumed constant.',$
     /cont
   
   block_mode=block_mode(0)
   
   hyd_cdf_varget, CDF, 'PHI_OFFSET', phi_offset, $
     rec_start=block_mode-1
   phi_offset= phi_offset # make_array( rec_count, value=1 ) 

   sc_phase= phase_start(*) + phi_offset(*) 

   spin_period=hd_spin_period(CDF=CDF, duntim=sun_time)

   if hydtim eq 0 then begin    ;xform sc_phase to mfe or "ppa" timetags

       sun_index=dindgen(n_elements(sun_time))

       spins=interpol(sun_index, sun_time, t)

       nspins=long(spins)
       dspins=(sun_time[nspins+1]-sun_time[nspins])/spin_period[nspins]
       extra_spins= (spins-nspins)*dspins+1.d0/4096.d0

       sc_phase=(extra_spins mod 1) *2*!dpi

   endif else begin

       spin_rate= ( sc_phase(1:100)-sc_phase(0:99) ) / $
         ( ( t(1:100) - t(0:99) ) * 1000 )
       s= sort( spin_rate )
       spin_rate= spin_rate(s(50))

   endelse

   if nocorrect then begin
       mfe_phase_offset=0.0
       in=0
       out= n_elements( bvalid ) -1
       if keyword_set( fixed_payload ) then begin
           zeta= sc_phase(in:out) - !dtor*mfe_phase_offset
           rotate=1
       endif else begin
           zeta= - !dtor*mfe_phase_offset
           if zeta ne 0.0 then rotate=1 else rotate=0
       endelse
       if rotate then begin
           bx3= bx0*cos(zeta)-by0*sin(zeta)
           by3= bx0*sin(zeta)+by0*cos(zeta)
           bz3= bz0 
       endif else begin
           bx3= bx0 
           by3= by0 
           bz3= bz0 
       endelse
       if xyz then begin
           bx= bx3
           by= by3
           bz= bz3
       endif
       
       if thetaphi then begin
           bmag1= fltarr( out-in+1 )

           r= where( bvalid(in:out) eq 0 )
           if r(0) ne -1 then bmag1(r) = 1e31                      

           rv= where( bvalid(in:out) eq 1 )
           btheta= fltarr( n_elements( bvalid ))
           bphi= fltarr( n_elements( bvalid ))
           if rv(0) ne -1 then begin
               bmag1(rv) = sqrt(bx3(rv)^2 +by3(rv)^2+bz3(rv)^2)

               bhatx = bx3(rv)/bmag1(rv)
               bhaty = by3(rv)/bmag1(rv)
               bhatz = bz3(rv)/bmag1(rv)

               btheta(in+rv)= acos(bhatz)
               phib1= acos( ( bhatx/sin(btheta(in+rv))) < 1.0 > (-1.0) )
               r = where(bhaty LT 0.)
               if r(0) ne -1 then phib1(r)= 2*!pi - phib1(r)
               bphi(in+rv)= phib1
           endif
           bmag= bmag1
           if nobmag then bmag(*)=1e31
       endif
       return
   endif ; no correct

; fill bvalid array.
;    B_AVG_RAW is three raw 16-bit components of B from SCR
   hyd_cdf_varget, CDF, 'B_AVG_RAW', b_avg_raw, $
     rec_start=rec_start, rec_count= rec_count
   
   b_avg_invalid= b_avg_raw(0,*) eq i2fill

   if n_elements( block_time ) gt 0 then begin
       dt= block_time(1:*)-block_time(0:*)
       r= where( abs( dt-13800 ) gt 3 ) ; msec
       if r(0) ne -1 then begin
           b_avg_invalid(r)= 1
       endif
   endif
   
;    MFE_STATUS bits: 1=OFF/ON, 2=Inb/Out Sensor, 4=Low/High Range,
;                     8=Left/Right Flipper
;                    15 = 1+2+4+8 = On, Out, High, Right
   
   hyd_cdf_varget, cdfcal, 'MFE_STATUS', mfe_status, $
     rec_start=rec_start, rec_count= rec_count

   mfe_status= mfe_status( hrec )
   b_avg_invalid= b_avg_invalid( hrec )
   
   rstat = where( ((mfe_status and 1) eq 0) or (b_avg_invalid) )
   if (rstat(0) ne -1) then begin
;       message, "Fill data found for B, no correction possible.", /cont
       bvalid(rstat) = 0 
   endif

   rokay = where( ((mfe_status and 1) eq 1) and (b_avg_invalid eq 0) )
;   if (rokay(0) eq -1) then return

   flipper_index = (mfe_status AND 8) eq  8
   range_index = (mfe_status AND 4) eq 4 
   sensor_index = (mfe_status AND 2) eq 2

;  mfe_offset_flag = 1 -> offsets in nT
;  mfe_offset_flag = 2 -> offsets in Counts
;
   hyd_cdf_varget, cdfcal, 'MFE_OFFSET_FLAG', mfe_offset_flag

;  mfe_gains: indexed by (x/y/z, Inb/Out, Low/High, Left/Right) in nT/Count
;
;
   if keyword_set( nogains ) then begin
       mfe_gains= make_array( 3, 2, 2, 4, /float, value=1.0 )
   endif else begin
       hyd_cdf_varget, CDFcal, 'MFE_GAINS', mfe_gains
   endelse

;  mfe_offset: indexed by (x/y/z, Inb/Out, Low/High, itime) in nT
;
;
   
;   read the mfe offset version.  zero indicates that no offset is
;   available from mfe.  In this case, see if a hydra calibration 
;   is available.

   hyd_cdf_varget, cdfcal, 'MFE_OFFSET_VERSION', mfe_offset_version
   if mfe_offset_version eq 0.0 then begin
       if cdf_varexist( cdfcal, 'HYD_MFE_OFFSET' ) then begin
           message, '## Using MFE Offset calculated by Hydra', /cont
           hyd_cdf_varget, cdfcal, 'HYD_MFE_OFFSET', mfe_offset
           hyd_cdf_varget, CDFcal, 'HYD_MFE_OFFSET_RAW', mfe_offset_raw
       endif else begin
           message, 'no mfe calibration of any sort found.', /cont
           hyd_cdf_varget, CDFcal, 'MFE_OFFSET', mfe_offset
           hyd_cdf_varget, CDFcal, 'MFE_OFFSET_RAW', mfe_offset_raw
       endelse
   endif else begin
       hyd_cdf_varget, CDFcal, 'MFE_OFFSET', mfe_offset
       hyd_cdf_varget, CDFcal, 'MFE_OFFSET_RAW', mfe_offset_raw
   endelse
   
   hyd_cdf_varget, CDFcal, 'MFE_OFFSET_START', mfe_offset_start
   hyd_cdf_varget, CDFcal, 'MFE_OFFSET_END', mfe_offset_end
   
;  mfe_matrix: indexed by (x/y/z, x/y/z, Inb/Out, Low/High) in nT/Count
;
;
   if keyword_set( nomatrix ) then begin
       mfe_matrix= fltarr( 3,3,2,2 )
       mfe_matrix(*,*,0,0)= [[ 1,0,0], [0,1,0], [0,0,1 ] ]
       mfe_matrix(*,*,0,1)= [[ 1,0,0], [0,1,0], [0,0,1 ] ]
       mfe_matrix(*,*,1,0)= [[ 1,0,0], [0,1,0], [0,0,1 ] ]
       mfe_matrix(*,*,1,1)= [[ 1,0,0], [0,1,0], [0,0,1 ] ]
   endif else begin
       hyd_cdf_varget, CDFcal, 'MFE_MATRIX', mfe_matrix
   endelse

;  mfe digital filter introduces a phase shift to their data.  
;  mfe_phase_offsets contains there offsets in phi.
   if mfetim and lag_corrected eq 0 then begin
        if mfe_offset_version ne 0. then begin
           hyd_cdf_varget, CDFcal, 'MFE_OFFSET_PHI', mfe_phase_offsets
       endif else begin
           if cdf_varexist( cdfcal, 'HYD_MFE_PHI_OFFSET' ) then begin
               message, 'using HYD_MFE_PHI_OFFSET', /cont
               hyd_cdf_varget, cdfcal, 'HYD_MFE_PHI_OFFSET', mfe_phase_offsets
           endif else begin
               message, 'no mfe_phi_offset found.', /cont
               message, '## using 6.0 ##', /cont
               mfe_phase_offsets= fltarr( 2,2,4 ) + 6.0
           endelse
       endelse
   endif else begin
       mfe_phase_offsets= fltarr( 2,2,4 )
   endelse
   
; break cdf epoch into h,m,s
   hr= long( rectimes mod 86400000 / 3600000. )
   mn= long( rectimes mod  3600000 /   60000. )
   sc= long( rectimes mod    60000 /    1000. )
   time= hr*10000L + mn*100L + sc ; HHMMSS

   idmode= sensor_index(*)*2 + range_index*6 + flipper_index*12
   u= where( (idmode(1:*)-idmode(0:*)) ne 0, n )
   if u(0) ne -1 then u=[0,u+1,n_elements(idmode)] else $
     u=[0,n_elements(idmode)]
   mode_start= u(0:(n_elements(u)-2))
   mode_count= u(1:*)-mode_start

;  fill itime index   
   itime= lonarr( n_elements( idmode ) )
   for imode= 0, n_elements(mode_start)-1 do begin
       i= mode_start(imode)
       rmode= mode_start(imode)+lindgen(mode_count(imode))
       mfe_time_start = $
         reform( mfe_offset_start(sensor_index(i),range_index(i),*) )
       mfe_time_end = $
         reform( mfe_offset_end(sensor_index(i),range_index(i),*) )

       time1= time(rmode)

       itime(rmode)= $
         ( time1 gt mfe_time_start(0) and time1 le mfe_time_end(0) ) * 1 + $
         ( time1 gt mfe_time_start(1) and time1 le mfe_time_end(1) ) * 2 + $
         ( time1 gt mfe_time_start(2) and time1 le mfe_time_end(2) ) * 3 + $
         ( time1 gt mfe_time_start(3) and time1 le mfe_time_end(3) ) * 4 
   
       r= where( itime(rmode) eq 0, count )       
       if count gt 0 then begin
;           message, 'MFE B Correction failed to get offset', /cont
           bvalid(rmode(r))=0
       endif
       itime= (itime-1)>0
   endfor

;  now find uniq modes also considering offsets
   idmode= sensor_index(*)*2 + range_index*6 + flipper_index*12 + itime*24
   u= where( (idmode(1:*)-idmode(0:*)) ne 0, n )
   if u(0) ne -1 then u=[0,u+1,n_elements(idmode)] else $
     u=[0,n_elements(idmode)]
   mode_start= u(0:(n_elements(u)-2))
   mode_count= u(1:*)-mode_start

   phase_out= fltarr( n_elements(t) )

   if xyz then begin
       bx= fltarr( n_elements(t) )
       by= fltarr( n_elements(t) )
       bz= fltarr( n_elements(t) )
   endif 

   if thetaphi then begin
       btheta= fltarr( n_elements(t) )
       bphi= fltarr( n_elements(t) )
       bmag= fltarr( n_elements(t) )
   endif

   for imode= 0, n_elements(mode_start)-1 do begin
       in= mode_start(imode)
       out= mode_start(imode)+mode_count(imode)-1 ; a(in:out) convention
       
       gains= mfe_gains(*,sensor_index(in),range_index(in),flipper_index(in))
       matrix= mfe_matrix(*,*,sensor_index(in),range_index(in))
       
       matrix_out=matrix
       gains_out= gains
       
       offset= mfe_offset(*,sensor_index(in),range_index(in),itime(in))
       offset_raw= mfe_offset_raw(*,sensor_index(in),range_index(in),itime(in))

       mfe_phase_offset= $
         mfe_phase_offsets( sensor_index(in),range_index(in),itime(in) ) 
       
       if n_elements( offset_phi_in ) eq 1 then begin
           message, 'overriding phi offset', /cont
           mfe_phase_offset= offset_phi_in
       endif

       if n_elements( matrix_in ) eq 9 then begin
           message, 'overriding matrix', /cont
           matrix= matrix_in
       endif

       if n_elements( gains_in ) eq 3 then begin
           message, 'overriding gains', /cont
           gains= gains_in 
       endif
       
       if n_elements( offset_raw_in ) eq 3 then begin
           message, 'overriding offset_raw', /cont
           offset_raw= offset_raw_in
           mfe_offset_flag= 0
       endif

       if n_elements( offset_in ) eq 3 then begin
           message, 'overriding offset', /cont
           offset= offset_in
           mfe_offset_flag= 0
       endif
      
       if keyword_set( verbose ) then begin
           print, '-----------------------------------'
           print, 'Mode ID:    ', idmode(in)
           print, 'si(inout),ri(highlow),fi(leftrigth),ti:', $
             sensor_index(in),range_index(in),flipper_index(in), itime(in)
           print, 'offset_raw: ', offset_raw
           print, 'offset:     ', offset
           print, 'gains:      ', gains
           print, 'matrix:     ', matrix(*,0)
           print, '            ', matrix(*,1)
           print, '            ', matrix(*,2)
           print, 'phi_off:    ', mfe_phase_offset
           print, 'lag_correct:', lag_corrected
           print, 'time res:   ', $
             hydtim ? 'hydtime' : ( mfetim ? 'mfetime' : 'hz54' )
           print, 'called from:   ', called_from()
           print, '-----------------------------------'
       endif

;       verify that we're using a valid calibration.
       if keyword_set(science_mode) then begin
           if total(abs([offset,offset_raw])) eq 0. then begin
               if hyd_interactive(dialog_parent=dialog_parent) then begin
                   x= hydra_message( [ 'No offset calibration found for', $
                                       'the B field.  Press okay to continue', $
                                       'without calibration'], $
                                     dialog_parent=dialog_parent, $
                                     /alert )
               endif else begin
                   message= $
                     [ '# No offset calibration found for      # ', $
                       '# the B field.  Press okay to continue # ', $
                       '# without calibration                  # ' ]
                   for ii=0,n_elements(message)-1 do $
                     message, message[ii], /cont
                   wait, 1.0
               endelse
           endif
       endif
       
;      clampbmag:
;      one more tweek for the hydra data.  Solve for bmag before
;      offsets such that the hydra_bmag = mfe_bmag       
       if Hz50 or hydtim and not keyword_set( noclampbmag ) then begin
           
           message, 'Clamping bmag....', /cont
           offraw = double(offset_raw)
           offraw(0) = offraw(0)-offset(0)/gains(0)
           offraw(1) = offraw(1)-offset(1)/gains(1)
           offraw(2) = offraw(2)-offset(2)/gains(2)
           
           rv= where( bvalid(in:out) )
           if rv(0) ne -1 then begin
               btmag = sqrt(bx0(in+rv)^2+by0(in+rv)^2+bz0(in+rv)^2)
               bhatx = bx0(in+rv)/btmag
               bhaty = by0(in+rv)/btmag
               bhatz = bz0(in+rv)/btmag
               offsq = total(offraw^2, /double)
               btdo = bhatx*offraw(0)+bhaty*offraw(1)+bhatz*offraw(2)
               
               desc= ((btdo)^2+bmag_interp1_sq(in+rv)-offsq) 
               bmag0p = btdo + sqrt( desc > 0.0 )
               bmag0m = btdo - sqrt( desc > 0.0 )
               
               bflags(in+rv) = bflags(in+rv) or byte( ( desc lt 0.0 ) * 2 )

               bmag0 = bmag0p
               bx0(in+rv) = bx0(in+rv)*bmag0/btmag
               by0(in+rv) = by0(in+rv)*bmag0/btmag
               bz0(in+rv) = bz0(in+rv)*bmag0/btmag
           endif                ; rv(0) ne -1
       endif

       if (mfe_offset_flag eq 1) then begin 
           
           bx1 = bx0(in:out)*gains(0) - offset(0)
           by1 = by0(in:out)*gains(1) - offset(1)
           bz1 = bz0(in:out)*gains(2) - offset(2)
           
       endif else begin 

           bx1 = (bx0(in:out)-offset_raw(0))*gains(0) - offset(0)
           by1 = (by0(in:out)-offset_raw(1))*gains(1) - offset(1)
           bz1 = (bz0(in:out)-offset_raw(2))*gains(2) - offset(2)
           
           bxerr(in:out)=bxerr(in:out)*gains(0)
           byerr(in:out)=byerr(in:out)*gains(1)
           bzerr(in:out)=bzerr(in:out)*gains(2)

           dbxs(in:out)=dbxs(in:out)*gains(0)
           dbys(in:out)=dbys(in:out)*gains(1)
           dbzs(in:out)=dbzs(in:out)*gains(2)

       endelse 

       bx2 = bx1*matrix(0,0) + by1*matrix(0,1) + bz1*matrix(0,2)
       by2 = bx1*matrix(1,0) + by1*matrix(1,1) + bz1*matrix(1,2)
       bz2 = bx1*matrix(2,0) + by1*matrix(2,1) + bz1*matrix(2,2)
       
       phase_out(in:out)= sc_phase(in:out) - !dtor*mfe_phase_offset
       
       if keyword_set( fixed_payload ) then begin
           zeta= sc_phase(in:out) - !dtor*mfe_phase_offset
           rotate=1
       endif else begin
           zeta= - !dtor*mfe_phase_offset
           if zeta ne 0.0 then rotate=1 else rotate=0
       endelse
       if rotate then begin
           bx3= bx2*cos(zeta)-by2*sin(zeta)
           by3= bx2*sin(zeta)+by2*cos(zeta)
           bz3= temporary( bz2 )       
       endif else begin
           bx3= temporary( bx2 )
           by3= temporary( by2 )
           bz3= temporary( bz2 )       
       endelse

;;        do spin_tone removal filter
;       b3= transpose( reform( [ bx3, by3, bz3 ], n_elements(bx3), 3 ) )
;       b4= gen_bfield_spin_filter( cdfcal, t, phase_out, b3, $
;                                   fixed_payload=fixed_payload )
;       stop

;        fill in data gaps
       if keyword_set( fill_in ) then begin
           bflags1= bflags(in:out)
           bvalid1= bvalid(in:out)
           gen_bfield_fill_in, bx3, by3, bz3, bflags1, bvalid1, t
           bflags(in:out)= bflags1
           bvalid(in:out)= bvalid1
       endif

       if xyz then begin
           bx(in:out)= bx3
           by(in:out)= by3
           bz(in:out)= bz3
       endif
       
       if thetaphi then begin
           bmag1= fltarr( out-in+1 )

           r= where( bvalid(in:out) eq 0 )
           if r(0) ne -1 then bmag1(r) = 1e31                      

           rv= where( bvalid(in:out) eq 1 )
           if rv(0) ne -1 then begin
               bmag1(rv) = sqrt(bx3(rv)^2 +by3(rv)^2+bz3(rv)^2)

               bhatx = bx3(rv)/bmag1(rv)
               bhaty = by3(rv)/bmag1(rv)
               bhatz = bz3(rv)/bmag1(rv)

               btheta(in+rv)= acos(bhatz)
               phib1= acos( ( bhatx/sin(btheta(in+rv))) < 1.0 > (-1.0) )
               r = where(bhaty LT 0.)
               if r(0) ne -1 then phib1(r)= 2*!pi - phib1(r)
               bphi(in+rv)= phib1
           endif
           bmag(in:out)= bmag1
       endif
   endfor

;   r=where(bvalid eq 0)
;   if r(0) ne -1 then begin
;       bx(r)=rfill_new
;       by(r)=rfill_new
;       bz(r)=rfill_new
;   endif
   theend:

END



