pro get_hr_bmag, f, rec_start, rec_count, bmag, t, $
                 hydtim=hydtim, Hz50=Hz50, nofilter=nofilter, $
                 reset=reset, cdfcal=cdfcal, bvalid=bvalid, $
                 xyz=xyz, bx=bx, by=by, bz=bz, lag_corrected=lag_corrected, $
                 phib=phib, thetab=thetab, time=time, hbvalid=bvalid_h, $
                 dbxs=dbxs, dbys=dbys, dbzs=dbzs

use_bmag_8Hz_nolag=keyword_set(lag_corrected)
;  get_hr_bmag 
; 
;  purpose:  retrieves 8Hz Bfield magnitude data from hydra l1 file.
;      Handles interpolation to hydra sample times.  Also, maintains
;      a buffer of the last load, in case of multiple reads.
;
;  /reset      clears internal buffer
;  /xyz   return B angles as well (x,y,z comp interpolated as well)
;
;  /lag_corrected corrects for MFE onboard time-filtering
;
;
;  written: ??/??/?? Jeremy Faden
;  history: Dec 5, 1997  Added buffer and above comments.
;           Jan 28 1998  Moderate testing.

;  buffer is identified by "BLOCK_TIME" of first measurement.
   common get_hr_bmag_common, id_, rec_start_, rec_count_, $
     bmag_, t_, bvalid_, filter_

   xyz= keyword_set( xyz )
   if n_elements( cdfcal ) eq 0 then cdfcal=f

   if keyword_set( reset ) then begin
       if n_elements( id_ ) gt 0 then begin
           message, 'clearing buffer', /cont
           z= temporary(id_)
       endif
       return
   endif
     
   if n_elements( id_ ) gt 0 then begin
       hyd_cdf_varget, f, 'BLOCK_TIME', id, rec_start=rec_start_
       if rec_start_ le rec_start and $
         rec_start_ + rec_count_ ge rec_start + rec_count  and  $ 
         id_ eq id and filter_ eq keyword_set(nofilter) and $
         not xyz then begin
           message, 'Using stored data...', /cont
           if keyword_set( hydtim ) then rhyd= indgen(384)*2+1 else $
             rhyd= indgen(768)
           bmag= bmag_(rhyd,*)
           t=t_(rhyd,*)
           bvalid= bvalid_(rhyd,*)           
           r= ( rec_start - rec_start_ ) + lindgen( rec_count )
           bmag= bmag(*,r)
           t= t(*,r)
           bvalid= bvalid(*,r)

           nxs=n_elements(bmag)
           dbxs=replicate(0.0,nxs)
           dbys=dbxs
           dbzs=dbxs

           return
       endif
   endif

;  get raw mfe data at 8Hz
   read_raw_mfe, f, rec_start, rec_count, bx, by, bz, mfetime, $
     bvalid=bvalid, nofilter= keyword_set( nofilter ), cdfcal=cdfcal, $
     bflags=bflags
   r= check_math( 0,1 )         ; turn off math err messages.

   if use_bmag_8Hz_nolag eq 1 then begin       
       message, 'Applying lag correction (Using bmag_8hz_nolag)', /cont
       bmag=bmag_8hz_nolag(mfetime, bx, by, bz, bvalid, $
                           phi=phi8, theta=theta8, $
                           dbxs=dbxs, dbys=dbys, dbzs=dbzs)
       rbmag=where(abs(bmag) gt 1d+10 and bvalid eq 1)
       if rbmag(0) ne -1 then begin
           message, 'Bmag error found'
       endif
   endif else begin
       bmag= sqrt( bx^2 + by^2 + bz^2 )

   endelse
   r= check_math( 0,0 )         ; we've squared fill values.
   
;   check for case where there is no valid timetag data and put in
;   bogus monotonic timetags  
   if max( mfetime ) lt 0 then begin
       message, 'No valid timetags found.', /cont       
       mfetime=findgen(n_elements(mfetime))*0.125
       bvalid(*)=0
   endif

;  get Hydra time tags.   
   hydtim= keyword_set( hydtim )
   Hz50=  keyword_set( Hz50 )
   
   hyd_cdf_varget, f, 'BLOCK_TIME', block_time, $
     rec_start=rec_start, rec_count=rec_count
   id= block_time(0)
   block_time= block_time(*)
   block_time= block_time mod 86400000. / 1000 ; convert to seconds
   hyd_cdf_varget, f, 'BLOCK_MODE', block_mode, $
     rec_start=rec_start
   hyd_cdf_varget, f, 'TIME_OFFSET', time_offset, rec_start=block_mode-1

   if hydtim then begin
       n=384
       t_h= make_array( n, /float, value=1 ) # block_time + $
         time_offset # make_array( rec_count, /float, value=1 )
   endif else begin
       n=768           
       dt1= -0.0138             ; first Bangle is 13.8ms before DDEIS
       dt2= +0.0042             ; second Bangle comes 4.2ms later then DDEIS 
       time_offset= rebin( time_offset, 768, /sample )
       time_offset(indgen(384)*2)=time_offset(indgen(384)*2)+dt1
       time_offset(indgen(384)*2+1)=time_offset(indgen(384)*2+1)+dt2
       t_h= make_array( n, /float, value=1 ) # block_time + $
         time_offset # make_array( rec_count, /float, value=1 )
   endelse

;  interpolate to hydra time tags.
   rnv= where( bvalid eq 0 )
   rv= where( bvalid eq 1 )
   if rnv(0) ne -1 then bmag(rnv)=1e31
   delta_t= mfetime(1)-mfetime(0)
   
   dt= mfetime(1:*)-mfetime(0:*)
   r= where( dt lt 0. )
   if r(0) ne -1 then begin
       message, 'Masking bogus time in MFE data.', /cont
       mfetime(r)= mfetime(r+1)
       bvalid(r)= 0
   endif
   findex= hyd_findex( mfetime, t_h(*) )
   bmag_h= interpolate( bmag, findex )
   if (use_bmag_8Hz_nolag eq 1) then begin
       theta_h= interpolate( theta8, findex )
       phi_h= wrap_phi(interpolate( unwrap_phi(phi8), findex ))
       dbxs=interpolate( dbxs, findex )
       dbys=interpolate( dbys, findex )
       dbzs=interpolate( dbzs, findex )
   endif else begin

       nxs=n_elements(bmag_h)
       dbxs=replicate(0.0,nxs)
       dbys=dbxs
       dbzs=dbxs

   endelse

   if keyword_set( xyz ) then begin
       bx_h= interpolate( reform(bx,n_elements(bx)), findex )
       by_h= interpolate( reform(by,n_elements(by)), findex )
       bz_h= interpolate( reform(bz,n_elements(bz)), findex )
   endif

;  propogate fill into interpolated data
   if rv(0) ne -1 then begin
       ceil= max( bmag(rv) )
       rnv_h= where( bmag_h gt ceil )
   endif else rnv_h= lindgen( n_elements( bmag_h ) )
   bvalid_h= intarr( n_elements( bmag_h ) ) + 1
   bflags_h= bytarr( n_elements( bmag_h ) ) 
   if rnv_h(0) ne -1 then bvalid_h(rnv_h)= 0
   if (size(bflags,/type) ne 0) then $ 
     bflags_h(*)= bflags(floor(findex)) or bflags(floor(findex)+1)
   
   bmag= reform(bmag_h,n,rec_count)
   if (use_bmag_8Hz_nolag eq 1) then begin
       phib=reform(phi_h,n,rec_count)   
       thetab=reform(theta_h,n,rec_count)   
   endif

   bvalid= reform( bvalid_h, n, rec_count )
   if xyz then begin
       bx= reform( bx_h, n, rec_count )
       by= reform( by_h, n, rec_count )
       bz= reform( bz_h, n, rec_count )
   endif
   t= t_h

; store in buffer for next guy
   if (Hz50) then begin
       id_= id
       rec_start_= rec_start
       rec_count_= rec_count
       bmag_= bmag
       bvalid_= bvalid
       t_= t       
       filter_= keyword_set( nofilter )
   endif

   return
end
