pro read_raw_mfe, f, rec_start, rec_count, bx, by, bz, t, $
                  hydtim=hydtim, Hz50=Hz50, bvalid=bvalid, nofilter=nofilter, $
                  cdfcal=fcal, bflags=bflags, recnum=recnum, $
                  lag_corrected=lag_corrected, angle_error=angle_error, $
                  bxerr=bxerr, byerr=byerr, bzerr=bzerr, $
                  dbxs=dbxs, dbys=dbys, dbzs=dbzs, nobmag=nobmag

;  read_raw_mfe returns raw bx, by, bz data.
;
;   /hydtim   return (interpol(8Hz Bmag data)->27Hz) + 27Hz Hydra angles.
;   /Hz50     return (interpol(8Hz BMag data)->54Hz) + 54Hz Hydra angles.
;   recnum    returns the record from which each measurement came.
;              This can be useful in debugging.
;
;   /lag_corrected   Correct 27/54 Hz data for on-board MFE time-filtering.
;
;  History:
;    sorted.
;    1-2x-98  jbf  runtime valid filters added
;    1-29-98  jbf  reads in bvalid from level1, if available.
;    5-23-01  rdh  ppa angles are filtered.

  rfill= -1e31
  
  filter_ppa_angles=1
  
  verbose= getenv('verbose') eq '1'
  
  if verbose then begin
      message, 'Entered ----------------', /cont
  endif
  
  if n_elements( fcal ) eq 0 then fcal=f
  
;  CHECK TO SEE IF MFE HIGH RES BFIELD DATA EXISTS IN L1 FILE
  highres= cdf_varexist( f, 'B_HIGH_RAW' )
   if highres then begin
       cdf_control, f, var='B_HIGH_RAW', get_var_info=r
       if r.maxrec eq -1 then highres=0
   endif else begin
       message, 'No high resolution B field data found.', /cont
   endelse

   lag_corrected=keyword_set(lag_corrected)

   hydtim= keyword_set( hydtim )
   Hz50= keyword_set( Hz50 )

  if verbose then begin
      message, 'time res   :   '+ $
        ( hydtim  ? 'hydtime' : ( hz50 ? 'hz54' : 'mfetim' ) ), /cont
      message, 'lag_correct: '+strtrim(keyword_set(lag_corrected),2), /cont     
  endif

   if hydtim or Hz50 then begin ; interpolate to hydra samples

       hyd_cdf_varget, f, 'BLOCK_MODE', modei, rec_start= rec_start
       hyd_cdf_varget, f, 'ELE_BIAS', elebiasi, rec_start= rec_start
       hyd_cdf_varget, f, 'ION_BIAS', ionbiasi, rec_start= rec_start
       
       counts_field= string(modei,format='(i3.3)')+'_e_'+ $
         string(elebiasi(0),format='(i2.2)')+'_'+ $
         string(elebiasi(1),format='(i2.2)')+'_i_'+ $
         string(ionbiasi(0),format='(i2.2)')+'_'+ $
         string(ionbiasi(1),format='(i2.2)')

       if not cdf_varexist( fcal,'VALIDB_'+counts_field) then begin
           message, 'runtime bvalid filter will no longer be used.', /cont
           message, 'visit hydra''s data serving website to collect ', /cont
           message, 'a new ddcal file.', /cont
           stop
;           bvalid_h= hydra_bdata_valid( time=t, $
;                                   /angles, theta=thetab, phi=phib ) 
       endif
       
       hyd_cdf_varget, f, 'BLOCK_INDEX', offset, $
         rec_start=rec_start, rec_count=rec_count       

       recnum= make_array(768,value=1,/float) # indgen(rec_count) + rec_start
       for i=0,rec_count-1 do $
         recnum(*,i)= recnum(*,i) + findgen(768)/768

       test= offset(1:*)-offset
       ii= where( test ne 1, count )+1
       if count eq 0 then ii=[0,rec_count] else ii=[0,ii,rec_count]
       
       for iii= 0,n_elements(ii)-2 do begin
           rs1= rec_start+ii(iii)
           rc1= ii(iii+1)-ii(iii)
           
           hyd_cdf_varget, f, 'BLOCK_MODE', modei, rec_start= rs1
           hyd_cdf_varget, f, 'ELE_BIAS', elebiasi, rec_start= rs1
           hyd_cdf_varget, f, 'ION_BIAS', ionbiasi, rec_start= rs1
       
           counts_field= string(modei,format='(i3.3)')+'_e_'+ $
             string(elebiasi(0),format='(i2.2)')+'_'+ $
             string(elebiasi(1),format='(i2.2)')+'_i_'+ $
             string(ionbiasi(0),format='(i2.2)')+'_'+ $
             string(ionbiasi(1),format='(i2.2)')
           
           hyd_cdf_varget, f, 'BLOCK_INDEX', offset, $
             rec_start=rs1, rec_count=rc1
           offset= offset(0)

           hyd_cdf_varget, f, 'PHIB_'+counts_field, phib1, $
             rec_start=offset-1, rec_count=rc1
       
           hyd_cdf_varget, f, 'THETAB_'+counts_field, thetab1, $
             rec_start=offset-1, rec_count=rc1

           hyd_cdf_varget, fcal, 'VALIDB_'+counts_field, bvalid8_1, $
             rec_start=offset-1, rec_count=rc1
           
           if iii eq 0 then begin
               phib=phib1
               thetab=thetab1
               bvalid8=bvalid8_1
           endif else begin
               phib= transpose( [ transpose( phib ), transpose( phib1 ) ] )
               thetab= transpose( [ transpose( thetab ), transpose( thetab1 ) ] )
               bvalid8= transpose( [ transpose( bvalid8 ), transpose( bvalid8_1 ) ])
           endelse

       endfor
       
       if (highres) then begin
           get_hr_bmag, f, rec_start, rec_count, bmag, t, $
             hydtim=hydtim, Hz50=Hz50, bvalid=bvalid_i, $
             nofilter=keyword_set(nofilter), $
             /xyz, bx=bx_i, by=by_i, bz=bz_i, cdfcal=fcal, $
             lag_corrected=lag_corrected, $
             phib=phib0, thetab=thetab0, $
             dbxs=dbxs, dbys=dbys, dbzs=dbzs
           if nobmag then begin               
               bmag(*)= 1.0
               bvalid_i(*)=1
           endif           

       endif else begin
           
           hyd_cdf_varget, f, 'B_AVG', b_avg, $
             rec_start=rec_start, rec_count=rec_count
           bmag1= sqrt( total( b_avg^2, 2 ) )

           hyd_cdf_varget, f, 'BLOCK_TIME', block_time, $
             rec_start=rec_start, rec_count=rec_count
           block_time= block_time(*)
           block_time= block_time mod 86400000. / 1000 ; convert to seconds
           hyd_cdf_varget, f, 'BLOCK_MODE', block_mode, $
             rec_start=rec_start
           hyd_cdf_varget, f, 'TIME_OFFSET', time_offset, rec_start=block_mode-1

           if (hydtim) then begin
               n=384 
               t= make_array( n, /float, value=1 ) # block_time + $
                 time_offset # make_array( rec_count, /float, value=1 )
           endif else begin 
               n=768
               dt1= -0.0138     ; first Bangle is 13.8ms before DDEIS
               dt2= +0.0042     ; second Bangle comes 4.2ms later then DDEIS 
               time_offset= rebin( time_offset, 768, /sample )
               time_offset(indgen(384)*2)=time_offset(indgen(384)*2)+dt1
               time_offset(indgen(384)*2+1)=time_offset(indgen(384)*2+1)+dt2
               t= make_array( n, /float, value=1 ) # block_time + $
                 time_offset # make_array( rec_count, /float, value=1 )
           endelse
           bmag= rebin( bmag1, rec_count*n, /sample )
           bvalid_i= make_array( size= size(bmag), value=1, /int )
           if papcoiscaller() then begin
               common plot_composer, widgetData
               if n_elements( widgetData ) ne 0 then $
                 parent= widget_info( widgetData.dw_draw, /parent ) $
               else $
                 parent=0
               r= widget_message( dialog_parent=parent, $
                                  [ 'MFE resolution B field', $
                                    'data not available.', $
                                    'Using 13.8 second resolution', $
                                    'B field magnitude for corrections.' ], $
                                  /info )
           endif
       endelse           

;       decode hydra bfield filter.
       if keyword_set(nofilter) then begin
           bvalid_h= make_array( size=size( thetab ), /int, value=1 )
       endif else begin
           bvalid_h= make_array( 768, rec_count, /int )
           idx= indgen( 96 ) * 8
           for i=0,7 do $
             bvalid_h( idx+i,* ) = ishft( bvalid8, -i ) and 1B
       endelse           

;       discard the odd measurements if DDEIS resolution (27Hz) is requested.
       if hydtim then begin
           rhyd= indgen(384)*2+1 ; use every other meas
       endif else begin
           rhyd= indgen(768)
       endelse
       phib= phib( rhyd, * )
       thetab= thetab( rhyd, * )
       bvalid_h= bvalid_h( rhyd,* )
       recnum= recnum( rhyd, * )

       bvalid= bvalid_i*bvalid_h
       bflags= bvalid_h ne 1 
       
       phib= ( phib + 0.5 ) * (2 * !pi / 256.)
       thetab= (thetab + 0.5 ) * (2 * !pi / 256.)

       if keyword_set(filter_ppa_angles) then begin
                                ;
                                ;  Note: We filter across invalid intervals
                                ;        since the invalid interval data
                                ;        is conservative, so it should
                                ;        be okay for the
                                ;        near-boundary-okay points
                                ;        to depend on the
                                ;        near-boundary-invalid
                                ;        points.
                                ; 
                                ;        RDH 5/23/01
                                ;
           thetab=hyd_filter_angle(thetab)
           cos_phib_f=hyd_filter_angle(phib, /use_cos)
           sin_phib_f=hyd_filter_angle(phib, /use_sin)
           phib=atan(sin_phib_f,cos_phib_f)
       endif

       cosphi = cos(phib)
       sinphi = sin(phib)
       costheta = cos(thetab)
       sintheta = sin(thetab)

       bx= (sintheta*cosphi*bmag)(*)
       by= (sintheta*sinphi*bmag)(*)
       bz= (costheta*bmag)(*)
       t=t(*)

       if (highres and lag_corrected) then begin
           
           gb_lag_corr_angle_error, t, thetab, phib, thetab0, phib0, $
             bvalid, rfill, angle_error

;           bxerr=bx-bcomponent_filter( t, bx, label='Bx', bvalid=bvalid,$
;                                     simple=2)
;           byerr=by-bcomponent_filter( t, by, label='By', bvalid=bvalid,$
;                                     simple=2)
;           bzerr=bz-bcomponent_filter( t, bz, label='Bz', bvalid=bvalid,$
;                                     simple=2)

           bxerr=bx*0+rfill
           byerr=by*0+rfill
           bzerr=bz*0+rfill

           rv=where(bvalid eq 0)
           if (rv(0) ne -1) then begin
               bxerr(rv)=rfill
               byerr(rv)=rfill
               bzerr(rv)=rfill
           endif
       endif else begin
           nxs=n_elements(bx)
           bxerr=replicate(0.0,nxs)
           byerr=bxerr
           bzerr=bxerr
       endelse


;      find the phi offset between MFE and Hydra's bfield stream
       l1_mfe_get_status, f, rec_start, rec_count, $
         sensor, range, flipper, itime

       hyd_cdf_varget, f, 'MFE_OFFSET_PHI', mfe_phase_offset_table

       mfe_phase_offset= mfe_phase_offset_table( range, flipper, itime ) * $
         !dtor * (-1.0)
       
       if hydtim then r= make_array( 384, value=1 ) else $
         r= make_array( 768, value=1 )

       cos_mpo= r # cos(mfe_phase_offset)
       sin_mpo= r # sin(mfe_phase_offset)

;        bx_i, by_i come from the mfe 8hz data and need to be
;        phase_offset corrected.
       r=  bx_i * cos_mpo  -  by_i * sin_mpo
       by_i=  bx_i * sin_mpo + by_i * cos_mpo
       bx_i=  r       
       
       r= where( bvalid_i eq 1 and bvalid eq 0 )
       if r(0) ne -1 then begin
           bx(r)= bx_i(r)
           by(r)= by_i(r)
           bz(r)= bz_i(r)
           bflags(r)= bflags(r) or 4
           bvalid(r)= 1
       endif
   endif else begin
       
;      find corresponding mjf.  We need 40, so we'll pick forty that
;         evenly surround the hydra blocks.

       hyd_cdf_varget, f, 'BLOCK_MJF', block_mjf, $
         rec_start=rec_start, rec_count=rec_count       
       in= min( block_mjf, max=out )

       hyd_cdf_varget, f, 'BLOCK_TIME', block_epoch, $
         rec_start=rec_start, rec_count=rec_count       
       blocktime= block_epoch mod 86400000 / 1000

       
       hyd_cdf_varget, f, 'B_HIGH_TIME', mjftime, $
         rec_start= 0, rec_count= cdf_nrec(f,'B_HIGH_TIME')
       mjftime= reform( mjftime )
       
       while ( ( mjftime[in] gt blocktime[0]) and (in gt 0) ) do $
         in=in-1
       
       nmjf= n_elements( mjftime )
       while ( ( mjftime[out] lt blocktime[rec_count-1] ) and (out lt nmjf-1) ) do $
         out=out+1

       num= out-in+1            ; grab two extras block:  mfe data lags hydra 
       if num lt 40 then begin  ; filter needs at least 40 mjf
           d= (40-num)/2
           in= (in-d)>0
           cdf_control, f, var='B_HIGH_RAW', get_var_info=r
           if in+40 gt r.maxrec then in=r.maxrec-39          
           num=40
       endif       
       
       hyd_cdf_varget, f, 'B_HIGH_RAW', b_raw, $
         rec_start= in, rec_count= num
       
       hyd_cdf_varget, f, 'B_HIGH_TIME', mjftime, $
         rec_start= in, rec_count= num
       mjftime= reform( mjftime )

       hyd_cdf_varget, f, 'B_HIGH_DT', deltat, $
         rec_start= in, rec_count= num
       deltat= reform( deltat )

       hyd_cdf_varget, f, 'B_HIGH_STATUS', b_high_status, $
         rec_start= in, rec_count= num

       r= where( mjftime eq -9999., count )
       for i=0,count-1 do begin
           r1= r(i)
           if r1 gt 0 then begin
               mjftime(r1)= mjftime(r1-1)+deltat(r1-1)*83
               deltat(r1)= 0.
           endif
       endfor


       r= where( mjftime ne rfill )
       if r(0) eq -1 then begin
           message, 'File does not contain valid MFE timetags', /cont
           b_raw(*)=0           ; invalidate data           
           mjftime(*)=0
           deltat(*)=0
       endif
       
       nn= n_elements( b_raw(*,0,0) ) 

       bx= intarr( num*nn )
       by= intarr( num*nn )
       bz= intarr( num*nn )
       t= dblarr( num,nn )

;  fill t
       offsets= intarr( nn, num )
       for i=0,num-1 do begin
           offsets( *, i ) = indgen(nn)
       endfor

       t(*)= rebin( mjftime, num*nn, /sample ) + offsets(*) * $
         rebin( deltat, num*nn, /sample )
       
       bx= reform(b_raw(*,0,*), nn, num )
       by= reform(b_raw(*,1,*), nn, num )
       bz= reform(b_raw(*,2,*), nn, num )

   
;      initialize hydra's bfield filter 
       if not keyword_set(nofilter) and cdf_varexist( fcal,'B_HIGH_VALID') $
         then begin
           use_runtime_mask=0
           hyd_cdf_varget, fcal, 'B_HIGH_VALID', bvalid8, $
             rec_start=in, rec_count=num
           bmask= make_array( size=size( bx ) )
           idx= indgen( 11 ) * 8
           for i=0,7 do $
             bmask( idx+i,* ) = ishft( bvalid8, -i ) and 1B
       endif else begin
           message, 'using runtime  mask.', /cont
           use_runtime_mask=1
           bmask= make_array( size=size(bx),value=1,/int )
       endelse

;      valid=0 indicates data is fill data.
       valid= make_array( size= size(bz), value=1, /int )

;      filter when MFE says the whole frame is not valid.
       r= where( b_high_status eq 128 )
       if r(0) ne -1 then begin
           valid(*,r)= 0
       endif

;      mfe tm block ends before the measurement where:
;        two components = -1      or
;        two components = 0, other= -1
;        three components = 0

       n1= ( ( bx eq -1 ) + ( by eq -1 ) + ( bz eq -1 ) )
       n0= ( ( bx eq 0 ) + ( by eq 0 ) + ( bz eq 0 ) )
       
       r= where( ( n1 ge 2 ) or $
                 ( n0 eq 2 and n1 eq 1 ) or $
                 ( n0 eq 3 ) )
       
       mjfr= r mod nn           ; index within mjf
       r_mjf= r / 83            ; which mjf r is in
       
       u= uniq(r_mjf)
       u= [0,u(0:(n_elements(u)-2))+1] ; pointer to first of each mjf
       
       for i=0,n_elements(u)-1 do begin
           valid( mjfr(u(i)):*, r_mjf(u(i)) ) = 0
       endfor
       
       r= where( mjftime eq -1d31 )
       if r(0) ne -1 then begin
           valid(*,r)=0
       endif
       
       r= where( bx eq -32768 and $
                 by eq -32768 and $
                 bz eq -32768 )
       if r(0) ne -1 then begin
           valid(r)=0
       endif

;      remove fill data since the timetags overlap.
       r= where( valid eq 1 )
       if r(0) eq -1 then begin
           bvalid= make_array( size=size(t), /int, value=0 )
           message, 'Exiting with bvalid(*)=0  -----------', /cont
           return
       endif

       t=t(r)
       bx=bx(r)
       by=by(r)
       bz=bz(r)
       valid=valid(r)
       bmask=bmask(r)

;      occasionally mfetimes are out of order.  Make invalid two
;      measurements that are out of order.
       r= where( t(1:*) le t(0:*) )
       if r(0) ne -1 then begin
           valid(r)=0
           valid(r+1)=0
           t(r)= t(r+1)
       endif

;      additional validity test: ajacent zero measurements in x or y
;      are bogus.  Jan 5, 1998
       yiszero= by eq 0 
       xiszero= bx eq 0
       rmask = where( yiszero(1:*) and yiszero(0:*) )
       if rmask(0) ne -1 then valid([rmask,rmask+1])=0
       rmask = where( xiszero(1:*) and xiszero(0:*) )
       if rmask(0) ne -1 then valid([rmask,rmask+1])=0
       
       bx=float(bx)
       by=float(by)
       bz=float(bz)

       rnv= where( valid eq 0  )   
       if rnv(0) ne -1 then begin
           bx(rnv)= 1e31
           by(rnv)= 1e31
           bz(rnv)= 1e31
       endif
       
       if not keyword_set( nofilter ) and use_runtime_mask then begin
           message, 'using runtime bvalid filter.', /cont
           message, 'runtime filter no longer supported.'
           bmask= hydra_bdata_valid( time=t, $
                                     /xyz, bx=bx, by=by, bz=bz, fill=1e31)
       endif
       bvalid= valid*bmask
       bflags= byte( bmask ne 1 )

       bxerr=make_array( value=-1e31, size=size(t) )
       byerr=make_array( value=-1e31, size=size(t) )
       bzerr=make_array( value=-1e31, size=size(t) )

       dbxs=make_array( value=-1e31, size=size(t) )
       dbys=make_array( value=-1e31, size=size(t) )
       dbzs=make_array( value=-1e31, size=size(t) )

   endelse

  if verbose then begin
      message, 'Exiting ----------------', /cont
  endif
   
   return
end

