pro solve_8hz_linear, bx, by, bz, t8, bxn, byn, bzn, $
              plot_values=plot_values, print=print
;
;   bx,by,bz,t8 input 8 Hz (after 7/8,1/8 filter)
;
;   bxn,byn,bzn output 8 Hz (before 7/8,1/8 filter)
;
;   Solve Matrix for 8 Hz points that, when linearly interpolated,
;   produce the 8 Hz points.
;
;      11    j                        12
;   1 --- /7\   (12-j)    (12)     /7\    (0)
;   - >   |-|  B       = B      -  |-|   B
;   8 --- \8/   before    after    \8/    after
;     j=0
;
;
;    k          k    (12)       /       k    (0)    \   
;   B       =  ---  B        +  | 1 -  ---  B       |
;    before    12    before     \      12    before /   
;
;
;
;
;                                              12
;       0               12         (12)     /7\    (0)
;   A  B         +  A  B        = B      -  |-|   B
;    1  before       2  before     after    \8/    after
;
;
;
;
;   M . B       =  R
;        before     after
;
;
use_matrix=0                    ; 1=Use Matrix Equation
                                ; 0=Solve Term-By-Term
solve_tridiagonal=1             ; 1=Use Tridiagonal (if use_matrix==1)
                                ; 0=Use SVSOL
use_b0_8Hz_at_start=0           ; 1=Use B_MFE_8Hz as first point
                                ; 0=Use Slope of B_MFE_8HZ between
                                ;                   1st & 2nd points.

plot_individual_intervals=keyword_set(plot_values)

one_eighth=1.0/8.0
seven_eights=7.0/8.0

a1=one_eighth*total((seven_eights^findgen(12))*(findgen(12)/12.0), /double)
a2=one_eighth*total((seven_eights^findgen(12))*(1.0-findgen(12)/12.0), /double)

n=n_elements(bx)-1
;n=2

if use_matrix then begin
    if not solve_tridiagonal then begin
        m=identity(n)*a2
        for i=1,n-1 do m(i-1,i)=a1
    endif else begin
        ma=replicate(a1,n)
        mb=replicate(a2,n)
        mc=replicate(0.0,n)
    endelse
endif else begin
    ma=replicate(a1,n)
    ma(0)=0.0
    mb=replicate(a2,n)
endelse
rx0=fltarr(n)
ry0=fltarr(n)
rz0=fltarr(n)
rx0(0:n-1)=bx(1:n)-(seven_eights^12)*bx(0:n-1)
ry0(0:n-1)=by(1:n)-(seven_eights^12)*by(0:n-1)
rz0(0:n-1)=bz(1:n)-(seven_eights^12)*bz(0:n-1)


rx=rx0
ry=ry0
rz=rz0

if use_b0_8Hz_at_start then begin
    rx(0)= rx(0)-a1*bx(0)
    ry(0)= ry(0)-a1*by(0)
    rz(0)= rz(0)-a1*bz(0)
endif else begin
    mb(0)=mb(0)+a1
    
    rx(0)= rx(0)-a1*(bx(0)-bx(1))
    ry(0)= ry(0)-a1*(by(0)-by(1))
    rz(0)= rz(0)-a1*(bz(0)-bz(1))
endelse

if use_matrix then begin
    if not solve_tridiagonal then begin
        svdc, m, w, u, v
        bxn=svsol(u, w, v, rx)
        byn=svsol(u, w, v, ry)
        bzn=svsol(u, w, v, rz)
    endif else begin
        bxn=trisol(ma,mb,mc,rx)
        byn=trisol(ma,mb,mc,ry)
        bzn=trisol(ma,mb,mc,rz)
    endelse
endif else begin

    bxn=fltarr(n)
    byn=fltarr(n)
    bzn=fltarr(n)
    bxn(0)=rx(0)/mb(0)
    byn(0)=ry(0)/mb(0)
    bzn(0)=rz(0)/mb(0)
    for i=1,n_elements(ma)-1 do begin
        bxn(i)=(rx(i)-ma(i)*bxn(i-1))/mb(i)
        byn(i)=(ry(i)-ma(i)*byn(i-1))/mb(i)
        bzn(i)=(rz(i)-ma(i)*bzn(i-1))/mb(i)
    endfor
endelse

if use_b0_8Hz_at_start then begin
    bxstart=bx(0)
    bystart=by(0)
    bzstart=bz(0)
endif else begin
    bxstart=-(bx(1)-bx(0))+bxn(0)
    bystart=-(by(1)-by(0))+byn(0)
    bzstart=-(bz(1)-bz(0))+bzn(0)
endelse
bxn=[bxstart,bxn]
byn=[bystart,byn]
bzn=[bzstart,bzn]

if keyword_set(plot_individual_intervals) then begin

    t100=t8(0)+findgen(n*12)*(t8(1)-t8(0))/12.0
    bxnn=interpol(bxn, t8(0:n), t100)
    bynn=interpol(byn, t8(0:n), t100)
    bznn=interpol(bzn, t8(0:n), t100)
    
    bxll=bxnn
    byll=bynn
    bzll=bznn
    
    bxll(0)=bx(0)
    byll(0)=by(0)
    bzll(0)=bz(0)
    for i=1,n*12-1 do begin
        bxll(i)=seven_eights*bxll(i-1)+one_eighth*bxnn(i)
        byll(i)=seven_eights*byll(i-1)+one_eighth*bynn(i)
        bzll(i)=seven_eights*bzll(i-1)+one_eighth*bznn(i)
    endfor
    
    n=5 < n
    rr=where(t100-t8(n-1) ge -0.017, nrr)
    if nrr gt 0 then n54=rr(0) else n54=n_elements(t100)
    if keyword_set(print) then begin
        set_ps, /landscape
    endif
    set_page, nv=2, nh=2
    min=min([bx(0:n-1),bxn(0:n-1),bxnn(0:n54-1),bxll(0:n54-1)],max=max)
    yrange=[min-0.1*abs(min),max+0.1*abs(max)]
    plot, t8, bx, chars=1.5, xrange=[t8(0),t8(n)], psym=1, yrange=yrange, $
      title='Bx RAW', xtitle='T (Seconds)', ytitle='Bx (counts)', $
      xtickformat='tick_hms_whole', xticks=xticks
    oplot, t8, bxn, psym=4
    oplot, t100, bxnn
    oplot, t100, bxll, psym=3
    
    min=min([by(0:n-1),byn(0:n-1),bynn(0:n54-1),byll(0:n54-1)],max=max)
    yrange=[min-0.1*abs(min),max+0.1*abs(max)]
    plot, t8, by, chars=1.5, xrange=[t8(0),t8(n)], psym=1, yrange=yrange, $
      title='By RAW', xtitle='T (Seconds)', ytitle='By (counts)', $
      xtickformat='tick_hms_whole', xticks=xticks
    oplot, t8, byn,psym=4
    oplot, t100, bynn
    oplot, t100, byll, psym=3
    
    min=min([bz(0:n-1),bzn(0:n-1),bznn(0:n54-1),bzll(0:n54-1)],max=max)
    yrange=[min-0.1*abs(min),max+0.1*abs(max)]
    plot, t8, bz, chars=1.5, xrange=[t8(0),t8(n)], psym=1, yrange=yrange, $
      title='Bz RAW', xtitle='T (Seconds)', ytitle='Bz (counts)', $
      xtickformat='tick_hms_whole', xticks=xticks
    oplot, t8, bzn,psym=4
    oplot, t100, bznn
    oplot, t100, bzll, psym=3
    
    bmag8=sqrt(bx^2+by^2+bz^2)
    bmagn=sqrt(bxn^2+byn^2+bzn^2)
    bmagnn=sqrt(bxnn^2+bynn^2+bznn^2)
    min=min([bmag8(0:n-1),bmagn(0:n-1),bmagnn(0:n54-1)],max=max)
    yrange=[min-0.1*abs(min),max+0.1*abs(max)]
    plot, t8, bmag8, chars=1.5, xrange=[t8(0),t8(n)], psym=1, yrange=yrange, $
      title='B Magnitude RAW', xtitle='T (Seconds)', ytitle='B Mag (Counts)', $
      xtickformat='tick_hms_whole', xticks=xticks
    oplot, t8, bmagn, psym=4
    oplot, t100, bmagnn
    
    if keyword_set(print) then begin
        end_of_page, /del, /print
    endif
    
endif

if keyword_set(print) then begin
    end_of_prog, /print, /delete
endif
return
end
