FUNCTION deficiency_string_web, cond, standard, testv, label
   IF (cond) THEN BEGIN 
      dstring = label+': '+strtrim(string(testv,format='(f6.2)'),2)+' -> '+ $
       strtrim(string(standard,format='(f6.2)'),2)+' '
   ENDIF ELSE BEGIN
      dstring = ""
   ENDELSE 

   return, dstring
END
 

PRO plot_version_web, plot_version, plot_number, plot_identifier, versions, $
                      quality_flag, deficiency=deficiency, $
                      init=init, echo=echo, verbose=verbose
;
;   Read plot version summary file.
;
;   INPUT:  plot_identifier = string indicating type of plot
;   INPUT:  versions = float array of versions to test
;
;           /verbose prints information about the input for debugging
;           /init rereads the plot_version file
;           /echo echos the lines of the plot_version file as it is read
;
;
;   OUTPUT: plot_version = absolute Plot Version
;   OUTPUT: plot_number = absolute HYDRA Plot Number
;   OUTPUT: quality_flag = .true. if it is up-to-date
;   
;   OUTPUT: deficiency returns a string stating what needs to be updated
;
;
;   ENVIRONMENT VARIABLES:
;        'wget_exe' =                     Path to wget executable
;        'HYDRA_PLOT_VERSION_ADDR' =      Address of directory with web-file
;        'HYDRA_PLOT_VERSION_PATH' =      Local path to file
;
;
;
   COMMON plot_version_web_comm, define, plot_vers, nversions, nplots, nparts


   IF keyword_set(define) THEN BEGIN  ;  Check for previous attempts
      IF (define EQ -1) AND (NOT keyword_set(init)) THEN BEGIN 
                                ;  Check for previous failure
         IF keyword_set(verbose) THEN $
          print, 'hydra plot version file (checked earlier) not available.'+$
          '  Aborting.'
         plot_version =  -1.0
         plot_number =  -1
         quality_flag = 0
         deficiency = 'NA'
         define = -1
         return
      ENDIF 
   ENDIF 
   IF NOT keyword_set(define) OR keyword_set(init) THEN BEGIN 

      filename = 'plot_version_summary_v00.dat'

      wget_exe = getenv('wget_exe')
      plot_version_addr = getenv('HYDRA_PLOT_VERSION_ADDR')


      read_local = 1
      IF (wget_exe NE '' AND plot_version_addr NE '') THEN BEGIN ; Attempt Web
         command = wget_exe+' -T 20 '+plot_version_addr+filename
         read_local = 0
         spawn, command+' -O - ', unit=unit

         error = 1
         ON_IOERROR, web_error
         line = string("")
         readf, unit, line
         error = 0

web_error:               
         IF (error EQ 1) THEN BEGIN   ; Web access Failed
            read_local = 1
            IF keyword_set(verbose) THEN BEGIN 
               print, 'Web Connection failed - Using local file if possible.'
               print, command
            ENDIF 
         ENDIF 
      ENDIF ELSE BEGIN
         IF keyword_set(verbose) THEN BEGIN
            IF (wget_exe EQ '') THEN print, 'env wget_exe not defined.'
            IF (plot_version_addr EQ '') THEN $
             print, 'env HYDRA_PLOT_VERSION_ADDR not defined.' 
            print, 'Attempting to read from local file.'
         ENDIF 
      ENDELSE 

      IF (read_local EQ 1) THEN BEGIN          ; Atttempt using local file
         plot_version_path = getenv('HYDRA_PLOT_VERSION_PATH')
         IF (plot_version_path NE '') THEN BEGIN  ; Read local file

            file = plot_version_path+filename
            r = findfile(file,count=nfile)
            IF (nfile EQ 0) THEN BEGIN   ;  File not found - Abort
               IF keyword_set(verbose) THEN BEGIN 
                  print, 'hydra plot version file not found.  Aborting.'
                  print, file
               ENDIF 
               plot_version =  -1.0
               plot_number =  -1
               quality_flag = 0
               deficiency = 'NA'
               define = -1
               return
            ENDIF 
            get_lun, unit
            openr, unit, file

            error = 1
            ON_IOERROR, file_error
            line = string("")
            readf, unit, line
            error = 0

file_error:               
            IF EOF(error EQ 1) THEN BEGIN   ;  File not found - Abort
               IF keyword_set(verbose) THEN BEGIN 
                  print, 'hydra plot version file empty.  Aborting.'
                  print, file
               ENDIF 
               plot_version =  -1.0
               plot_number =  -1
               quality_flag = 0
               deficiency = 'NA'
               define = -1
               return
            ENDIF 
         ENDIF ELSE BEGIN ; Abort - No reading possible
            IF keyword_set(verbose) THEN $
             print, 'env HYDRA_PLOT_VERSION_PATH not defined.  Aborting.'
            plot_version =  -1.0
            plot_number =  -1
            quality_flag = 0
            deficiency = 'NA'
            define = -1
            return
         ENDELSE 
      ENDIF 

      ON_IOERROR, NULL
      IF keyword_set(verbose) THEN print, 'Reading plot_version Record.'

      nversions = 100
      nplots = 100
      nparts = 6

      plot_vers1 = {plot_v1, $
                    plotv: float(0.0), $
                    nparts: 0L, $
                    parts: fltarr(nparts) }

      plot_vers2 = {plot_v2, $
                    nversions: 0L, $
                    identifier: string(""), $
                    versions: replicate(plot_vers1, nversions)}

      plot_vers = replicate(plot_vers2, nplots)

      iplot = -1

      IF keyword_set(echo) THEN print, line
      WHILE (strmid(line, 0, 3) NE 'END') DO BEGIN
         CASE strmid(line,0,4) OF
            'DDEI': BEGIN
               ident = (str_sep(strcompress(line),' '))(1)
               iplot = iplot+1
               iversion = -1
               plot_vers(iplot).nversions = iversion
               plot_vers(iplot).identifier = ident
            END
            'VERS': BEGIN
               iversion = iversion+1
               plot_vers(iplot).nversions = iversion
               npos = strpos(line, '#')
               IF (npos NE -1) THEN BEGIN
                  line0 = strtrim(strmid(line, 0, npos),2)
               ENDIF ELSE BEGIN
                  line0 = strtrim(line,2)
               ENDELSE 
               line1 = str_sep(strcompress(line0),' ')
               nums = float(line1(1:n_elements(line1)-1))
               plot_vers(iplot).versions(iversion).plotv = nums(0)
               plot_vers(iplot).versions(iversion).nparts = n_elements(nums)-1
               plot_vers(iplot).versions(iversion).parts = $
                nums(1:n_elements(nums)-1)
            END
            ELSE: BEGIN
            END
         ENDCASE 
         readf, unit, line
         IF keyword_set(echo) THEN print, line
      ENDWHILE 
      close, unit
      free_lun, unit

      define = 1
   ENDIF ELSE BEGIN
      IF keyword_set(verbose) THEN print, 'Not Reading plot_version Record.'
   ENDELSE


   nv = n_elements(versions)
   iver = lonarr(nv)
   FOR i=0,nv-1 DO BEGIN
      IF (versions(i) NE -1.0) THEN iver(i) = 0 ELSE iver(i) = 1
   ENDFOR 
   


   IF keyword_set(verbose) THEN BEGIN 
      print, 'INPUT Version Parameters: '
      FOR i=0,n_elements(versions)-1 DO BEGIN
         IF (versions(i) NE -1.0) THEN $
          print, i, versions(i) ELSE print, i, '*'
      ENDFOR 
   ENDIF 


   vers = long(versions*100+0.01)/100.0

   iplot = 0
   WHILE (plot_vers(iplot).identifier NE plot_identifier) AND $
    (iplot LT nplots-1) DO BEGIN
      iplot = iplot+1
   ENDWHILE 
   IF (plot_vers(iplot).identifier NE plot_identifier) THEN BEGIN
      print, 'HYDRA PLOT IDENTIFIER "'+plot_identifier+$
       '" NOT REGISTERED. ABORTING.'
      stop
   ENDIF 
   plot_number = iplot
   nv = plot_vers(iplot).nversions

   iversion = 0

   logical_v = 1
   FOR i=0,plot_vers(iplot).versions(iversion).nparts-1 DO BEGIN 
      logical_v = logical_v AND $
       ((plot_vers(iplot).versions(iversion).parts(i) EQ vers(i)) OR iver(i))
   ENDFOR
;   print, vers
;   print, plot_vers(iplot).versions(iversion).parts
   WHILE ((NOT logical_v) AND (iversion LT nv)) DO BEGIN
      iversion = iversion + 1

      logical_v = 1
      FOR i=0,plot_vers(iplot).versions(iversion).nparts-1 DO BEGIN 
         logical_v = logical_v AND $
          ((plot_vers(iplot).versions(iversion).parts(i) EQ vers(i)) $
           OR iver(i))
      ENDFOR
;      print, vers
;      print, plot_vers(iplot).versions(iversion).parts
      
   ENDWHILE 
   IF (logical_v) THEN BEGIN
      plot_version = plot_vers(iplot).versions(iversion).plotv
      IF (plot_version EQ plot_vers(iplot).versions(0).plotv) THEN BEGIN
         quality_flag = 1      
      ENDIF ELSE BEGIN
         quality_flag = 0
      ENDELSE 
   ENDIF ELSE BEGIN
      plot_version = 0.00
      quality_flag = 0
   ENDELSE 
   label_string = ["FILEV", "LIBV", "CODEV"]

   np = n_elements(vers)
   diff = fltarr(np)
   deficiency = ""
   FOR i=0, np-1 DO BEGIN 
      diff(i) = plot_vers(iplot).versions(0).parts(i) - vers(i)
;      print, plot_vers(iplot).versions(iversion).parts(i), vers(i)
      deficiency = deficiency + $
       deficiency_string_web((diff(i) NE 0) AND (NOT iver(i)), $
                             plot_vers(iplot).versions(0).parts(i), $
                             vers(i), $
                             Label_string(i))
   ENDFOR 
 
   return
END
