pro average_smear, v_in, timetags_in, timetags_id, $
                   nx=nx, ny=ny, xrange=xrange, zrange=zrange, $
                   coords=coords, midpoint=midpoint_in, $
                   ion=ion, version=version, $
                   matrix=matrix, save=save, interval=interval, $
                   test=test, recalc=recalc, trange=trange, $
                   prefix=prefix, avg_method=avg_method, $
                   mfe_8hz=mfe_8hz, angle_filter=angle_filter
;
;    Return the x,y,z coords in 'coords'
;    
;    Inputs:
;              v:           Quantity to be plotted.
;              timetags:    hyd_access timetags.
;              timetags_id: hyd_access timetags identifier from
;                             hd_table. (interval can be used instead)
;
;
;    Keyword Inputs:
;              nx, ny:         Size for averaging matrix
;              xrange, zrange: Limits of matrix.
;              trange:         Limits of timetags - for universal smear
;              /midpoint:      Use midpoint instead of full smear.
;              /ion:           Look for 'ion' instead of 'ele' 1.15s smears.
;              /save:          Use idlsav files if available.
;              interval:       Repetition rate of timetags.
;              prefix:         Path to prefix to smear files.
;              avg_method:     Method of averaging to use:
;                                  'A' = weighted Average (default)
;                                  'B' = Max
;                                  'C' = Min
;                                  'D' = |Max|
;              /mfe_8hz:       Uses 54Hz interpolated MFE 8Hz data.
;              /angle_filter:  Uses 54Hz w/filtered ppa angles.
;
;    Keyword Outputs:
;              coords:    Return the (x,z,v) triple from the averaged matrix. 
;              matrix:    Return the averaged matrix - does not work!
;              recalc:    Do not read the saved smear.
;              version:   Return version of average_smear.
;              footprint: Append the footprint for the smear.
;
;    N.B. Do NOT send fill points!
;
version = hydra_rcs_version_number('$Name: Smear_Plot_2_8 $', /num) ; VERSION NUMBER OF SMEAR
midpoint=keyword_set(midpoint_in)
smear_dots=1
rfill=-1e31
generator='jds'                 ; 'jds' or 'xxc'
v=v_in
timetags=timetags_in
create=0

if not keyword_set(avg_method) then avg_method='A'
 
fill_holes=1
treat_54Hz_differently=1
if size(v,/n_dimension) ne 1 then $
  message, 'ERROR: Input quantity dimensions incorrect - check your input.'
if (size(timetags, /dimension))(0) ne (size(v, /dimension))(0) then $
  message, 'ERROR: Input dimensions for quantity to not match timetags.'
;
;    Save Existing-footprint
;
hd_add_footprint, 'Average_Smear: version '+strtrim(version,2)
footprint_save=hd_get_footprints()
;
;    Construct time-intervals
;
;
;  Time interval repetition
;
if not keyword_set(interval) then begin
    hd_table, table, /get_table
    r=where(timetags_id eq table(*).id)
    if r(0) ne -1 then interval=table(r).interval else interval=rfill
endif
if interval eq rfill then begin
    hyd_data, timetags, 'time_bin_start_ssm', tstart
    dtstart=tstart(1:*)-tstart
    res=moment(dtstart, /double, sdev=sdev)
    r=where(abs(dtstart-res(0)) lt sdev)
    if r(0) ne -1 then begin
        res=moment(dtstart(r), /double, sdev=sdev)
        if sdev lt 0.1*res(0) then begin
            message, 'Interval being set from timetag differencs.', /cont
            nmax=(trange(1)-trange(0))/res(0)
            if nmax/float(n_elements(v_in)) lt 1.25 then interval=res(0)
        endif
    endif
endif
;
;   Check for 54Hz
;
if (treat_54Hz_differently and abs(interval) lt 0.019)  then begin
    hz54=1
    fill_holes=0
;    midpoint=1
endif else begin
;    hz54=0
    if keyword_set(midpoint) then hz54=1 else hz54=0
endelse
;
;   Check for holes in the data and fill them.
;
if keyword_set(fill_holes) then begin
    if not keyword_set(trange) then trange=0
    v=hyd_fill_out_points(timetags, v, interval, trange=trange)
    timetags=hyd_fill_out_points(timetags, timetags, interval, trange=trange)
endif
hyd_data, timetags, 'time_bin_start_ssm', tstart
hyd_data, timetags, 'time_bin_width_seconds', twidth
tend=tstart+twidth
if n_elements(v) ne n_elements(tstart) then $
  message, 'ERROR: Timetag length does not match input quantity - '+$
  'check your input.'
;
;   Check for idlsav based on timetags.
;
dt=interval
if dt lt 0.019 and dt gt 0.017 then $
  resolution=1 $
else if dt lt 0.026 and dt gt 0.024 then $
  resolution=10 $
else if dt lt 0.122 and dt gt 0.119 then $
  resolution=9 $
else if dt lt 1.2 and dt gt 1.1 then $
  resolution=2 $
else if dt lt 2.4 and dt gt 2.2 then $
  resolution=4 $
else if dt lt 3.5 and dt gt 3.4 then $
  resolution=5 $
else if dt lt 4.7 and dt gt 4.5 then $
  resolution=6 $
else if dt lt 5.8 and dt gt 5.7 then $
  resolution=7 $
else if dt lt 7.0 and dt gt 6.8 then $
  resolution=8 $
else if dt lt 13.9 and dt gt 13.7 then $
  resolution=3 $
else if dt eq rfill then $
  resolution=0 $
else $
  resolution=-1

case resolution of
    0: begin
        res='_variable'
    end
    1: begin
        res='_54Hz'
    end
    2: begin
        res='_1.15s'
    end
    3: begin
        res='_13.8s'
    end
    4: begin
        res='_2.3s'
    end
    5: begin
        res='_3.45s'
    end
    6: begin
        res='_4.6s'
    end
    7: begin
        res='_5.75s'
    end
    8: begin
        res='_6.9s'
    end
    9: begin
        res='_8Hz'
    end
    10: begin
        res='_40Hz'
    end
    -1: begin
        message, 'Unrecognised timetags.'
    end
endcase
if keyword_set(midpoint) and (resolution ne 1) then res=res+'_mid'
if keyword_set(mfe_8hz) then res=res+'_8hz'
if keyword_set(filter_angles) then res=res+'_AF'
;
;   Check for file and read if available.
;
if not keyword_set(prefix) then prefix=''
filename0=prefix+'smear'+res
if resolution eq 2 then begin
    if keyword_set(ion) then filename=filename0+'_ion' $
    else filename=filename0+'_ele' 
endif else begin
    filename=filename0
endelse
f=hd_get_footprints()           ; Delete footprint
if keyword_set(save) and not keyword_set(recalc) then begin
    file=findfile(filename+'.idlsav',count=c)
    if (c eq 1) then begin
        smear_version=-1
        restore, filename+'.idlsav', /verbose
        if smear_version eq version then begin
            check_smear, smear_index, smear_times, smear_coords, v, $
              tstart, tend, check=check, /fix, interval=interval, $
              xrange=xrange, zrange=zrange, nx=nx, ny=ny
            if check eq 0 then print, 'IDLSAV does not match, recalculating.'
        endif else begin
            message, 'IDLSAV Smear Version does not match: recalculating.', $
              /continue
            check=0
        endelse
    endif else begin
        message, 'IDLSAV does not exist: '+filename, /cont
        check=0
    endelse
endif else begin
    check=0
endelse
if check eq 0 then begin
    smear_coords=0
    If not keyword_set(nx) then nx=1000
    if not keyword_set(ny) then ny=1000

    if not keyword_set(xrange) then xrange=[-2,2]
    if not keyword_set(zrange) then zrange=[-3,3]

    if generator eq 'xxc' then begin
;
;    Get NIF coordinates at tstart tend
;
        hyd_get_nif_coords, timetags, xnif_start, xnif_end, $
          znif_start, znif_end

        r=where(xnif_start ne rfill and znif_start ne rfill and $
                xnif_end ne rfill and znif_end ne rfill)
        if r(0) eq -1 then message, 'No nif_traj found for input timetags.'
;
; Filter timetags, quantity, nif
;
        tstart=tstart(r)
        tend=tend(r)
        v=v(r)
        xnif_start=xnif_start(r)
        xnif_end=xnif_end(r)
        znif_start=znif_start(r)
        znif_end=znif_end(r)
;
;   The jds separator file is already relative to spacecraft
;
        if generator eq 'jds' then begin
            print, 'Modifying s/c nif position to zero.'
            xnif_start(*)=0.0
            xnif_end(*)=0.0
            znif_start(*)=0.0
            znif_end(*)=0.0
        endif
    endif else begin
        xnif_start=replicate(0.0,n_elements(tstart))
        xnif_end=replicate(0.0,n_elements(tstart))
        znif_start=replicate(0.0,n_elements(tstart))
        znif_end=replicate(0.0,n_elements(tstart))
    endelse
;
;    Get Separator Coordinates
;
    hyd_get_separator_coords, txs, xs, tzs, zs, generator=generator, $
      mfe_8hz=mfe_8hz
    if generator eq 'jds' then begin
        print, 'Modifying sign of xs, zs.'
        xs=-xs
        zs=-zs
    endif
    

    if hz54 or interval eq rfill then begin

        if keyword_set(smear_dots) then begin
            plot_separator_dots_54, tstart, tend, $
              xs, txs, zs, tzs, matrix_dimensions=[nx,ny], $
              xrange=xrange, yrange=zrange, $
              smear_coords=smear_coords, $
              smear_index=smear_index, smear_times=smear_times
        endif else begin
            plot_separator_54hz, tstart, tend, $
              xnif_start, xnif_end, znif_start, znif_end, $
              xs, txs, zs, tzs, matrix_dimensions=[nx,ny], $
              xrange=xrange, yrange=zrange, $
              smear_coords=smear_coords, $
              smear_index=smear_index, smear_times=smear_times
        endelse

    endif else begin
        if keyword_set(smear_dots) then begin
            plot_separator_dots, tstart, tend, xs, txs, zs, tzs, $
              dimensions=[nx,ny], xrange=xrange, yrange=zrange, $
              smear_coords=smear_coords, $
              smear_index=smear_index, smear_times=smear_times, $
              interval=interval
        endif else begin
            plot_separator, v, tstart, tend, $
              xnif_start, xnif_end, znif_start, znif_end, $
              xs, txs, zs, tzs, dimensions=[nx,ny], $
              xrange=xrange, yrange=zrange, $
              /use_smear_input, smear_coords=smear_coords, $
              smear_index=smear_index, smear_times=smear_times, $
              /accumulate_smears, midpoint=midpoint
        endelse
    endelse
    create=1

    footprint_s=hd_get_footprints()
endif else footprint_s=footprint_smear

if keyword_set(save) and keyword_set(create) then begin
    file=findfile(filename+'.idlsav',count=c)
    if (c ne 0) then $
      message, 'IDLSAV already exists, overwriting: '+filename+'.', /cont

    
    footprint_smear=['Restore: ', footprint_s, 'End Restore']
    smear_version=version
    save, smear_coords, smear_times, smear_index, footprint_smear, $
      smear_version, filename=filename+'.idlsav', /verbose 
endif

if not keyword_set(coords) then coords=1
plot_separator, v, tstart, tend, $
  /use_smear_input, smear_coords=smear_coords, smear_index=smear_index, $
  smear_times=smear_times, return_points=coords, matrix=matrix, $
  /plot, midpoint=midpoint, interval=interval, avg_method=avg_method

hd_add_footprint, footprint_save
hd_add_footprint, footprint_s
return
end

