pro get_smear_mapping, resolution, date, iblock, isub, iblock_range, $
                       timetag=timetag, all_timetags=all_timetags, $
                       data_range, data_dim, dir=dir, label=label, $
                       ixg, iyg, weight, fraction, $
                       reload=reload, verbose=verbose, save_dir=save_dir
;
;   Read the complete smear mapping for 4.6s data and return indexes
;   to specified array for each sub-block.
;
;   Input:
;      resolution            - '4.6s'
;      date                  - defined for 19960529
;      iblock                - Block Index to return
;      isub                  - Sub-block index to return
;      iblock_range          - [Block Index Start, End]
;      data_range            - [xmin, ymin, xmax, ymax]
;      data_dim              - [nx, ny]
;
;
;   Input Keyword:
;      dir                   - Sets the directory to find the smear file.
;      label                 - Additional prefix to file.
;      timetag               - Timetag for the interval in question.
;      all_timetags          - Timetags for all the intervals in question.
;      reload                - causes the file to be reread.
;      save_dir              - cache result as idlsav file in this directory.
;      verbose               - prints extra messages about failures.
;
;
;   Output:
;      ixg, iyg, weight      - Separator Position Indexes and weights
;                              associated with the given timetags.
;      fraction              - Fraction of total Time inside matrix
;
;
;   A Block that is not in the smear mapping returns as ixg=iyg=weight=-1
;
;
;   Subsidiary Routines:
;      read_smear_mapping, read_smear_mapping_indiv.
;
;
;   Example:
;
;    get_smear_mapping, '4.6s', 19960529, 1800, 0, $
;      [-2,-4,2,4], [400,800], $
;      dir='/home/ddm/bigdisk2/plot_devel/release/savsmr/0630_0730/', $
;      label='e_restr', smear_indices, /verbose, /save
;
;
   if not keyword_set(save_dir) then $
     save_dir='/opt/hydra/tmp_data/other_data_sets/' 
   if keyword_set(timetag) then begin
       filename=save_dir+'smear_indices_'+strtrim(date,2)+'_'+$
         label+'_'+resolution+'_timetags.idlsav'
   endif else begin
       filename=save_dir+'smear_indices_'+strtrim(date,2)+'_'+$
         label+'_'+resolution+'_'+$
         string(iblock_range(0),format='(i4.4)')+'_'+$
         string(iblock_range(1),format='(i4.4)')+'.idlsav'
   endelse

   if not keyword_set(reload) then begin
       f=findfile(filename,count=nf)
       if nf eq 1 then begin
           restore, f(0)
           if (where(float(data_range) ne $
                     smear_indices.data_range))(0) ne -1 or $
             (where(data_dim ne smear_indices.data_dim))(0) ne -1 then begin
               reload=1
           endif
       endif
   endif

   if not keyword_set(timetag) then begin
       if keyword_set(reload) then begin
           read_smear_mapping, resolution, date, iblock_range, $
             data_range, data_dim, dir=dir, label=label, smear_indices, $
             reload=reload, verbose=verbose, save_dir=save_dir
       endif
       r=where(iblock eq smear_indices.iblock and $
               isub eq smear_indices.isub,nr)
       if nr eq 0 then begin
           if keyword_set(verbose) then $
             print, 'Sub-Block not found in idlsav.'
           ixg=-1
           iyg=-1
           weight=-1
           fraction=0.0
       endif else begin
           nb=n_elements(smear_indices.iblock(*,0))
           rb=r(0) mod nb
           rs=r(0)/nb
           
           rr=where(smear_indices.weight(*,rb,rs) gt 0.0, nrr)
           if nrr eq 0 then begin
               if keyword_set(verbose) then $
                 print, 'No valid points found for sub-block.'
               ixg=-1
               iyg=-1
               weight=-1
               fraction=0.0
           endif else begin
               ixg=smear_indices.ixg(rr,rb,rs)
               iyg=smear_indices.iyg(rr,rb,rs)
               weight=smear_indices.weight(rr,rb,rs)
               fraction=smear_indices.fraction(rb,rs)
           endelse
       endelse
   endif else begin
       if not keyword_set(reload) then begin
           if size(smear_indices, /type) eq 8 then begin
               for i=0,n_elements(all_timetags)-1 do begin
                   r=where(all_timetags(i).bin_start eq $
                           smear_indices.timetags.bin_start, nr)
                   if nr eq 0 then reload=1
               endfor
           endif else begin
               reload=1
           endelse
       endif
       if keyword_set(reload) then begin
           read_smear_mapping, resolution, date, 0, $
             timetags=all_timetags, $
             data_range, data_dim, dir=dir, label=label, smear_indices, $
             reload=reload, verbose=verbose, save_dir=save_dir
       endif
       r=where(timetag.bin_start eq smear_indices.timetags.bin_start, nr)
       if nr eq 0 then begin
           if keyword_set(verbose) then $
             print, 'Timetag not found in idlsav.'
           ixg=-1
           iyg=-1
           weight=-1
           fraction=0.0
       endif else begin
           rt=r(0)
           
           rr=where(smear_indices.weight(*,rt) gt 0.0, nrr)
           if nrr eq 0 then begin
               if keyword_set(verbose) then $
                 print, 'No valid points found for timetag.'
               ixg=-1
               iyg=-1
               weight=-1
               fraction=0.0
           endif else begin
               ixg=smear_indices.ixg(rr,rt)
               iyg=smear_indices.iyg(rr,rt)
               weight=smear_indices.weight(rr,rt)
               fraction=smear_indices.fraction(rt)
           endelse
       endelse
   endelse

end
