function hd_synchronize_nearest_neighbor_rdh, a_timetags, $
                                          b_timetags_in, b_data_in
;+
; NAME: hd_synchronize_nearest_neighbor
;
; PURPOSE: synchronize data set to new timetags by selecting nearest neighbors.
;   Require that nearest neighbor picked from b_timetags and 
;   a_timetag must "overlap"--either A centeriod must be in B bin
;   or B centriod must be in A bin.
;   This is the default synchronization method used by hyd_access to 
;   synchronize extracted data with requested timetags.
;
; CATEGORY: hyd_access
;
; CALLING SEQUENCE: 
;   synchronized_data= hd_synchronize_nearest_neighbor( new_timetags, $
;                                                       original_timetags,$
;                                                       original_data )
; 
; INPUTS: 
;   new_timetags,        array of hyd_access timetag structures.
;                          MUST BE MONOTONICALLY INCREASING.
;   original_timetags    hyd_access timetags.
;   original_data        array of scalars (eg. fltarr(n)) or 
;                          vectors (fltarr(n,3))
;
; KEYWORD PARAMETERS: none.
;
; OUTPUTS: synchronized data set.
;
; COMMON BLOCKS: presently uses hyd_findex, which maintains one sort
;                  so that repeated inputs are not recalculated.
;
; RESTRICTIONS: new_timetags must be monotonically increasing.
;
; PROCEDURE: use hyd_findex to find nearest neighbors, then verify that
;    found neighbors' timetags overlap the original timetags.
;
; EXAMPLE: 
;  hyd_access, tt_mid1, 'ele_n', data_mid1, get_timetags= time_range
;  hyd_access, tt_mid2, 'efi_vz', data_mid2, get_timetags= time_range
;  !p.multi=[0,1,2]
;  hd_plot, tt_mid1, data_mid1, xrange= time_range, title='Raw Data'
;  hd_plot, tt_mid1, data_mid1, xrange= time_range, title='Nearest Neighbor'
;  xx= hd_synchronize_nearest_neighbor( tt_mid2, tt_mid1, data_mid1 )
;  hd_plot, /overplot, tt_mid2, xx, xrange=!x.crange,  $
;    color=get_color_index('yellow')
;
; MODIFICATION HISTORY: 
;  Written, Summer 1999, Jeremy Faden
;-
  
@hd_include
  on_error, 2

  if n_elements( a_timetags ) eq 0 then begin
      message, 'Target timetags undefined'
  endif
  if n_elements( b_timetags_in ) eq 0 then begin
      message, 'Source timetags undefined'
  endif

;   Synchronize data sets by picking nearest neighbors.  

   a_time= a_timetags.bin_start + a_timetags.bin_width / 2

   b_timetags= b_timetags_in
   b_data= b_data_in
   
;   detect holes in data set and insert fill data.

   b_time= b_timetags.bin_start + b_timetags.bin_width / 2

   s= size( b_data )
   type= s(n_elements(s)-2)

   n= n_elements( a_timetags )
   m= n_elements( b_data(0,*) )
   l= n_elements( b_data(0,0,*) )
   result= make_array( n, m, l, type=type, value=hd_fill )

   if n_elements( b_time ) eq 1 then begin
       ii= intarr( n_elements( a_time ) )
   endif else begin
       ii= hyd_findex( b_time, a_time )
   endelse
   
   r=where(b_timetags.bin_width ne b_timetags(0).bin_width,nr)
   if nr eq 0 then begin
       index= long(ii+0.5)
   endif else begin
       w0=b_timetags.bin_width
       w1=b_timetags(1:*).bin_width
       f1=w0/(w0+w1)
       f2=1-f1

       index= long(ii+0.5)
       r=where(index ge 0)
       index(r)=long(ii(r)+f2(ii(r)))
                                ;
                                ;  N.B. Left and Right B.C. not done.
                                ;
   endelse



;   index= long(ii+0.5)
   result= b_data( index, *, * )

   btt= (temporary(b_timetags))(index)

;   check that nearest neighbor timetags overlap original.
   rnv= where(( a_time lt btt.bin_start or $
                a_time gt btt.bin_start + btt.bin_width ) and $
              ( b_time(index) lt a_timetags.bin_start or $
                b_time(index) gt a_timetags.bin_start + a_timetags.bin_width ))
   if rnv(0) ne -1 then result(rnv,*,*)= hd_fill
   
   rnv= where( result(*,0) eq hd_fill )
   if rnv(0) ne -1 then a_timetags(rnv).valid_register=0
   
   return, result
   
end
