FUNCTION hyd_day_of_year, year, month, day, date=date, set_year=set_year, $
                      yyyymmdd=yyyymmdd, verbose=verbose, stop=stop
;+
; NAME:
;      hyd_day_of_year
;
;
; PURPOSE:
;      Calculate the day of the year given the date as a string or the
;               year, month, day arguments.
;
; CATEGORY:
;      Utility
;
;
; CALLING SEQUENCE:
;      doy = HYD_DAY_OF_YEAR, year, month, day
;      doy = HYD_DAY_OF_YEAR, date=date
; 
; INPUTS:
;      None.
;
; OPTIONAL INPUTS:
;     year, month, day
;	
; KEYWORD PARAMETERS:
;     DATE: DATE='05/29/96' - can be used instead of year, month, day
;     YYYYMMDD: YYYYMMDD=19960529
;
;     SET_YEAR:  Sets day 1 as Jan 1 of a particular year.
;
;     Debugging:
;       /verbose: prints debugging information.
;       /stop: stops before exit for variable examination.
;
; OUTPUTS:
;     Returns the Day of Year.
;
; OPTIONAL OUTPUTS:
;     None.
;
; COMMON BLOCKS:
;     None.
;
; SIDE EFFECTS:
;     None.
;
; RESTRICTIONS:
;     None.
;
; PROCEDURE:
;     Counts days from the start of the year, month by month.
;
; EXAMPLE:
;     print, hyd_day_of_year(1996, 5, 29)
;     print, hyd_day_of_year(date='05/29/96', set_year=1995)
;
; MODIFICATION HISTORY:
;  $Log: hyd_day_of_year.pro,v $
;  Revision 1.2  1999/06/09 15:55:59  rdh
;  Added documentation.
;
;  Revision 1.4  1998/11/16 22:06:03  rdh
;  Added doc_library comment header.
;
;  Revision 1.3  1998/03/11 17:20:25  rdh
;  Added keyword yyyymmdd for input.
;
;  Revision 1.2  1998/03/10 18:06:01  rdh
;  Added keyword set_year to allow easily going across year boundaries.
;
;  Revision 1.1  1998/03/10 17:46:08  rdh
;  Initial revision
;
;-
;
n = n_params()

IF (n EQ 0) THEN BEGIN
   IF keyword_set(date) THEN BEGIN 
      year = fix(strmid(date,6,2))
      month = fix(strmid(date,0,2))
      day = fix(strmid(date,3,2))
   ENDIF ELSE BEGIN
      IF keyword_set(yyyymmdd) THEN BEGIN
         year = fix(strmid(strtrim(yyyymmdd,2),0,4))
         month = fix(strmid(strtrim(yyyymmdd,2),4,2))
         day = fix(strmid(strtrim(yyyymmdd,2),6,2))
      ENDIF ELSE BEGIN 
         print, "USAGE: function hyd_day_of_year(year,month,day "+$
          "[,/stop][,/verbose][,set_year=yyyy])"
         print, "           OR   hyd_day_of_year(date='mm/dd/yy' "+$
          "[,/stop][,/verbose][,set_year=yyyy])"
         print, "           OR   hyd_day_of_year(yyyymmdd=yyyymmdd "+$
          "[,/stop][,/verbose][,set_year=yyyy])"
         stop
      ENDELSE 
   ENDELSE
ENDIF ELSE BEGIN
   IF (n NE 3) THEN BEGIN 
      print, 'ERROR in hyd_day_of_year: Need 3 arguments.'
      stop
   ENDIF
ENDELSE
IF (year LT 90) THEN BEGIN
   year = year+2000
ENDIF ELSE BEGIN
   IF (year LT 100) THEN BEGIN
      year = year + 1900
   ENDIF
ENDELSE

month_length = [31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31]

dday = day
IF (month GT 2) THEN BEGIN
   IF (year MOD 4 EQ 0) THEN BEGIN
      IF (year MOD 100 EQ 0) THEN BEGIN
         IF (year MOD 400 EQ 0) THEN BEGIN
            dday = dday + 1
         ENDIF
      ENDIF ELSE BEGIN
         dday = dday + 1
      ENDELSE
   ENDIF
ENDIF


FOR i=1,month-1 DO BEGIN
   dday = dday + month_length(i-1)
   IF keyword_set(verbose) THEN print, i, year, month, day, dday
ENDFOR

IF keyword_set(set_year) THEN BEGIN
   IF (set_year GT 1990) THEN BEGIN 
      year1 = set_year
   ENDIF ELSE BEGIN
      IF (set_year GT 90) THEN year1 = 1900+set_year $
      ELSE year1 = 2000+set_year
   ENDELSE

   FOR i=year1,year-1 DO BEGIN
      dday = dday + hyd_day_of_year(i,12,31)
   ENDFOR

ENDIF

IF keyword_set(stop) THEN stop

return, dday
END
