function hyd_poisson_slope, c1, c2, error=error, status=status, $
                            err_hdl=err_hdl, $
                            sdispersed_fac=sdispersed_fac, $
                            surface=surface, $
                            redchisq=redchisq, sfunction=sfunction, $
                            straw_slope=straw_slope, $
                            stop=stop
;+
;
;   Function Hyd_Poisson_Slope
;
;   Arguments: c1, c2: Counting Rates for successive measurements
;
;   Returns: The Best-Fit slope c2 = m * c1
;
;   Keywords:
;
;       error: Output contains the error on the slope.
;
;       status:  0=Okay
;               -1=Error Total over c1 or c2 is Zero!
;                1=Warning, Slope very small, error not well defined.
;               -2=Straw slope used.  Slope output not defined!
;
;
;       err_hdl: Translated status
;                Status     err_hdl
;                  0           1
;                 -1          -1
;                  1           5
;                 -2          -2
;       sdispersed_fac: Fraction of "s" gt 3.
;
;       sfunction:      "signed" "s" 
;
;       straw_slope:    Slope used for calculating "s"
;                    ###ONLY sfunction is VALID###
;
;
;-
status=0
tc1=total(c1, /double)
tc2=total(c2, /double)
if tc1 ne 0 and tc2 ne 0 then begin
    mp=tc2/tc1
    error=sqrt(mp*(mp+1)/tc1)
    if mp lt 1e-6 then begin
        status=1
    endif
endif else begin
    status=-1
    err_hdl=-1
    mp=-1e31
    error=-1e31
    redchisq=-1e31
    return, mp
endelse
if keyword_set(straw_slope) then begin
    status=-2
    mp=straw_slope
endif


if keyword_set(surface) then begin
    set_page, nv=2
endif
;
;Calculate s^2 = 2(lnPmax(m_min)-lnP(m)) Line
;
done=0
nm=100
dm=error/float(nm/4) > 0.001
nc=n_elements(c1)
while done eq 0 do begin

    m=(findgen(nm)-nm/2)*dm+mp
    lnp=make_array(size=size(m),value=-1e31)
    r=where(c1+c2 gt 0, nr)
    lnp_const=0.0
    lnp_max_const=0.0
    lnp_consts=fltarr(nc)
    if nr gt 0 then begin
        cc1=c1[r]
        cc2=c2[r]
        lnp_const=$
          total((cc1+cc2)*(alog(cc1+cc2)-1))-$
          total(hyd_factorial(cc1,/ln)+$
                hyd_factorial(cc2,/ln),/double)
        lnp_consts[r]=(cc1+cc2)*(alog(cc1+cc2)-1)-$
          hyd_factorial(cc1,/ln)-hyd_factorial(cc2,/ln)

                                ;   
        mu1s=(cc1+cc2)/(mp+1)
        mu2s=mu1s*mp
        cm1=float(long(mu1s))
        cm2=float(long(mu2s))
        lnp_max_const=$
          total((cm1+cm2)*alog(cc1+cc2)-(cc1+cc2),/double)-$
          total(hyd_factorial(cm1,/ln)+$
                hyd_factorial(cm2,/ln),/double)
                                ;
    endif
    r=where(c1+c2 eq 0, nr)
    if nr gt 0 then begin
        cc1=c1[r]
        cc2=c2[r]
        lnp_const=lnp_const-$
          total(hyd_factorial(cc1,/ln)+$
                hyd_factorial(cc2,/ln),/double)
        lnp_consts[r]=-hyd_factorial(cc1,/ln)-hyd_factorial(cc2,/ln)
                                ;   
        mu1s=(cc1+cc2)/(mp+1)
        mu2s=mu1s*mp
        cm1=float(long(mu1s))
        cm2=float(long(mu2s))
        lnp_max_const=lnp_max_const-$
          total(hyd_factorial(cm1,/ln)+$
                hyd_factorial(cm2,/ln),/double)
                                ;
    endif



    for im=0,nm-1 do begin
        lnp[im]=-total(c1+c2,/double)*alog(m[im]+1)+$
          total(c2,/double)*alog(m[im])
    endfor
    lnps=-(c1+c2)*alog(mp+1)+c2*alog(mp)
    lnps=lnps+lnp_consts
                                ;
    mu1s=(c1+c2)/(mp+1)
    mu2s=mu1s*mp
    cc1=float(long(mu1s))
    cc2=float(long(mu2s))
    lnp_maxs=-(cc1+cc2)*alog(mp+1)+cc2*alog(mp)
    lnp_max=-total(cc1+cc2,/double)*alog(mp+1)+$
          total(cc2,/double)*alog(mp)
    r=where(cc1+cc2 eq 0,nr)
    lnp_maxs_const=(cc1+cc2)*(alog(cc1+cc2)-1)-hyd_factorial(cc1,/ln)-$
                hyd_factorial(cc2,/ln)
    if nr gt 0 then lnp_maxs_const[r]=0.0
                                ;
    lnp_maxs=lnp_maxs+lnp_maxs_const
    lnp=lnp+lnp_const
;
;   Determine the Max Probability at each pair.
;
;    mu1s=(c1+c2)/(mp+1)
;    mu2s=mu1s*mp
;    lnpmaxs=poisson_max(mu1s,/ln)+poisson_max(mu2s,/ln)
;    lnpmax=total(lnpmaxs,/double)
;
;   s^2
;
;    s_sq0=2*(lnpmax-lnp)
    s_sq=2*(lnp_max+lnp_max_const-lnp)
    r=where(m lt 0, nr)
    if nr gt 0 then s_sq[r]=s_sq[r[nr-1]+1]
;    if (where(finite(s_sq) eq 0))[0] ne -1 then stop
    s_sq_min=min(s_sq,ip)
    r0=where(s_sq ge s_sq_min+1 and s_sq[1:*] lt s_sq_min+1 and m gt 0.0, nr0)
    if m[0] lt 0.0 and nr0 eq 0 then begin
        r0=where(m lt 0.0 and m[1:*] ge 0.0, nr0)
    endif
    r1=where((s_sq lt s_sq_min+1) and (s_sq[1:*] ge s_sq_min+1) and $
             (m gt 0.0), nr1)

    if nr0 eq 0 or nr1 eq 0 then dm=dm*2 else done=1
    if dm gt (mp > 0.1) then stop
endwhile

error0=m[ip]-m[r0[0]]
error1=m[r1[0]]-m[ip]

errors=[error0,error1]
    
if keyword_set(surface) then begin
    plot, m, s_sq, psym=-1, symsize=0.5, ystyle=2, $
      title='ln(P/P!dmax!n) - Joint Poisson Probability', $
      xtitle='Slope', ytitle='ln(P/P!dmax!n)', /ynoz
    oplot, [m[ip]],[s_sq[ip]], psym=2
    oplot, (mp)*[1,1], !y.crange, linestyle=2
    
    if nr0 gt 0 and nr1 gt 0 then begin
        print, mp, m[r0[0]], m[r1[0]]
    endif else begin
        message, 'No Error term determined.'
    endelse

    min=min(abs(m-(mp+error)),im0)
    oplot, [m[im0]],[s_sq[im0]], psym=2
    oplot, (mp-error)*[1,1], !y.crange, linestyle=1

    min=min(abs(m-(mp-error)),im1)
    oplot, [m[im1]],[s_sq[im1]], psym=2
    oplot, (mp+error)*[1,1], !y.crange, linestyle=1

    oplot, [m[r0[0]]],[s_sq[r0[0]]], psym=1, symsize=2
    oplot, [m[r1[0]]],[s_sq[r1[0]]], psym=1, symsize=2
endif
;
;   "Reduced s^2"
;
s_sq_norm=s_sq/(nc-2)
s_sq_norm_min=min(s_sq_norm,ip)
redchisq=s_sq_norm_min

if keyword_set(surface) then begin
    plot, m, s_sq_norm, psym=-1, symsize=0.5, $
      title='ln(P/P!dmax!n) - "Reduced" Joint Poisson Probability', $
      xtitle='Slope', ytitle='ln(P/P!dmax!n)/(N-2)', /ynoz
    oplot, [m[ip]],[s_sq_norm[ip]], psym=2
    
    min=min(abs(m-(mp+error)),im0)
    oplot, [m[im0]],[s_sq_norm[im0]], psym=2

    min=min(abs(m-(mp-error)),im1)
    oplot, [m[im1]],[s_sq_norm[im1]], psym=2


    print, error, mp-m[r0[0]], m[r1[0]]-mp
endif
;
;   Translate status to err_hdl
;
case status of
    0:  err_hdl=1
    1:  err_hdl=5
    -1: err_hdl=-1
    -2: err_hdl=-2
endcase
;
;   Calculate sdispersed_fac
;
if status eq 0 then begin
    sfunction=2.0*(lnp_maxs-lnps)
    r0=where(sfunction gt 9, nr0)
    sdispersed_fac=nr0/float(n_elements(c1))
    r=where(c2/float(c1) lt mp, nr)
    if nr gt 0 then sfunction[r]=-sfunction[r]
endif else begin
    sdispersed_fac=-1e31
    sfunction=-1e31
endelse
error=errors
if keyword_set(stop) then stop
return, mp
end
