;+
; NAME:
;      HYDRA_ID
;
; PURPOSE:
;      Returns a unique HYDRA_ID string to attach to a plot
;      for later identification purposes. -1 indicates an error occured.
;      -2 indicates the HYDRA_PLOT_ID environment variable is not defined.
;      -3 indicates not enough diskspace for hydraid.
;
;      On the sun 'S' is appended to the end of the string.
;      On the pc 'W' is appended.
;
; CATEGORY:
;      Utility.
;
; CALLING SEQUENCE:
;      ident = HYDRA_ID, identification_text_string_array
; 
; INPUTS:
;      input_string_array: A character string array to be written to the
;                          IDNUMBER.log file.
;
; OPTIONAL INPUTS:
;      None.
;	
; KEYWORD PARAMETERS:
;      /draft: Draft mode for debugging codes.  Does not actually
;                save to log that is kept beyond 1 week.
;
;
; OUTPUTS:
;      HYDRA_ID plot number
;
; OPTIONAL OUTPUTS:
;      None.
;
; COMMON BLOCKS:
;      hydra_id_common contains the previous input_string_array to prevent
;      the simplest kind of multiple redundant entries.
;
; SIDE EFFECTS:
;      Causes a lockfile to be written to in 
;         /opt/hydra/dstat/plot_idnumbers and and appends to 
;         IDNUMBER.log and increments IDNUMBER.
;
; RESTRICTIONS:
;      None.
;
; PROCEDURE:
;      The routine gets the next unique HYDRA_ID_NUMBER and returns it.
;      It writes a string to the log file.
;
; EXAMPLE:
;      num = HYDRA_ID('My_plot_program; Data: DDCAL 3.45, DD 0.20')
;
;-
;
FUNCTION hydra_id, input_string_array, $
                   stop=stop, $
                   draft=draft
common hydra_id2_common, input_string_array_save, id_save, csum_save, $
  csum_vector_save, save_flag, draft_log
;
;..Set lockfile and id directory
;
dir = getenv('HYDRA_PLOT_ID_DIR')
if (dir eq '') then begin
    message, 'Environment variable HYDRA_PLOT_ID_DIR not set', /cont
    return, -2
endif
lock_file = dir+'/checking_id'
id_file = dir+'/IDNUMBER'
csum_file = dir+'/CSUM'
;
;..Set OS family letter
;
if (strlowcase(!version.os_family) eq 'unix' and $
    strlowcase(!version.os) eq 'sunos') then $
  os_letter='S' $
else if (strlowcase(!version.os_family) eq 'windows') then $
  os_letter='W' $
else $
  os_letter=''

case n_params() of
    0: BEGIN
        message, 'Input string required in call to hydra_id.'
        return, -1
    END
    1: BEGIN
        is = input_string_array
        csum = check_sum(is)

        if not keyword_set(csum_vector_save) then begin
            csum_vector_save=intarr(32768)
            if (findfile(csum_file))(0) ne '' then begin
                get_lun, unit
                openr, unit, csum_file
                readf, unit, csum_vector_save
                close, unit
                free_lun, unit
            endif
        endif
        num=csum_vector_save(csum)
        if (num ne 0) then begin
            nfile=strtrim(num,2)+'.log'
            if (findfile(nfile) ne '') then begin
                get_lun, unit
                openr, unit, nfile
                s1=''
                readf, unit, s1
                readf, unit, s1
                s=[s1]
                while not eof(unit) do begin
                    readf, unit, s1
                    s=[s,s1]
                endwhile
                s=s(0:n_elements(s)-2)
                close, unit
                free_lun, unit

                if n_elements(is) eq n_elements(s) and $
                  (where(is ne s))(0) eq -1 then begin
                    message, 'Existing Match found:'+strtrim(num,2), /cont
                    return, num+os_letter
                endif

            endif
        endif

        if not keyword_set(save_flag) then begin
            save_flag=1
            input_string_array_save=is
            draft_log=0
            csum_save=csum
            id_save=-999

        endif else begin

            if (csum_save eq csum) then begin
                if n_elements(is) eq n_elements(input_string_array_save) and $
                  (where(is ne input_string_array_save))(0) eq -1 then begin
                    if keyword_set(draft) then begin
                        return, strtrim(id_save,2)+os_letter
                    endif else begin
                        if draft_log ne 0 then begin
                            return, strtrim(id_save,2)+os_letter
                        endif
                    endelse
                endif else begin
                    input_string_array_save=is
                    csum_save=csum
                    draft_log=0
                endelse
            endif else begin
                input_string_array_save=is
                csum_save=csum
                draft_log=0
            endelse
        endelse
    END 
    ELSE: BEGIN
        message, 'Too many arguments in call to hydra_id.'
        return, -1
    END 
ENDCASE
;
;..The help command gets data for the calling routine name
;
help, calls=call_stack
routine = (str_sep(call_stack(1),' '))(0)
;
;..System Calls get time/date and user information
;
time = systime()
user = getenv('USER')
;
;..Get Hostname information
;
if os_letter ne 'W' then begin
    spawn, 'uname -n', host
    spawn, '/opt/hydra/bin/check_diskspace -d '+dir, free_space
    if (long(free_space(0)) lt 5000) then begin
        if (long(free_space(0)) lt 1000) then begin
            idl_mail, 'rdh', 'Space TOO LOW for hydra_id storage', $
              'kB Free Space available for hydraid storage: '+free_space
            message, 'ERROR: Diskspace not available to store HYDRA ID.', /cont
            return, -3
        endif else begin
            idl_mail, 'rdh', 'Space running low for hydra_id storage', $
              'kB Free Space available for hydraid storage: '+free_space
            message, 'WARNING: Diskspace low for storage of HYDRA ID.', /cont
        endelse
    endif
endif else begin
    host='PC'
endelse
;
;..Wait for lockfile
;
lock_string=user+' '+host+' '+time
id_string=user+' '+host+' '+string(routine)+' '+time
if os_letter eq 'S' then begin
    status=strtrim(idl_lockfile(lock_file,id=lock_string),2)
    if (strmid(status(0),0,1) ne 0) then begin
        message, 'IDNUMBER Lockfile acquire failed.'
        return, -1
    endif
endif
;
;  Increment ID file.
;
get_lun, unit
openr, unit, id_file, error=open_error
if (open_error ne 0) then begin
    num=long(0)
endif else begin
    readf, unit, num
    close, unit
    num=long(num)
endelse
command2 = ''
id_save = long(num)

openw, unit, id_file
printf, unit, num+1
close, unit

hs='BEGIN_PLOT_ID '+strtrim(num,2)+' '+id_string
ts='END_PLOT_ID '+strtrim(num,2)

IF NOT keyword_set(draft) THEN BEGIN 
    draft_log=1
    id_log_file = dir+'/'+strtrim(num,2)+'.log'
endif else begin
    id_log_file = dir+'/'+strtrim(num,2)+'.draft'
    hs=hs+' (draft)'
endelse

openw, unit, id_log_file
printf, unit, hs
FOR i=0,n_elements(is)-1 DO printf, unit, is(i)
printf, unit, ts
close, unit

csum_vector=intarr(32768)
if (findfile(csum_file))(0) ne '' then begin
    openr, unit, csum_file
    readf, unit, csum_vector
    close, unit
endif
csum_vector(csum)=num
openw, unit, csum_file
printf, unit, csum_vector
close, unit
free_lun, unit
;
;..Remove the lockfile
;
if os_letter eq 'S' then begin
    status=strtrim(idl_lockfile(lock_file,id=lock_string, /rm),2)
    if (strmid(status(0),0,1) ne 0) then begin
        message, 'IDNUMBER Lockfile removal failed.'
        return, -1
    endif
endif

IF keyword_set(stop) THEN stop

num=strtrim(num,2)+os_letter
return, num
END
