function idl_lockfile, filename, rm=rm, wait=wait, ident=ident, $
                       timeout=timeout, debug=debug, init=init, $
                       delete_lockf=delete_lockf
;+
; NAME:
;      idl_lockfile
;
; PURPOSE:
;      Create or delete a specified lockfile.  (It waits if the lockfile
;      already exists.)  Returns a zero if successful.  Note that it
;      checks for the same identifier in the lockfile that it put there.
;
; CATEGORY:
;      File Utility.
;
; CALLING SEQUENCE:
;
;      Create the lockfile:
;      status=idl_lockfile(filename, wait=30)
;
;      Delete the lockfile:
;      status=idl_lockfile(filename, /rm)
; 
; INPUTS:
;      filename: The name of the lockfile.
;
; OPTIONAL INPUTS:
;      None.
;	
; KEYWORD PARAMETERS:
;      /rm:            Used to signal that it should delete.
;      wait=<seconds>: The number of seconds to wait for the lock to
;                      become available.
;      timeout=<seconds>: Valid Time period of lockfile. (To handle
;                         abandoned files.)
;      ident=<string>: Used to explicitly set the id used in the lockfile.
;
;      /delete_lockf:  Deletes the lockfile.lockf file used for nfs 
;                      compatibility.
;
;      /init:          Clear the identifier common block storage
;
;      /debug:         Print extra information.
;
; OTHER INPUTS:
;
;      Environment variable: IDL_LOCKFILE_DISABLE eq 1 disables operation
;
; OUTPUTS:
;      Return values is 0 for success.
;
; OPTIONAL OUTPUTS:
;      None.
;
; COMMON BLOCKS:
;      common lockfile_common, idstring, id_saved
;
;      Saves the identifier between calls for default use in deleting.
;
; SIDE EFFECTS:
;      None.
;
; RESTRICTIONS:
;      Directory for file must be writable by user.
;
; PROCEDURE:
;      Construct the identifier, 
;      Use lockfile script to do the mechanics.
;
; EXAMPLE:
;      status=idl_lockfile('LOCK')
;      if (status ne 0) then begin
;          message, 'Error in lockfile creation.'
;          stop
;      endif
;
;      <Protected File commands>
;
;      status=idl_lockfile('LOCK', /rm)
;      if (status ne 0) then begin
;          message, 'Error in lockfile deletion.'
;          stop
;      endif
;
; MODIFICATION HISTORY:
;      $log$
;-
;
;  Create and Remove lockfile
;
;  Return value of 0 indicates success.  -Not implemented yet.
;
;  status=idl_lockfile(filename)          Waits to get the lock and creates
;                                         the lockfile.
;
;  status=idl_lockfile(filename, /rm)     Deletes the lockfile.
;
;
common lockfile_common, idstring, id_saved

if getenv('IDL_LOCKFILE_DISABLE') eq '1' then begin
    message, 'Lockfile operation disabled by env.', /cont
    return, 0
endif

if keyword_set(init) then id_saved=0

if keyword_set(ident) then begin
    id_saved=1
    idstring=ident
    id_string=" -s '"+idstring(0)+"'"
endif else begin
    if not keyword_set(rm) then begin
        date = systime()
        spawn, 'uname -n', node
        user=getenv('USER')
        id_saved=1
        idstring=node+' '+user+' '+date
        id_string=" -s '"+idstring(0)+"'"
    endif else begin
        if keyword_set(id_saved) then begin
            id_string=" -s '"+idstring(0)+"'"
            id_saved=0
        endif else begin
                                ; The case where /rm is called before
                                ; it is ever opened.
            print, 'RM called first...'
            return, 0
        endelse
    endelse
endelse
if (!version.arch eq 'sparc') then begin
    lockfile='/opt/hydra/bin/lockfile.sparc -v'
endif else begin
    if (!version.arch eq 'x86' and !version.os eq 'sunos') then begin
        lockfile='/opt/hydra/bin/lockfile.x86 -v'
    endif else begin
        message, 'only defined for sunos.'
        return, 0
    endelse
endelse

if keyword_set(debug) then begin
    print, 'Lockfile='+lockfile
    print, 'Identifier command='+id_string
endif
if keyword_set(rm) then begin
    command=lockfile+' -rm'+id_string+' '+filename
endif else begin
    if keyword_set(timeout) then timeout_str=' -t '+strtrim(timeout,2)+' ' $
    else timeout_str=' -t 45 '
    if keyword_set(wait) then wait_str=' -w '+strtrim(wait,2)+' ' $
    else wait_str=' '
    command=lockfile+id_string+timeout_str+wait_str+filename
endelse
lockf=filename+'.lockf'

if keyword_set(debug) then print, command
spawn, command, result
if keyword_set(debug) then help, result
if keyword_set(debug) then for i=0, n_elements(result)-1 do print, result(i)
result=long( (str_sep(result(0),':'))(1) )

user= getenv('USER')
if user eq '' then user='rdh'

if result ne 0 then idl_mail, user, 'idl_lockfile error', $
  ['Return Status = '+strtrim(result,2), $
   'Lockfile: '+filename]
if keyword_set(delete_lockf) then spawn, 'rm -f '+lockf $
else spawn, '/bin/chmod -f 666 '+lockf
return, result
end






