;+
; NAME:
;      LOG_MAGPLOT
;
; PURPOSE:
;      Plots a function with a linear section from (-1,1) and log10
;      sections above and below.
;
; CATEGORY:
;      Plot Utility
;
; CALLING SEQUENCE:
;      log_magplot, xin, yin, <plot keywords>
; 
; INPUTS:
;      xin, yin: The coordinates to be plotted.
;
; OPTIONAL INPUTS:
;      None.
;	
; KEYWORD PARAMETERS:
;      Plot Keywords: 
;      (Note that yminor, yticks, ytickformat, yrange are used by 
;              log_magplot.)
;
; OUTPUTS:
;      A plot on the current page position.
;
; OPTIONAL OUTPUTS:
;      None.
;
; COMMON BLOCKS:
;      None.
;
; SIDE EFFECTS:
;      None.
;
; RESTRICTIONS:
;      Plot keywords yminor, yticks, ytickformat, yrange are used by 
;              log_magplot.
;
; PROCEDURE:
;      Construct the y-space:       alog10(y):         y>1
;                              y -> y:             -1<=y<=1
;                                   -alog10(-y):       y<-1
;          Then make plot and write in the proper tickmarks and labels.
;
; EXAMPLE:
;      x=findgen(1000)*!pi/500.0
;      y=10*sin(x)-5*cos(x)
;      log_magplot, x,y, title='Example Plot'
;
; MODIFICATION HISTORY:
;      Created RDH 4/13/99
;-
FUNCTION tick_lmp, axis, index, Value
;
;  Return the string for the axis
;
IF axis EQ 1 THEN BEGIN

   IF (value GT 1.0) THEN BEGIN
      v = 10.0^(value-1)
   ENDIF ELSE BEGIN
      IF (value LT -1.0) THEN BEGIN
         v = -10.0^(-value-1)
      ENDIF ELSE BEGIN 
         v = value
      ENDELSE 
   ENDELSE 
   v = strtrim_rdh(v,2)
ENDIF ELSE v = ' '

return, v
END


PRO log_magplot, xin, yin, _extra=e
;
;  Plot with log of magnitude in y, but linear -1,1
;
   nny = 10
   IF !p.multi(2) GT 1 THEN nny = nny/!p.multi(2)

   r0 = where(abs(yin) LE 1)
   rp = where(yin GT 1)
   rm = where(yin LT -1)
   y = alog10(abs(yin))
   IF r0(0) NE -1 THEN y(r0) = yin(r0)
   IF rp(0) NE -1 THEN y(rp) = y(rp)+1.0
   IF rm(0) NE -1 THEN y(rm) = -y(rm)-1.0
   
   ymax = ceil(max(y))
   ymin = floor(min(y))
   ny = ymax-ymin
;   print, ymax-ymin, ymin, ymax


   IF keyword_set(yminor) OR keyword_set(yticks) OR $
    keyword_set(ytickformat) OR keyword_set(yrange) THEN BEGIN
      print, 'Cannot use yminor, yticks, ytickformat, or yrange '+$
       'with log_magplot.'
      return
   ENDIF 

   plot, xin, y, yminor=1, yticks=ny, ytickformat='tick_lmp', $
    _extra=e, yrange=[ymin,ymax]

   dx = 0.01*(!x.crange(1)-!x.crange(0))

   FOR i=fix(ymin),fix(ymax)-1 DO BEGIN 
      IF (i GE 1) THEN yv = alog10((findgen(10)+1)*10.0^(i-1))+1 $
      ELSE IF (i LT -1) THEN yv = -alog10((findgen(10)+1)*10.0^(-i-2))-1 $
      ELSE yv = i+0.1*(findgen(10)+1)
      IF (ny GT nny) THEN BEGIN
         nj = 5
         yv = yv(2*(indgen(5))+1)
      ENDIF ELSE BEGIN
         nj = 10
      ENDELSE 
;      print, i, yv
      FOR j=0,nj-1 DO $
       oplot, [!x.crange(0),!x.crange(0)+dx], [1,1]*yv(j)
   ENDFOR 
   return
END
