function read_ddeis_qe, date=date_in, time=time_in, dbm=dbm_in, $
                        gain_id=gain_id, $
                        plot=plot, print=print, stop=stop
;+
;
;   qe=read_ddeis_qe( date=date, time=time, dbm=dbm, gain_id=gain_id, $ 
;               plot=plot,   print=print)
;
;   Return the ddeis quantum efficiencies by:
;   
;      date (and time 'hh:mm' and dbm if necessary) or gain_id by keyword
;
;   The values as a function of energy are returned.
;   A plot may be requested, and printed.
;
;   Error return values are:
;
;   -1: Trivial Error in input
;   -2: No matching interval found
;   -3: Ambiguous inputs
;   -4: No RG file found
;   -5: No IB file found
;   -6: Inconsistency between RG/IB files.
;
;-
read_gain_ids, gains_ids

if keyword_set(date_in) then begin
    date=strtrim(date_in,2)
    idate=long(date)

    if not keyword_set(time_in) then time='12:00' else time=time_in
    time_hr=strmid(time,0,2)+strmid(time,3,2)/60.0

    start_times=strmid(gains_ids.start_time,0,2)+$
      strmid(gains_ids.start_time,3,2)/60.0
    end_times=strmid(gains_ids.end_time,0,2)+$
      strmid(gains_ids.end_time,3,2)/60.0

    r=where(idate ge gains_ids.start_date and $
            idate le gains_ids.end_date $
            , nr)
    if nr gt 1 and keyword_set(dbm_in) then begin
        rr=where(gains_ids[r].mode eq dbm_in, nrr)
        if nrr eq 1 then begin
            r=r[rr]
            nr=1
        endif
    endif
    if nr gt 1 then begin
        rr=where((idate eq gains_ids[r].start_date and $
                time_hr ge start_times[r]) or $
                (idate eq gains_ids[r].end_date and $
                time_hr le end_times[r]) $
                , nrr)
        if nrr eq 1 then begin
            r=r[rr]
            nr=1
        endif
    endif

    if keyword_set(stop) then stop

endif else if keyword_set(gain_id) then begin
    r=where(gain_id eq gains_ids.gain_id, nr)
    date_in=gains_ids[r[0]].end_date

    if keyword_set(stop) then stop
endif else begin
    message, 'Must specify date/time/dbm or gain_id keyword.', /cont
    return, -1
endelse
if nr eq 0 then begin
    message, 'No gain interval found for date/gain_id.', /cont
    return, -2
endif
if nr gt 1 then begin
    message, 'input specification ambiguous: please use time and/or dbm'
    return, -3
endif
   
dbm=strtrim(gains_ids[r].mode,2)
date=strtrim(gains_ids[r].end_date,2)
gid=strtrim(gains_ids[r].gain_id,2)
dir='/opt/hydra/gain_data/block_mode_'+dbm+'/'
filename_rg='po_rg_???_'+date+'_v??_'+dbm+'.dat'
filename_ib='po_ib_???_'+date+'_v??_'+dbm+'.dat'

f_rg=findfile(dir+filename_rg, count=nf_rg)
if nf_rg eq 0 then begin
    message, 'IntRA-box gain not found for '+date, /cont
    return, -4
endif

f_ib=findfile(dir+filename_ib, count=nf_ib)
if nf_ib eq 0 then begin
    message, 'IntER-box gain not found for '+date, /cont
    return, -5
endif

read_rg_quick, 'best', date=date, mode=dbm, rg, dir=dir
read_ib_quick, 'best', date=date, mode=dbm, ib, dir=dir

nr=n_elements(rg[0,*])/10
ni=n_elements(ib[0,*])

if ni ne nr then begin
    message, 'Inconsistency between RG and IB files.', /cont
    return, -6
endif
n=ni

qe={$
     energy:fltarr(n,12), $
     qe:fltarr(n,12), $
     dqe:fltarr(n,12) $
   }

for i=0,11 do begin
    if i mod 2 eq 0 then begin
        rg_energy=rg[3,0:n-1]

        s=where(rg_energy[0] eq ib[1,*])
        for j=1,n-1 do begin
            s=[s,where(rg_energy[j] eq ib[1,*])]
        endfor
        qe_ref=ib[2,s]
        dqe_ref=ib[3,s]
    endif else begin
        rg_energy=rg[3,n:2*n-1]

        ib_energy=ib[1,s]
        qe_ref=ib[4,s]
        dqe_ref=ib[5,s]
    endelse

    qe.energy[*,i]=rg_energy
    qe.qe[*,i]=1.0/qe_ref
    qe.dqe[*,i]=dqe_ref/qe_ref^2

    if i lt 10 then begin
        ids0=i*n
        ids1=(i+1)*n-1

        qe.energy[*,i]=rg[3,ids0:ids1]
        qe.qe[*,i]=qe.qe[*,i]*rg[6,ids0:ids1]
        qe.dqe[*,i]=sqrt((qe.dqe[*,i]/qe.qe[*,i])^2+$
                         (rg[7,ids0:ids1]/rg[6,ids0:ids1])^2)

    endif
endfor

if keyword_set(plot) then begin
    if keyword_set(print) then set_ps, /landscape
    set_page, nv=3, nh=4
    for i=0,11 do begin
        plot, qe.energy[*,i], qe.qe[*,i], charsize=1.5, $
          /xlog, xrange=[10,20000], xstyle=1, $
          xtitle='Energy (eV)', $
          /ylog, yrange=[0.01, 10], ystyle=1, $
          ytitle='Det '+strtrim(i+1,2)+' Q.E.(E)', $
          title='QE '+gid+' ('+strtrim(date_in,2)+')', $
          psym=3
        oploterr, qe.energy[*,i], qe.qe[*,i], qe.dqe[*,i], 3
    endfor
    if keyword_set(print) then begin
        if print eq 2 then del=1 else del=0
        end_of_prog, /print, delete=del
    endif
endif

return, qe
end
