pro read_smear_mapping, resolution, date, iblock_range, $
                        data_range, data_dim, dir=dir, label=label, $
                        timetags=timetags, $
                        smear_indices, $
                        reload=reload, verbose=verbose, save_dir=save_dir
;
;   Read the complete smear mapping for 4.6s data and return indexes
;   to specified array for each sub-block.
;
;   Input:
;      resolution            - '4.6s'
;      date                  - defined for 19960529
;      iblock_range          - [Block Index Start, End]
;      data_range            - [xmin, ymin, xmax, ymax]
;      data_dim              - [nx, ny]
;
;
;   Input Keyword:
;      dir                   - Sets the directory to find the smear file.
;      label                 - Additional prefix to file.
;      timetags              - Timetags of the intervals to be smeared.
;      reload                - causes the file to be reread.
;      save_dir              - cache result as idlsav file in this directory.
;      verbose               - prints extra messages about failures.
;
;
;   Output:
;      smear_indices
;
;   Output Structure fields:
;      iblock, isub          - Timing
;      ixg, iyg, weight      - Separator Position Indexes and weights
;                              associated with the given time_range.
;      fraction              - Fraction of total Time inside matrix
;
;
;   A Block that is not in the smear mapping returns as ixg=iyg=weight=-1
;
;
;   Return Structure of indices
;
;   Fields:
;      iblock
;      isub
;      fraction
;      ixg
;      iyg
;      weight
;
;
;
;
;   Example:
;
;    read_smear_mapping, '4.6s', 19960529, [1800, 1900], $
;      [-2,-4,2,4], [400,800], $
;      dir='/home/ddm/bigdisk2/plot_devel/release/savsmr/0630_0730/', $
;      label='e_restr', smear_indices, /verbose, /save
;
;    hd_make_timetags, ...
;    read_smear_mapping, 'variable', 19960529, 0, $
;      [-2,-4,2,4], [400,800], $
;      timetags=timetags
;      dir='/home/ddm/bigdisk2/plot_devel/release/savsmr/0630_0730/', $
;      label='e_restr', smear_indices, /verbose, /save
;
;
   if not keyword_set(save_dir) then save_dir='/opt/hydra/tmp_data/other_data_sets/' 
   case resolution of
       '4.6s': begin
           filename=save_dir+'smear_indices_'+strtrim(date,2)+'_'+$
             label+'_'+resolution+'_'+$
             string(iblock_range(0),format='(i4.4)')+'_'+$
             string(iblock_range(1),format='(i4.4)')+'.idlsav'
       end
       'variable': begin
           filename=save_dir+'smear_indices_'+strtrim(date,2)+'_'+$
             label+'_'+resolution+'_timetags.idlsav'
       end
   endcase


   if not keyword_set(reload) then begin
       f=findfile(filename,count=nf)
       if nf eq 1 then begin
           restore, f(0)
           if (where(data_range ne smear_indices.data_range))(0) eq -1 and $
             (where(data_dim ne smear_indices.data_dim))(0) eq -1 then begin
               return
           endif
       endif
   endif

   case resolution of
       '4.6s': begin
           
           nb=(iblock_range(1)-iblock_range(0)+1)
           ns=3
           nmax_pixels_per_sub_block=256
           smear_indices={ $
                           data_range:float(data_range), $
                           data_dim:long(data_dim), $
                           iblock:(iblock_range(0)+indgen(nb))# $
                           replicate(1,3), $
                           isub:replicate(1,nb)#[0,1,2], $
                           fraction:replicate(0.0,nb,ns), $
                           ixg:replicate(-1,nmax_pixels_per_sub_block,nb,ns), $
                           iyg:replicate(-1,nmax_pixels_per_sub_block,nb,ns), $
                           weight:replicate(0.0,$
                                            nmax_pixels_per_sub_block,nb,ns)}
           
           for ib=0,nb-1 do $
             for is=0,2 do begin
               iblock=smear_indices.iblock(ib,is)
               isub=smear_indices.isub(ib,is)
               if keyword_set(verbose) then $
                 print, 'Read Smear Mapping for '+strtrim(iblock,2)+', '+$
                 strtrim(isub,2)
               read_smear_mapping_indiv, resolution, date, $
                 iblock, isub, $
                 data_range, data_dim, dir=dir, label=label, $
                 ixg, iyg, weight, fraction, reload=reload, verbose=verbose
               if ixg(0) ne -1 then begin
                   if n_elements(ixg) gt nmax_pixels_per_sub_block then begin
                       message, 'Maximum number of active pixels exceeded'+$
                         ' for '+resolution+': '+$
                         strtrim(n_elements(ixg),2)
                   endif
                   smear_indices.fraction(ib,is)=fraction
                   ni=n_elements(ixg)
                   smear_indices.ixg(0:ni-1,ib,is)=ixg
                   smear_indices.iyg(0:ni-1,ib,is)=iyg
                   smear_indices.weight(0:ni-1,ib,is)=weight
               endif
               if keyword_set(reload) then reload=0
           endfor
       end
       'variable': begin
           
           nt=n_elements(timetags)
           nmax_pixels=1000
           smear_indices={ $
                           data_range:float(data_range), $
                           data_dim:long(data_dim), $
                           timetags:timetags, $
                           fraction:replicate(0.0,nt), $
                           ixg:replicate(-1,nmax_pixels,nt), $
                           iyg:replicate(-1,nmax_pixels,nt), $
                           weight:replicate(0.0,nmax_pixels,nt)}
           
           for it=0,nt-1 do begin
               time_range=((timetags(it).bin_start-150)+$
                           [0,timetags(it).bin_width])*86400.d0
               if keyword_set(verbose) then $
                 print, 'Read Smear Mapping for time_range=', time_range
               
               read_smear_mapping_indiv, resolution, date, $
                 0, 0, time_range=time_range, $
                 data_range, data_dim, dir=dir, label=label, $
                 ixg, iyg, weight, fraction, reload=reload, verbose=verbose
               if ixg(0) ne -1 then begin
                   if n_elements(ixg) gt nmax_pixels then begin
                       message, 'Maximum number of active pixels exceeded'+$
                         ' for '+resolution+': '+$
                         strtrim(n_elements(ixg),2)
                   endif
                   smear_indices.fraction(it)=fraction
                   ni=n_elements(ixg)
                   smear_indices.ixg(0:ni-1,it)=ixg
                   smear_indices.iyg(0:ni-1,it)=iyg
                   smear_indices.weight(0:ni-1,it)=weight
               endif
               if keyword_set(reload) then reload=0
           endfor
       end
   endcase
   save, file=filename, smear_indices

end

