pro fill_read_smear_mapping_indiv, ixg, iyg, weight
ixg=-1
iyg=-1
weight=-1
end


pro read_smear_mapping_indiv, resolution, date, iblock, isub, $
                              time_range=time_range, $
                              data_range, data_dim, dir=dir, label=label, $
                              ixg, iyg, weight, fraction, $
                              reload=reload, verbose=verbose
;
;   Read the smear mapping for 4.6s data.
;
;   Input:
;      resolution            - '4.6s'
;      date                  - defined for 19960529
;      iblock, isub          - Block and sub-block indexes
;      data_range            - [xmin, ymin, xmax, ymax]
;      data_dim              - [nx, ny]
;
;
;   Input Keyword:
;      dir                   - Sets the directory to find the smear file.
;      label                 - Additional prefix to file.
;      time_range=time_range - [Start, End Times in ssm]  (Use dummies for
;                              iblock, isub.)
;      reload                - causes the file to be reread.
;      verbose               - prints extra messages about failures.
;
;
;   Output:
;      ixg, iyg, weight      - Separator Position Indexes and weights
;                              associated with the given time_range.
;      fraction              - Fraction of total Time inside matrix
;
;
;   A Block that is not in the smear mapping returns as ixg=iyg=weight=-1
;
;   Example:
;
;    read_smear_mapping_indiv, '4.6s', 19960529, 1645, 0, $
;      [-2,-4,2,4], [400,800], $
;      dir='/home/ddm/bigdisk2/plot_devel/release/savsmr/0630_0730/', $
;      label='e_restr', ixgm, iyg, weight, fraction, /verbose
;
;
; 
;
common read_smear_mapping_indiv_common, dir_save, label_save, $
  resolution_save, date_smear, $
  smear_coords, smear_index, smear_times, iblock_smear, isub_smear
;
;  Load Smear Data and convert time_intervals to Sub-Blocks
;
if not keyword_set(label) then label='' else label=label+'_'
if not keyword_set(label_save) then label_save=''
if not keyword_set(date_smear) then date_smear=0
if not keyword_set(dir) then $
  dir='/home/ddm/bigdisk2/plot_devel/release/savsmr/0630_0730/'
if not keyword_set(dir_save) then dir_save=''
if keyword_set(reload) or not keyword_set(resolution_save) or $
  date ne date_smear or dir ne dir_save then begin
    smear_file=label+'smear_'+resolution+'.idlsav'
    f=findfile(dir+smear_file,count=nf)
    if nf eq 0 then begin
        if keyword_set(verbose) then $
          print, 'smear file mapping not found for '+smear_file
        fill_read_smear_mapping_indiv, ixg, iyg, weight
        return
    endif
    footprint_smear=-1
    smear_version=-1
    
    restore, f(0)
    resolution_save=resolution
    dir_save=dir
    label_save=label
    date_smear=date
    if date ne date_smear then begin
        print, 'read_smear_mapping only coded for ', date_smear
        fill_read_smear_mapping_indiv, ixg, iyg, weight
        return
    endif
    if keyword_set(verbose) then print, 'Setting time-to-subblock translation.'

    if not keyword_set(time_range) then begin
        nt=n_elements(smear_times(*,0))
        iblock_smear=intarr(nt)
        isub_smear=intarr(nt)
        iblock_smear(0:1)=-1
        isub_smear(0:1)=-1
        htime=gen_time(strtrim(date,2))
        ftime=smear_times(*,1)/86400.d0
        for i=1,nt-1 do begin
            block_str=time_to_block(htime+ftime(i))
            iblock_smear(i)=block_str.block
            isub_smear(i)=block_str.third
        endfor
    endif
endif

x0_smear=smear_coords(0,0)
y0_smear=smear_coords(0,1)

x1_smear=smear_coords(1,0)
y1_smear=smear_coords(1,1)

nx_smear=smear_index(0,0)
ny_smear=smear_index(1,1)

dx_smear=(x1_smear-x0_smear)/float(nx_smear)
dy_smear=(y1_smear-y0_smear)/float(ny_smear)


case resolution of
    '4.6s': begin
        
        rt=where(iblock eq iblock_smear and isub eq isub_smear, nrt)
        
        if nrt eq 0 then begin
            if keyword_set(verbose) then begin
                print, 'No matching sub-blocks found.'
            endif
            fill_read_smear_mapping_indiv, ixg, iyg, weight
            return
        endif
    end
    
    'variable': begin
        rt=where(smear_times(*,0)+0.001 lt time_range(1) and $
                smear_times(*,1)-0.001 gt time_range(0), nrt)
        if nrt eq 0 then begin
            if keyword_set(verbose) then begin
                print, 'No matching variable-time-interval found.'
            endif
            fill_read_smear_mapping_indiv, ixg, iyg, weight
            return
        endif
        
    end
    else: begin
        print, 'Read_smear_mapping only coded for 4.6s repetition data.'
        fill_read_smear_mapping_indiv, ixg, iyg, weight
        return
    end
endcase

x0_out=data_range(0)
y0_out=data_range(1)

x1_out=data_range(2)
y1_out=data_range(3)

nx_out=data_dim(0)
ny_out=data_dim(1)

dx_out=(x1_out-x0_out)/float(nx_out)
dy_out=(y1_out-y0_out)/float(ny_out)

ixg1=0
iyg1=0
weight1=0

for ir=0,nrt-1 do begin

    i0=smear_index(rt(ir)+1,0)
    i1=smear_index(rt(ir)+1,1)

    if i1-i0 gt 256 then begin
        print, 'Error found in smear mapping for this sub-block.'
        fill_read_smear_mapping_indiv, ixg, iyg, weight
        return
    endif

    smear_time_range=smear_times(rt(ir),1)-smear_times(rt(ir),0)

    ix_smear=smear_coords(i0:i1,0)
    iy_smear=smear_coords(i0:i1,1)
    w_smear=smear_coords(i0:i1,2)

    r=where(ix_smear ne -1e6 and iy_smear ne -1e6, nr)
    if nr eq 0 then begin
        if keyword_set(verbose) then $
          print, 'Point is FILL in mapping.'
        fill_read_smear_mapping_indiv, ixg, iyg, weight
        return
    endif
    r=where(ix_smear lt nx_smear and ix_smear ge 0 and $
            iy_smear lt ny_smear and iy_smear ge 0, nr)
    if nr eq 0 then begin
        if keyword_set(verbose) then $
          print, 'Sub-Block maps to points OUTSIDE grid.'
        fill_read_smear_mapping_indiv, ixg, iyg, weight
        return
    endif

    if (nx_out gt nx_smear or ny_out gt ny_smear) then begin
        if keyword_set(verbose) then $
          print, 'Output matrix dimensions larger than smear dimensions:', $
          nx_smear, ny_smear
        fill_read_smear_mapping_indiv, ixg, iyg, weight
        return
    endif
    if ((nx_smear mod nx_out) ne 0 or (ny_smear mod ny_out) ne 0) then begin
        if keyword_set(verbose) then $
          print, 'Output matrix dimensions not proper subset of ', $
          'smear dimensions:', nx_smear, ny_smear
        fill_read_smear_mapping_indiv, ixg, iyg, weight
        return    
    endif
    if ((x0_smear gt x0_out) or (x1_smear lt x1_out)) or $
      ( (y0_smear gt y0_out) or (y1_smear lt y1_out)) then begin
        if keyword_set(verbose) then $
          print, 'Output matrix range larger than ', $
          'smear range:', x0_smear, y0_smear, x1_smear, y1_smear
        fill_read_smear_mapping_indiv, ixg, iyg, weight
        return    
    endif
    if ((x0_smear mod x0_out ne 0) or (x1_smear mod x1_out ne 0)) or $
      ( (y0_smear mod y0_out ne 0) or (y1_smear mod y1_out ne 0)) then begin
        if keyword_set(verbose) then $
          print, 'Output matrix range not proper subset of ', $
          'smear range:', x0_smear, y0_smear, x1_smear, y1_smear
        fill_read_smear_mapping_indiv, ixg, iyg, weight
        return    
    endif
    ixg0=long(ix_smear(r))
    iyg0=long(iy_smear(r))
    weight0=w_smear(r)
    
    xmult=dx_out/dx_smear
    ymult=dy_out/dy_smear
    
    ixoffset=(x0_out-x0_smear)/dx_smear
    iyoffset=(y0_out-y0_smear)/dy_smear
    
    ixg_out=long((ixg0-ixoffset)/xmult)
    iyg_out=long((iyg0-iyoffset)/ymult)
    
    if keyword_set(verbose) then print, 'Handling multiple passes.'
    w_out=fltarr(nx_out, ny_out)
    w_out(ixg_out, iyg_out)=weight0
    h2d=hist_2d(ixg_out, iyg_out, max1=nx_out-1, max2=ny_out-1)
    r=where(h2d gt 1, nr)
    if nr gt 0 then begin
        for i=0,nr-1 do begin
            ix=r(i) mod nx_out
            iy=r(i)/nx_out
            rr=where(ixg_out eq ix and iyg_out eq iy, nrr)
            w_out(ix,iy)=total(weight0(rr))
        endfor
    endif
    
    r=where(h2d ne 0, nr)
    if nr gt 0 then begin
        ixg0=r mod nx_out
        iyg0=r/nx_out
        weight0=w_out(r)
        fraction0=total(w_out, /double)/smear_time_range
;    print, fraction
    endif else begin
        print, 'Output matrix empty for this sub-block.'
        fill_read_smear_mapping_indiv, ixg, iyg, weight
        return    
    endelse

    if not keyword_set(ixg1) then begin
        ixg1=ixg0
        iyg1=iyg0
        weight1=weight0
        fraction1=fraction0
        total_time1=smear_time_range
    endif else begin
        ixg1=[ixg1,ixg0]
        iyg1=[iyg1,iyg0]
        fraction1=(fraction0*smear_time_range+fraction1*total_time1)/ $
          (total_time1+smear_time_range)
        total_time1=total_time1+smear_time_range
    endelse
endfor

ixg=ixg1
iyg=iyg1
weight=weight1
fraction=fraction1
end


