;+
; NAME:
;      SET_PS
;
; PURPOSE:
;      Sets the poscript device using friendly keywords, including sane
;      XSIZE,YSIZE, XOFFSET,YOFFSET.
;
; CATEGORY:
;      Program Utility
;
; CALLING SEQUENCE:
;      SET_PS, FILE = filename, ORIENTATION = orient, $
;            SCALE = scale_factor, INCHES = inches, RATIONAL = rational, $
;            XOFFSET = xoffset_in, XSIZE = xsize_in, $
;            YOFFSET = yoffset_in, YSIZE = ysize_in, COLOR = color, $
;            LANDSCAPE=landscape
;
; INPUTS:
;      None except keywords.
;
; OPTIONAL INPUTS:
;      None except keywords.
;	
; KEYWORD PARAMETERS:
;      /RESET: Reset if already in postscript.
;      FILE = filename: Output filename (idl.ps/idl.pcl/idl.gif default)
;      ORIENTATION = orient: 'L[andscape]' or 'P[ortrait]'
;      /LANDSCAPE:  Sets ORIENTATION to be Landscape, Default is Portrait.
;      SCALE = scale_factor: Set scale factor for device.
;      INCHES = inches: Set X/YOFFSETs, X/YSIZEs in inches. (Default
;                       is cm.)
;      RATIONAL = rational: Makes X and Y behave like you think they
;                           should in landscape mode.
;      XOFFSET = xoffset: Distance from left to plot surface.
;      YOFFSET = yoffset: Distance from bottom to plot surface.
;      XSIZE = xsize: Size of plot surface in x.
;      YSIZE = ysize: Size of plot surface in y.
;      COLOR = color: Set device to be color.
;
; OUTPUTS:
;      Nothing returned.  The device is set.
;
; OPTIONAL OUTPUTS:
;      None.
;
; COMMON BLOCKS:
;      None.
;
; SIDE EFFECTS:
;      Device is set to PS.
;
; RESTRICTIONS:
;      Use only at the start or after END_OF_PROG to avoid losing plots.
;
; PROCEDURE:
;      Sets the poscript device using friendly keywords, including sane
;      XSIZE,YSIZE, XOFFSET,YOFFSET, set COLOR if desired, orientation.
;
; EXAMPLE:
;      SET_PS, file='idl_test.ps', /landscape
;      plot, [0,1,4], [1,3,4], psym=-2
;      END_OF_PROG, /print, /delete
;
; MODIFICATION HISTORY:
;      None Yet.
;-

PRO SET_PS, FILE = filename, ORIENTATION = orient, $
            SCALE = scale_factor, INCHES = inches, RATIONAL = rational, $
            XOFFSET = xoffset_in, XSIZE = xsize_in, $
            YOFFSET = yoffset_in, YSIZE = ysize_in, COLOR = color, $
            LANDSCAPE=landscape, RESET=reset, number_pages=number_pages
;+
; NAME:
;      SET_PS
;
; PURPOSE:
;      Set the IDL device to postscript, offsets, sizes, orientation, 
;      postscript filename.
;
; CATEGORY:
;      Utility
;
;
; CALLING SEQUENCE:
;      SET_PS, FILE = filename, $
;            SCALE = scale_factor, INCHES = inches, RATIONAL = rational, $
;            XOFFSET = xoffset_in, XSIZE = xsize_in, $
;            YOFFSET = yoffset_in, YSIZE = ysize_in, COLOR = color, $
;            LANDSCAPE=LANDSCAPE
; 
; INPUTS:
;      None.
;
; OPTIONAL INPUTS:
;      None.
;	
; KEYWORD PARAMETERS:
;		FILENAME = name of file
;		SCALE = 1.0 for normal size plots (Used in rdh plot routines)
;		XOFFSET = left margin spacing
;		XSIZE = horizontal plot size
;		YOFFSET = bottom margin spacing
;		YSIZE = vertical plot size
;		/INCHES  if defined means size/offset values in inches
;		/RATIONAL  I have added a provision for offsets to 
;			behave as you (I) would expect 
;			(not default IDL (for landscape)).
;               /COLOR for color postscript
;               /LANDSCAPE to landscape printing (Defaults to Portrait)
;
; OUTPUTS:
;      Sets the device and size and offsets.
;
; OPTIONAL OUTPUTS:
;      None
;
; COMMON BLOCKS:
;      COMMON print_file, file_name, prev_device, present_device
;      COMMON device_char, xsize_c, ysize_c, aspect_ratio
;
; SIDE EFFECTS:
;      None.
;
; RESTRICTIONS:
;      Use with end_of_prog and end_of_page.
;
; PROCEDURE:
;      Set things up to PS using the usual IDL calls.
;
; EXAMPLE:
;      set_ps, /landscape
;      set_page, nv=2, nh=3
;      .
;      .
;      .
;      end_of_prog, /print, /del
;
;
; MODIFICATION HISTORY:
;      $Log: set_ps.pro,v $
;      Revision 1.1  2002/06/12 20:00:04  jds
;      Initial revision
;
;      Revision 1.5  2000/03/28 14:53:07  rdh
;      Added facility to number page files when using end_of_page.
;
;      Revision 1.4  2000/02/21 20:56:48  rdh
;      Added /reset keyword.
;
;      Revision 1.3  1999/03/26 16:36:25  rdh
;      Removed print of filename.
;
;      Revision 1.2  1999/02/11 18:22:43  rdh
;      Added documentation header.
;
;      Revision 1.1  1998/11/19 23:34:00  rdh
;      Initial revision
;
;-
COMMON print_file, file_name, prev_device, present_device
COMMON device_char, xsize_c, ysize_c, aspect_ratio
COMMON page_numbers, page
;
if keyword_set(number_pages) then page=1 else page=0
PREV_DEVICE0 = !D.NAME
if ((PREV_DEVICE0 NE 'PS') or (keyword_set(reset))) then begin
   SET_PLOT, 'PS'
   spawn, 'echo $$', pid
   IF KEYWORD_SET(FILENAME) THEN file_name = filename $
                            ELSE file_name = 'idl_'+strtrim(pid(0),2)+'.ps'

   DEVICE, FILENAME = file_name
   PREV_DEVICE = PREV_DEVICE0
endif else begin
    message, "Device already ps.  Type 'end_of_prog' manually or use "+$
      "'/reset' to reset device state.', /cont
endelse

IF KEYWORD_SET(ORIENT) THEN BEGIN
   IF (strlowcase(STRMID(ORIENT,0,1)) EQ 'l') THEN BEGIN
      DEVICE, /LANDSCAPE
      orient = 'L'
   ENDIF ELSE BEGIN
      DEVICE, /PORTRAIT 
      orient = 'P'
   ENDELSE
ENDIF ELSE BEGIN
   DEVICE, /PORTRAIT
   orient = 'P'
ENDELSE
IF KEYWORD_SET(LANDSCAPE) THEN BEGIN
      DEVICE, /LANDSCAPE
      orient = 'L'
ENDIF 

IF KEYWORD_SET(COLOR) THEN device, /color, bits=8

IF KEYWORD_SET(SCALE_FACTOR) THEN scale_size = scale_factor $
                             ELSE scale_size = 1.0


IF KEYWORD_SET(INCHES) THEN BEGIN

   IF (ORIENT EQ 'P') THEN BEGIN
      IF KEYWORD_SET(XOFFSET_IN) THEN xoff = xoffset_in ELSE xoff = 0.75
      IF KEYWORD_SET(YOFFSET_IN) THEN yoff = yoffset_in ELSE yoff = 0.75

      IF KEYWORD_SET(XSIZE_IN) THEN xsiz = xsize_in ELSE xsiz = 7.0
      IF KEYWORD_SET(YSIZE_IN) THEN ysiz = ysize_in ELSE ysiz = 9.5

   ENDIF ELSE BEGIN
      IF KEYWORD_SET(XOFFSET_IN) THEN xoff = xoffset_in ELSE xoff = 0.75
      IF KEYWORD_SET(YOFFSET_IN) THEN yoff = yoffset_in ELSE yoff = 10.25

      IF KEYWORD_SET(XSIZE_IN) THEN xsiz = xsize_in ELSE xsiz = 9.5
      IF KEYWORD_SET(YSIZE_IN) THEN ysiz = ysize_in ELSE ysiz = 7

      IF (KEYWORD_SET(RATIONAL) AND KEYWORD_SET(XOFFSET_IN) AND $
          KEYWORD_SET(YOFFSET_IN)) THEN BEGIN
	 temp = xoff
	 xoff = yoff
	 yoff = 11 - temp
      ENDIF

   ENDELSE

   DEVICE, /INCHES, XOFFSET=xoff, YOFFSET=yoff, $
                    XSIZE=xsiz, YSIZE=ysiz, $
                    SCALE_FACTOR = scale_size

;	print, xoff,yoff,xsiz,ysiz

ENDIF ELSE BEGIN

   IF (ORIENT EQ 'P') THEN BEGIN
      IF KEYWORD_SET(XOFFSET_IN) THEN xoff = xoffset_in ELSE xoff = 1.905
      IF KEYWORD_SET(YOFFSET_IN) THEN yoff = yoffset_in ELSE yoff = 1.905

      IF KEYWORD_SET(XSIZE_IN) THEN xsiz = xsize_in ELSE xsiz = 17.78
      IF KEYWORD_SET(YSIZE_IN) THEN ysiz = ysize_in ELSE ysiz = 24.13
   ENDIF ELSE BEGIN
      IF KEYWORD_SET(XOFFSET_IN) THEN xoff = xoffset_in ELSE xoff = 1.905
      IF KEYWORD_SET(YOFFSET_IN) THEN yoff = yoffset_in ELSE yoff = 26.035

      IF KEYWORD_SET(XSIZE_IN) THEN xsiz = xsize_in ELSE xsiz = 24.13
      IF KEYWORD_SET(YSIZE_IN) THEN ysiz = ysize_in ELSE ysiz = 17.78

      IF (KEYWORD_SET(RATIONAL) AND KEYWORD_SET(XOFFSET_IN) AND $
	  KEYWORD_SET(YOFFSET_IN)) THEN BEGIN
	 temp = xoff
	 xoff = yoff
	 yoff = 27.94 - xoff
      ENDIF
   ENDELSE


   DEVICE, XOFFSET=xoff, YOFFSET=yoff, $
           XSIZE=xsiz, YSIZE=ysiz, $
           SCALE_FACTOR = scale_size

;	print, xoff/2.54, yoff/2.54, xsiz/2.54, ysiz/2.54 

ENDELSE

xsize_c = xsiz
ysize_c = ysiz
aspect_ratio = float(xsize_c)/float(ysize_c)


END



