function cache_variable, label=label, data=data, $
                         read=read, save=save, display=display, reset=reset
;+
; NAME:
;
;      cache_variable
;
; PURPOSE:
;
;      Cache a variable, referred to by label, for running a code
;      multiple times with minimal I/O.
;
;
; CATEGORY:
;
;      Data Utility
;
; CALLING SEQUENCE:
;
;      status=cache_variable(label='dene', data=dene, /save)
; 
; INPUTS:
;
;      No Arguments.
;
; OPTIONAL INPUTS:
;
;      None.
;	
; KEYWORD PARAMETERS:
;
;      label: To refer to variable
;      data:  Variable to save/read
;      read:  /read to read info.
;      save:  /save to save variable.
;      display: Runs help on saved variable structure.
;      reset: Clears out all saved variables.
;
; OUTPUTS:
;
;      Return status:
;          Status= 0: Success for 'Read' or 'Save'.
;                  1: Label not found for 'Read'.
;                 -1: Error for 'Read' or 'Save'.
;
; OPTIONAL OUTPUTS:
;
;
;
; COMMON BLOCKS:
;
;      cache_variable_common_block
;
; SIDE EFFECTS:
;
;      Sets common block
;
; RESTRICTIONS:
;
;      None.
;
; PROCEDURE:
;
;      Read a variable, save it by label, access it by label for the
;      next run.
;
; EXAMPLE:
;
;  Save a variable called 'dene':
;
;       status=cache_variable(label='dene', data=dene, /save)
;
;  Read a variable called 'dene':
;
;       status=cache_variable(label='dene', data=dene, /read)
;
;  Display the contents of the cache:
;
;       status=cache_variable(/display)
;
;  Reset the contents of the cache:
;
;       status=cache_variable(/reset)
;
;
;  Status= 0: Success for 'Read' or 'Save'.
;          1: Label not found for 'Read'.
;         -1: Error for 'Read' or 'Save'.
;
; MODIFICATION HISTORY:
;
;-
;
common cache_variable_common_block, variables
;
;
;
if (not keyword_set(display) and not keyword_set(reset)) and $
  ((not keyword_set(label)) or $
  (not keyword_set(save) and not keyword_set(read))) then begin
    message, 'USAGE: status=cache_variable(label=<name>, data=<var>, '+$
      '/read || /save || /display)'
endif

if not keyword_set(variables) then begin
    init=1
    variables={init:init}
endif
if keyword_set(reset) then begin
    init=1
    variables={init:init}
    return, 0
endif
status=0
if keyword_set(display) then begin
    help, /str, variables
    return, 0
endif
if keyword_set(save) then begin
    if n_elements(data) gt 0 then begin
        variables=create_struct(variables, label, data)
    endif else begin
        print, 'No data element found. Nothing stored.'
        status=-1
    endelse
    read=0
endif
if keyword_set(read) then begin
    names=tag_names(variables)
    r=where(strlowcase(label) eq strlowcase(names),nr)
    if nr eq 0 then begin
        status=1
    endif else begin
        res=execute('data=variables.'+names(r(0)))
        if res ne 1 then status=-1
    endelse
endif

return, status
end
