pro error_algorithm,ele_dens_cgs,ion_dens_cgs,ele_temp_K,ion_temp_K,$
                    ion_bulk_spd_cgs=ion_bulk_spd_cgs,ele_frac_err,ele_err,$
                    ion_frac_err,ion_err,resolution=resolution

;+  
;   Program for estimating the fractional error on integrated density
;   measurements.  We assume the fractional error to be 1/N^(1/2),
;   where N is the maximum counts in a given measurement.
;   For a distribution function f, flux into a given detector is given by
;
;               Flux = v x f
;
;   To compute the number of counts in the measurement, take
;
;            N = Flux x solid angle of detector
;                     x area of detector
;                     x time interval of a measurement at E
;                     x v^2 dv  (velocity space volume increment)
;
;   Available data are
;   
;         delta omega:     solid angle of detector = 0.0153 steradians
;         delta A:         area of detector = 0.242 cm^2
;         delta t:         time interval of a measurement at E = 0.0124
;         delta v/v:       fractional velocity space covered = 0.03
;
;   Therefore errors are calculated from 
;
;         N = v * f * v^2 * delta_v/v * v * delta_omega * delta_A *
;   delta_t 
;           = f * v^4 * delta_v/v * delta_omega * delta_A * delta_t
;
;   N.B.:  This program is for use with descendents of FXY's Test4,
;          e. g. find_q_eff.pro
;
;   INPUTS:  ele_dens_cgs,ion_dens_cgs,ele_temp_K,ion_temp_K,
;            ion_bulk_spd_cgs
;   OUTPUTS: ele_frac_err,ele_err,ion_frac_err,ion_err
;
;   OPTIONAL KEYWORD: resolution,  possible values
;'one_swp','two_swp','block'.  Default is 'block'
;
;-

if (not keyword_set(resolution)) then resolution='block'
if (resolution eq 'one_swp') then begin
    num_sample=1.0
endif else if (resolution eq 'two_swp') then begin
    num_sample=2.0
endif else if (resolution eq 'block') then begin
    num_sample=6.0
endif

ele_mass_cgs=9.11d-28
ion_mass_cgs=1.67d-24

del_area=.242d0
del_t=.024d0
del_omega=0.0153d0
del_v_over_v=0.03d0

C=4*exp(-2.d0)*del_v_over_v*del_omega*del_area*del_t/!dpi^1.5
;    Electron error
ele_mean_nrg_ev=hd_divide(ele_temp_K,11.6d3)
ele_frac_err=1.d0/sqrt(C*ele_dens_cgs* $
                       sqrt(2*1.6d-12*ele_mean_nrg_ev/ele_mass_cgs))
ele_err=ele_frac_err*ele_dens_cgs
if (not keyword_set(ion_bulk_spd_cgs)) then begin
;    Original ion error
    ion_mean_nrg_ev=hd_divide(ion_temp_K,11.6d3)
    ion_frac_err=1.d0/sqrt(C*ion_dens_cgs* $
                           sqrt(2*1.6d-12*ion_mean_nrg_ev/ion_mass_cgs))
    ion_err=ion_frac_err*ion_dens_cgs
endif else begin    
;;;  Ion estimate takes bulk speed into account.
    C_ion=del_v_over_v*del_omega*del_area*del_t/!dpi^1.5
    ion_mean_nrg_ev=hd_divide(ion_temp_K,11.6d3)
    ion_th_spd_sqd_cgs=2*ion_mean_nrg_ev*1.6d-12/ion_mass_cgs
    ion_th_spd_cgs=hd_sqrt(ion_th_spd_sqd_cgs)
    v_max_0=0.5*(ion_bulk_spd_cgs+$
                 sqrt(ion_bulk_spd_cgs^2+8*ion_th_spd_sqd_cgs))
    v_max_1=0.5*(ion_bulk_spd_cgs-$
                 sqrt(ion_bulk_spd_cgs^2+8*ion_th_spd_sqd_cgs))
    max_counts_0=(C_ion*v_max_0)*(v_max_0/ion_th_spd_cgs)^3* $
      ion_dens_cgs*exp(-((v_max_0-ion_bulk_spd_cgs)/ion_th_spd_cgs)^2)
    max_counts_1=(C_ion*v_max_1)*(v_max_1/ion_th_spd_cgs)^3* $
      ion_dens_cgs*exp(-((v_max_1-ion_bulk_spd_cgs)/ion_th_spd_cgs)^2)
    max_counts=max_counts_0
    a_gt=where(max_counts_1 gt max_counts_0,gt_count)
    if (gt_count gt 0) then max_counts(a_gt)=max_counts_1(a_gt)    
    ion_frac_err=1.d0/sqrt(max_counts)
    ion_err=ion_frac_err*ion_dens_cgs    
endelse
ele_frac_err=ele_frac_err/sqrt(num_sample)
ele_err=ele_err/sqrt(num_sample)
ion_frac_err=ion_frac_err/sqrt(num_sample)
ion_err=ion_err/sqrt(num_sample)

return

end



