PRO histogram_2d_fxy, array1, array2, x, y, z, $
                      binsize1=binsize1, binsize2=binsize2, $
                      min1=min1, min2=min2, max1=max1, max2=max2, $
                      residual=residual, log1=log1, log2=log2
;+
; NAME:
;      HISTOGRAM_2d_fxy
;
; PURPOSE:
;      Create a 1-D histogram and provide x,y values for plotting.
;
; CATEGORY:
;      Histogram
;
;
; CALLING SEQUENCE:
;      histogram_2d_Trdh, array1, array2, x, y, z, $
;                     binsize1=binsize1, binsize2=binsize2, $
;                     min1=min1, min2=min2, max1=max1, max2=max2, $
;                     residual=residual, log1=log1, log2=log2
;
; INPUTS:
;      array1, array2: vectors to be histogrammed.
;
; OPTIONAL INPUTS:
;      None.
;
;	
; KEYWORD PARAMETERS:
;      BINSIZE1, BINSIZE2: The binsizes for the histogram.  (Defaults to 1.)
;      MIN1, MIN2, MAX1, MAX2: The minimum and maximum values for the 
;                histogram in the two directions.
;                If not set they default to the limits of the data
;                with some effort made to have reasonable bin intervals.
;      RESIDUAL: An Output array giving the number of hits outsize the
;                minimum, maximum interval: [# < MIN, # > MAX].
;      LOG1, LOG2:    Histogram the Log10 of the data arrays.
;
; OUTPUTS:
;      x, y, z: The values to plot w/psym=10
;      NOTE that an additional bucket is placed at the beginning and
;                the end of x and y (with z=0) to clean-up the plot.
;
; OPTIONAL OUTPUTS:
;      None.
;
; COMMON BLOCKS:
;      None.
;
; SIDE EFFECTS:
;      None.
;
; RESTRICTIONS:
;      None.
;
; PROCEDURE:
;      Histogram the array using the IDL routine hist_2d and then
;      prepare the x, y vectors to reflect the bin centers.
;
; EXAMPLE:
;      histogram_2d_rdh, array1, array2, x, y, z
;      surface, x, y, z, /lego
;
;-
;
IF n_params() EQ 0 THEN BEGIN
   print, 'USAGE: histogram_rdh, array1, array2, x, y, z, '+ $
     'binsize1=binsize1, binsize2=binsize2, min1=min1, min2=min2, '+ $
     'max1=max1, max2=max2, residual=residual'
   return
ENDIF 
IF NOT keyword_set(binsize1) THEN binsize1 = 1.0
IF NOT keyword_set(binsize2) THEN binsize2 = 1.0
if keyword_set(log1) then array_1=alog10(array1) else array_1=array1
if keyword_set(log2) then array_2=alog10(array2) else array_2=array2

IF n_elements(min1) EQ 0 THEN min1 = floor(min(array_1)/binsize1)*binsize1
IF n_elements(min2) EQ 0 THEN min2 = floor(min(array_2)/binsize2)*binsize2
IF n_elements(max1) EQ 0 THEN max1 = floor(max(array_1)/binsize1+1)*binsize1
IF n_elements(max2) EQ 0 THEN max2 = floor(max(array_2)/binsize2+1)*binsize2
;
;   Check number of bins
;
nbins1=(max1-min1)/binsize1
if nbins1 gt (2L)^31-1 then begin
    binsize1=(max1-min1)/(1000000.d0)
    message, 'Number of bins (1) too large: Setting binsize to '+$
      strtrim(binsize1,2), /cont
endif

nbins2=(max2-min2)/binsize2
if nbins2 gt (2L)^31-1 then begin
    binsize2=(max2-min2)/(1000000.d0)
    message, 'Number of bins (2) too large: Setting binsize to '+$
      strtrim(binsize2,2), /cont
endif

if min1 ge max1 then $
  message, 'ERROR: min1 ge max1: min1='+strtrim(min1,2)+$
  ', max='+strtrim(max1,2)
if min2 ge max2 then $
  message, 'ERROR: min2 ge max2: min2='+strtrim(min2,2)+$
  ', max2='+strtrim(max2,2)

n1=(max1-min1)/binsize1+1
x = findgen(n1)*binsize1+min1+0.5*binsize1

n2=(max2-min2)/binsize2
y = findgen(n2)*binsize2+min2+0.5*binsize2

z=replicate(0L, n1, n2)

for j=0,n2-1 do begin
    r=where(array_2 ge y(j)-0.5*binsize2 and array_2 lt y(j)+0.5*binsize2, nr)
    if nr gt 0 then begin
      h=histogram(array_1(r), min=min1, max=max1, binsize=binsize1)  
      z(*,j)=h
    endif
endfor

residual = replicate(0, 2, 2)

residual(*) = -99


return
stop
END
