PRO histogram_jds_1


, array, x, y, binsize=binsize, min=min, max=max, $
                   residual=residual, binnum=binnum, log=log
;+
; NAME:
;      HISTOGRAM_RDH
;
; PURPOSE:
;      Create a 1-D histogram and provide x,y values for plotting.
;
; CATEGORY:
;      Histogram
;
;
; CALLING SEQUENCE:
;      histogram_rdh, array, x, y, binsize=binsize, min=min, max=max, $
;                   residual=residual
;
; INPUTS:
;      array: vector to be histogrammed.
;
; OPTIONAL INPUTS:
;      None.
;
;	
; KEYWORD PARAMETERS:
;      BINSIZE: The binsize for the histogram.  (Defaults to 1.)
;      MIN, MAX: The minimum and maximum values for the histogram.
;                If not set they default to the limits of the data
;                with some effort made to have reasonable bin intervals.
;      RESIDUAL: An Output array giving the number of hits outsize the
;                minimum, maximum interval: [# < MIN, # > MAX].
;      BINNUM: The vector that gives the bin number of each input point.
;      LOG:    Histogram the Log10 of the data array.
;
; OUTPUTS:
;      x, y: The values to plot w/psym=10
;      NOTE that an additional bucket is placed at the beginning and
;                the end of x (with y=0) to clean-up the plot.
;
; OPTIONAL OUTPUTS:
;      None.
;
; COMMON BLOCKS:
;      None.
;
; SIDE EFFECTS:
;      None.
;
; RESTRICTIONS:
;      None.
;
; PROCEDURE:
;      Histogram the array using the IDL routine histogram and then
;      prepare the x vector to reflect the bin centers.
;
; EXAMPLE:
;      histogram, array, x, y
;      plot, x, y, psym=10
;
; MODIFICATION HISTORY:
;      $Log: histogram_rdh.pro,v $
;      Revision 1.6  2000/11/30 17:30:03  rdh
;      Added log keyword.
;
;      Revision 1.5  2000/10/09 15:18:30  rdh
;      Oops 2.
;
;      Revision 1.4  2000/10/09 13:46:35  rdh
;      Oops.
;
;      Revision 1.3  2000/10/06 19:55:23  rdh
;      Added sanity-check for number of bins.
;
;      Revision 1.2  1999/04/09 22:37:47  rdh
;      Added binnum keyword to return the bin numbers of each input point.
;
;      Revision 1.1  1998/11/17 18:26:39  rdh
;      Initial revision
;
;-
;
IF n_params() EQ 0 THEN BEGIN
   print, 'USAGE: histogram_rdh, array, x, y, binsize=binsize, '+$
    'min=min, max=max, residual=residual'
   return
ENDIF 
IF NOT keyword_set(binsize) THEN binsize = 1.0
if keyword_set(log) then array1=alog10(array) else array1=array

IF n_elements(min) EQ 0 THEN min = floor(min(array1)/binsize)*binsize
IF n_elements(max) EQ 0 THEN max = floor(max(array1)/binsize+1)*binsize
;
;   Check number of bins
;
nbins=(max-min)/binsize
if nbins gt (2L)^31-1 then begin
    binsize=(max-min)/(1000000.d0)
    message, 'Number of bins too large: Setting binsize to '+$
      strtrim(binsize,2), /cont
endif

h = histogram(array1, min=min, max=max, binsize=binsize)
n = n_elements(h)+2
x = findgen(n)*binsize+min-0.5*binsize
y = lonarr(n)
y(0) = 0
y(n-1) = 0
y(1:n-2) = h

residual = replicate(0, 2)
r = where(array1 GT max)
IF r(0) NE -1 THEN residual(1) = n_elements(r)
r = where(array1 LT min)
IF r(0) NE -1 THEN residual(0) = n_elements(r)

   binnum = fix((array1-x(0)+0.5*binsize)/binsize)
   print, 'binnum=', binnum


return
END
