; PROGRAM minvar.pro
; BEGIN PROLOGUE minvar
; PURPOSE: FIND THE MINIMUM VARIANCE DIRECTION USING THE BOOTSTRAP METHOD.
; AUTHOR: ARTHUR J. HULL
; TYPE: DOUBLE PRECISION
; REFERENCES: Kawano and Higuchi (GRL, vol 22, pg 307, 1995)
; REVISION HISTORY:
;              9/19/96 DATE WRITTEN
; END PROLOGUE
;=============================================================================
PRO INITIALIZE
COMMON INITDAT,FILE,T1,T2,NSEQ
COMMON TIMESELDAT,T_U1,T_U2,T_D1,T_D2,T_L1,T_L2

FILE = 't921205_08:00-08:12.dat'

; 1ST SHOCK EVENT FIELD INTERVAL
;T1 = 29185.2  ;18 SECOND TIME INTERVAL
;T2 = 29203.2
;T1 = 29180    ;30 SECOND TIME INTERVAL
;T2 = 29210
;T1 = 29165    ;60 SECOND TIME INTERVAL
;T2 = 29225
T1 = 29100    ;150 SECOND TIME INTERVAL
T2 = 29250
;T1 = 29045    ;5 MIN TIME INTERVAL
;T2 = 29345

T1 = 29104.68
T2 = 29328.079

T_U1 = T1
T_U2 = 29142.342
T_D1 = 29290.520
T_D2 = T2
T_L1 = 29194.92
T_L2 = 29196.92

;T_U1 = 29150.
;T_U2 = 29160.342
;T_D1 = 29290.520
;T_D2 = T2
;T_L1 = 29180.
;T_L2 = 29260.

NSEQ = 10   ;NUMBER OF REPLICATED FIELD SEQUENCES USED IN BOOTSTRAP TECHNIQUE

END
;=============================================================================
PRO MAGDAT2
COMMON FIELD,NPTS,TBI,BX,BY,BZ,B

DIM = 100000

TBI = DBLARR(DIM)
BX = DBLARR(DIM)
BY = DBLARR(DIM)
BZ = DBLARR(DIM)

OPENR,10,'bsim.dat'
NPTS = 0
WHILE NOT EOF(10) DO BEGIN
   READF,10, DUM0,DUM1,DUM2,DUM3
   TBI(NPTS) = DUM0
   BX(NPTS) = DUM1
   BY(NPTS) = DUM2
   BZ(NPTS) = DUM3
   NPTS = NPTS + 1
ENDWHILE
CLOSE,10

TBI = TBI(0:NPTS-1)
BX = BX(0:NPTS-1)
BY = BY(0:NPTS-1)
BZ = BZ(0:NPTS-1)

B = SQRT(BX^2 + BY^2 + BZ^2)

RETURN
END
;=============================================================================
PRO MAGDAT
;PURPOSE:  SCAN APPROPRIATE MAGNETOMETER DATA FILE FOR MAGNETIC FIELD DATA
;          SERIES WHICH MATCHES THE SELECTED TIME INTERVAL.
COMMON INITDAT,FILE,T1,T2,NSEQ
COMMON FIELD,NPTS,TBI,BX,BY,BZ,B

DIM = 100000

YEAR = INTARR(1)
MO = INTARR(1)
DAY =  INTARR(1)
HH =  INTARR(1)
MM =  INTARR(1)
SCDS = DBLARR(1)
BBX = DBLARR(1)
BBY = DBLARR(1)
BBZ = DBLARR(1)
BB = DBLARR(1)

HRS =  INTARR(DIM)
MNTS =  INTARR(DIM)
SEC = DBLARR(DIM)
BX = DBLARR(DIM)
BY = DBLARR(DIM)
BZ = DBLARR(DIM)
B = DBLARR(DIM)

OPENR,30,'/home/ajh/bigdisk/magdat/'+FILE

;SEARCH MAGNETOMETER DATA FOR START TIME
TIME = 0D0
WHILE (TIME LT T1) DO BEGIN
   READF,30,$
   FORMAT = '(30X,2(I2,1X),D6.3)' $
    ,HH,MM,SCDS
   TIME = 3600D0*HH(0) + 60D0*MM(0) + SCDS(0)
ENDWHILE

;REPOSITION POINTER
POINT = FSTAT(30)
POINT_LUN,30,POINT.CUR_PTR - LONG(2*104)

;LOAD MAGNETOMETER DATA 
NPTS = 0
WHILE (TIME LT T2) DO BEGIN
  READF,30,$
  FORMAT = '(19X,I4,4(1X,I2),1X,D6.3,1X,5X,D10.6,3(6X,D9.6))' $
   ,YEAR,MO,DAY,HH,MM,SCDS,BBX,BBY,BBZ,BB
  TIME = 3600D0*HH(0) + 60D0*MM(0) + SCDS(0)
  HRS(NPTS) = HH
  MNTS (NPTS) = MM
  SEC(NPTS) = SCDS
  BX(NPTS) = BBX
  BY(NPTS) = BBY
  BZ(NPTS) = BBZ
  B(NPTS) = BB
  NPTS = NPTS + 1
ENDWHILE
CLOSE,30
;REDIMENSIONLIZE ARRAYS TO APPROPRIATE LENGTH

HRS = HRS(0:NPTS-1)
MNTS = MNTS(0:NPTS-1)
SEC = SEC(0:NPTS-1)
BX = BX(0:NPTS-1)
BY = BY(0:NPTS-1)
BZ = BZ(0:NPTS-1)
B = B(0:NPTS-1)
TBI = 3600D0*HRS + 60D0*MNTS + SEC
RETURN
END
;=============================================================================
PRO BINTVALS
COMMON TIMESELDAT,T_U1,T_U2,T_D1,T_D2,T_L1,T_L2
COMMON FIELD,NPTS,TBI,BX,BY,BZ,B
COMMON BINTVALDAT,BXU1U2,BYU1U2,BZU1U2,BXD1D2,BYD1D2,BZD1D2,BXU2L1 $
,BYU2L1,BZU2L1,BXL2D1,BYL2D1,BZL2D1,NPTS_AVG,NPTS_U2L1,NPTS_L2D1
COMMON INTVALDAT,IND_U1U2,IND_D1D2,IND_U2L1,IND_L2D1

IND_U1U2 = WHERE(TBI GE T_U1 AND TBI LE T_U2,NPTS_U1U2)
IND_D1D2 = WHERE(TBI GE T_D1 AND TBI LE T_D2,NPTS_D1D2)
IND_U2L1 = WHERE(TBI GE T_U2 AND TBI LE T_L1,NPTS_U2L1)
IND_L2D1 = WHERE(TBI GE T_L2 AND TBI LE T_D1,NPTS_L2D1)

BXU1U2 = BX(IND_U1U2)
BYU1U2 = BY(IND_U1U2)
BZU1U2 = BZ(IND_U1U2)

BXD1D2 = BX(IND_D1D2)
BYD1D2 = BY(IND_D1D2)
BZD1D2 = BZ(IND_D1D2)

BXU2L1 = BX(IND_U2L1)
BYU2L1 = BY(IND_U2L1)
BZU2L1 = BZ(IND_U2L1)

BXL2D1 = BX(IND_L2D1)
BYL2D1 = BY(IND_L2D1)
BZL2D1 = BZ(IND_L2D1)

NPTS_AVG = ROUND((NPTS_U1U2 + NPTS_D1D2)/4D0) ;AVG # POINTS IN DATA SAMPLES

PRINT,'NPTS_U1U2 = ',NPTS_U1U2
PRINT,'NPTS_D1D2 = ',NPTS_D1D2
PRINT,'NPTS_U2L1 = ',NPTS_U2L1
PRINT,'NPTS_L2D1 = ',NPTS_L2D1
PRINT,'NPTS_AVG = ',NPTS_AVG
RETURN
END
;=============================================================================
PRO BSAMP
COMMON BINTVALDAT,BXU1U2,BYU1U2,BZU1U2,BXD1D2,BYD1D2,BZD1D2,BXU2L1 $
,BYU2L1,BZU2L1,BXL2D1,BYL2D1,BZL2D1,NPTS_AVG,NPTS_U2L1,NPTS_L2D1
COMMON SAMPDAT,BXSAMP,BYSAMP,BZSAMP,BXUSAMP,BYUSAMP,BZUSAMP
;COMMON INTVALDAT,IND_U1U2,IND_D1D2,IND_U2L1,IND_L2D1
;COMMON FIELD,NPTS,TBI,BX,BY,BZ,B
COMMON SEEDDAT,S1,S2

SEL_U2L1 = FIX(NPTS_U2L1*RANDOMU(S1,NPTS_AVG))
SEL_L2D1 = FIX(NPTS_L2D1*RANDOMU(S2,NPTS_AVG))

BXSAMP = [BXU1U2,BXU2L1(SEL_U2L1),BXL2D1(SEL_L2D1),BXD1D2]
BYSAMP = [BYU1U2,BYU2L1(SEL_U2L1),BYL2D1(SEL_L2D1),BYD1D2]
BZSAMP = [BZU1U2,BZU2L1(SEL_U2L1),BZL2D1(SEL_L2D1),BZD1D2]

BXUSAMP = (MOMENT([BXU1U2,BXU2L1(SEL_U2L1)]))(0)
BYUSAMP = (MOMENT([BYU1U2,BYU2L1(SEL_U2L1)]))(0)
BZUSAMP = (MOMENT([BZU1U2,BZU2L1(SEL_U2L1)]))(0)

BXUAVG = (MOMENT(BXU1U2))(0)
BYUAVG = (MOMENT(BYU1U2))(0)
BZUAVG = (MOMENT(BZU1U2))(0)

BXDAVG = (MOMENT(BXD1D2))(0)
BYDAVG = (MOMENT(BYD1D2))(0)
BZDAVG = (MOMENT(BZD1D2))(0)


STOP
RETURN
END
;=============================================================================
PRO FINDNORM,BX,BY,BZ
;PURPOSE:  CONSTRUCT THE COVARIANCE MATRIX AND DETERMINE THE EIGENVALUES
;          AND EIGENVECTORS.  THE SHOCK NORMAL CORRESPONDS TO THE EIGENVECTOR
;          ASSOCIATED WITH THE SMALLEST EIGENVALUE.
;          EIGENVALUES ARE STORED IN D AND CORRESPONDING EIGENVECTORS
;          ARE STORED IN THE COLLUMNS OF M.
COMMON EIGVECT,EVAL,EVECT1,EVECT2,EVECT3,EVECT

MOMBX = MOMENT(BX)
MOMBY = MOMENT(BY)
MOMBZ = MOMENT(BZ)
MOMBXBY = MOMENT(BX*BY)
MOMBXBZ = MOMENT(BX*BZ)
MOMBYBZ = MOMENT(BY*BZ)
MOMBXBX = MOMENT(BX*BX)
MOMBYBY = MOMENT(BY*BY)
MOMBZBZ = MOMENT(BZ*BZ)

EVECT = DBLARR(3,3)
EVAL = DBLARR(3)
E = DBLARR(3)

EVECT(0,0) = MOMBXBX(0) - MOMBX(0)^2
EVECT(0,1) = MOMBXBY(0) - MOMBX(0)*MOMBY(0)
EVECT(0,2) = MOMBXBZ(0) - MOMBX(0)*MOMBZ(0)
EVECT(1,0) = EVECT(0,1)
EVECT(1,1) = MOMBYBY(0) - MOMBY(0)^2
EVECT(1,2) = MOMBYBZ(0) - MOMBY(0)*MOMBZ(0)
EVECT(2,0) = EVECT(0,2)
EVECT(2,1) = EVECT(1,2)
EVECT(2,2) = MOMBZBZ(0) - MOMBZ(0)^2

TRIRED,EVECT,EVAL,E,/DOUBLE
TRIQL,EVAL,E,EVECT,/DOUBLE

SORTEVAL
END
;=============================================================================
PRO SORTEVAL
COMMON EIGVECT,EVAL,EVECT1,EVECT2,EVECT3,EVECT

INDEX = SORT(EVAL)
EVAL = EVAL(INDEX)
EVECT1 = EVECT(*,INDEX(0))
EVECT2 = EVECT(*,INDEX(1))
EVECT3 = EVECT(*,INDEX(2))
EVECT = EVECT(*,INDEX)

;MAKE SURE RHS IS GIVEN
IF (EVECT1(0) LT 0) THEN EVECT1 = -EVECT1
IF (TOTAL(CROSS(EVECT1,EVECT2)*EVECT3) LT 0.0) THEN EVECT2 = -EVECT2
END
;=============================================================================
PRO FINDERR
COMMON EIGVECT,EVAL,EVECT1,EVECT2,EVECT3,EVECT
COMMON FIELD,NPTS,TBI,BX,BY,BZ,B
COMMON ERRDAT,DBN

BXROT = EVECT1(0)*BX + EVECT1(1)*BY + EVECT1(2)*BZ
BYROT = EVECT2(0)*BX + EVECT2(1)*BY + EVECT2(2)*BZ
BZROT = EVECT3(0)*BX + EVECT3(1)*BY + EVECT3(2)*BZ

BYAVG = (MOMENT(BYROT))(0)
BZAVG = (MOMENT(BZROT))(0)

DLAMBDA = EVAL(0)*SQRT(2D0/NPTS)
DBN = SQRT(EVAL(0)/NPTS + BYAVG^2*DLAMBDA/(EVAL(1) - EVAL(0)) $
+ BZAVG^2*DLAMBDA/(EVAL(2) - EVAL(0)))

RETURN
END
;=============================================================================
PRO REPB
COMMON INITDAT,FILE,T1,T2,NSEQ
COMMON SAMPDAT,BXSAMP,BYSAMP,BZSAMP,BXUSAMP,BYUSAMP,BZUSAMP
COMMON EIGVECT,EVAL,EVECT1,EVECT2,EVECT3,EVECT
COMMON REPBDAT,NX,NY,NZ,LX,LY,LZ,MX,MY,MZ,BXU,BYU,BZU,BU,THBNU
COMMON SEEDDAT,S1,S2

NX = DBLARR(NSEQ)
NY = DBLARR(NSEQ)
NZ = DBLARR(NSEQ)

LX = DBLARR(NSEQ)
LY = DBLARR(NSEQ)
LZ = DBLARR(NSEQ)

MX = DBLARR(NSEQ)
MY = DBLARR(NSEQ)
MZ = DBLARR(NSEQ)



BXU = DBLARR(NSEQ)
BYU = DBLARR(NSEQ)
BZU = DBLARR(NSEQ)

S1 = SYSTIME(1)
S2 = 1E8+S1
FOR I = 0,NSEQ-1 DO BEGIN
   BSAMP

   BXU(I) = BXUSAMP
   BYU(I) = BYUSAMP
   BZU(I) = BZUSAMP

   FINDNORM,BXSAMP,BYSAMP,BZSAMP

   NX(I) = EVECT1(0)
   NY(I) = EVECT1(1)
   NZ(I) = EVECT1(2)

   LX(I) = EVECT2(0)
   LY(I) = EVECT2(1)
   LZ(I) = EVECT2(2)

   MX(I) = EVECT2(0)
   MY(I) = EVECT2(1)
   MZ(I) = EVECT2(2)
ENDFOR

BU = SQRT(BXU^2 + BYU^2 + BZU^2)

THBNU = ACOS((NX*BXU + NY*BYU + NZ*BZU)/BU)

THBNUAVG = (MOMENT(THBNU))(0)

IF(THBNUAVG GT !PI/2D0) THEN THBNU = !PI - THBNU

END
;=============================================================================
PRO HISTREPB
COMMON REPBDAT,NX,NY,NZ,LX,LY,LZ,MX,MY,MZ,BXU,BYU,BZU,BU,THBNU
COMMON HISTDAT,NXHIST,NXBINS,NYHIST,NYBINS,NZHIST,NZBINS,BXHIST,BXBINS $
,BYHIST,BYBINS,BZHIST,BZBINS,BHIST,BBINS,THBNHIST,THBNBINS

BINS = 100

NXMIN = MIN(NX)
NXMAX = MAX(NX)
NXSIZE = (NXMAX - NXMIN)/DOUBLE(BINS)

NYMIN = MIN(NY)
NYMAX = MAX(NY)
NYSIZE = (NYMAX - NYMIN)/DOUBLE(BINS)

NZMIN = MIN(NZ)
NZMAX = MAX(NZ)
NZSIZE = (NZMAX - NZMIN)/DOUBLE(BINS)

BXMIN = MIN(BXU)
BXMAX = MAX(BXU)
BXSIZE = (BXMAX - BXMIN)/DOUBLE(BINS)

BYMIN = MIN(BYU)
BYMAX = MAX(BYU)
BYSIZE = (BYMAX - BYMIN)/DOUBLE(BINS)

BZMIN = MIN(BZU)
BZMAX = MAX(BZU)
BZSIZE = (BZMAX - BZMIN)/DOUBLE(BINS)

BMIN = MIN(BU)
BMAX = MAX(BU)
BSIZE = (BMAX - BMIN)/DOUBLE(BINS)

THBNMIN = MIN(THBNU)
THBNMAX = MAX(THBNU)
THBNSIZE = (THBNMAX - THBNMIN)/DOUBLE(BINS)

NXHIST = HISTOGRAM(NX,BINSIZE=NXSIZE)
NXBINS = NXMIN + NXSIZE*DINDGEN(BINS+1)

NYHIST = HISTOGRAM(NY,BINSIZE=NYSIZE)
NYBINS = NYMIN + NYSIZE*DINDGEN(BINS+1)

NZHIST = HISTOGRAM(NZ,BINSIZE=NZSIZE)
NZBINS = NZMIN + NZSIZE*DINDGEN(BINS+1)

BXHIST = HISTOGRAM(BXU,BINSIZE=BXSIZE)
BXBINS = BXMIN + BXSIZE*DINDGEN(BINS+1)

BYHIST = HISTOGRAM(BYU,BINSIZE=BYSIZE)
BYBINS = BYMIN + BYSIZE*DINDGEN(BINS+1)

BZHIST = HISTOGRAM(BZU,BINSIZE=BZSIZE)
BZBINS = BZMIN + BZSIZE*DINDGEN(BINS+1)

BHIST = HISTOGRAM(BU,BINSIZE=BSIZE)
BBINS = BMIN + BSIZE*DINDGEN(BINS+1)

THBNHIST = HISTOGRAM(THBNU,BINSIZE=THBNSIZE)
THBNBINS = THBNMIN + THBNSIZE*DINDGEN(BINS+1)

END
;=============================================================================
PRO PLOTHIST
COMMON HISTDAT,NXHIST,NXBINS,NYHIST,NYBINS,NZHIST,NZBINS,BXHIST,BXBINS $
,BYHIST,BYBINS,BZHIST,BZBINS,BHIST,BBINS,THBNHIST,THBNBINS
COMMON MOMDAT,NORM,DNORM,BUVECT,DBUVECT,BUAVG,DBU,THBNAVG,DTHBN $
,LVECT,DLVECT,MVECT,DMVECT

SNORM = STRMID(ROUND(NORM*100D0)/100D0,4,6)
SDNORM = STRMID(ROUND(DNORM*100D0)/100D0,3,6)

SBVECT = STRMID(ROUND(BUVECT*100D0)/100D0,5,6)
SDBVECT = STRMID(ROUND(DBUVECT*100D0)/100D0,3,6)

SB = STRMID(ROUND(BUAVG*100D0)/100D0,5,6)
SDB = STRMID(ROUND(DBU*100D0)/100D0,3,6)

STHBN = STRMID(ROUND(THBNAVG*10D0*180D0/!PI)/10D0,5,6)
SDTHBN = STRMID(ROUND(DTHBN*10D0*180D0/!PI)/10D0,3,6)

!X.STYLE=1
SET_PLOT,'PS'
DEVICE,/LANDSCAPE,FILENAME = 'minvar.ps'
!P.MULTI=[0,2,2]

PLOT,NXBINS,NXHIST,PSYM=10,YTITLE='!17# PTS',XTITLE='!17Nx'
PLOT,NYBINS,NYHIST,PSYM=10,YTITLE='!17# PTS',XTITLE='!17Ny'
PLOT,NZBINS,NZHIST,PSYM=10,YTITLE='!17# PTS',XTITLE='!17Nz'
XYOUTS,.5,.4,/NORMAL,'!17HISTOGRAM OF COMPONENTS OF THE SHOCK NORMAL'
XYOUTS,.5,.38,/NORMAL,'!17DETERMINED FROM A STATISTICAL ENSEMBLE OF'
XYOUTS,.5,.36,/NORMAL,'!17MAGNETIC FIELD DATA.'
XYOUTS,.5,.32,/NORMAL,'!17!Sn!R!A^!N = ('+SNORM(0)+' !9+!17'+SDNORM(0)+ $
', '+SNORM(1)+' !9+ !17'+SDNORM(1)+', '+ $
SNORM(2)+' !9+!17'+SDNORM(2)+' )'

!P.MULTI(0)=0
PLOT,BXBINS,BXHIST,PSYM=10,YTITLE='!17# PTS',XTITLE='!17Bx!D1!N (NT)'
PLOT,BYBINS,BYHIST,PSYM=10,YTITLE='!17# PTS',XTITLE='!17By!D1!N (NT)'
PLOT,BZBINS,BZHIST,PSYM=10,YTITLE='!17# PTS',XTITLE='!17Bz!D1!N (NT)'
XYOUTS,.5,.4,/NORMAL,'!17HISTOGRAM OF COMPONENTS OF THE UPSTREAM MAGNETIC'
XYOUTS,.5,.38,/NORMAL,'!17FIELD DETERMINED FROM A STATISTICAL ENSEMBLE OF'
XYOUTS,.5,.36,/NORMAL,'!17MAGNETIC FIELD DATA IN GSE COORDINATES.'
XYOUTS,.5,.32,/NORMAL,'!17!SB!R!A!96!17!N = ('+SBVECT(0)+ $
' !9+!17'+SDBVECT(0)+ $
', '+SBVECT(1)+' !9+ !17'+SDBVECT(1)+', '+ $
SBVECT(2)+' !9+!17'+SDBVECT(2)+' )'

!P.MULTI(0)=0
PLOT,BBINS,BHIST,PSYM=10,YTITLE='!17# PTS',XTITLE='!17B!D1!N (NT)'
PLOT,THBNBINS*180D0/!PI,THBNHIST,PSYM=10,YTITLE='!17# PTS' $
,XTITLE='!7H!17!DBn 1!N'
XYOUTS,.5,.4,/NORMAL,'!17HISTOGRAM OF THE UPSTREAM MAGNETIC FIELD AND'+ $
' !7H!17!DBn 1!N '
XYOUTS,.5,.38,/NORMAL,'!17DETERMINED FROM A STATISTICAL ENSEMBLE OF'
XYOUTS,.5,.36,/NORMAL,'!17MAGNETIC FIELD DATA IN GSE COORDINATES.'
XYOUTS,.5,.32,/NORMAL,'!10#!17B!10#!17 = '+SB+' !9+!17'+SDB
XYOUTS,.5,.28,/NORMAL,'!7H!17!DBn 1!N = '+STHBN+' !9+!17'+SDTHBN

!P.MULTI=0
DEVICE,/CLOSE

SET_PLOT,'X'
!P.MULTI=[0,2,2]
WINDOW,0
PLOT,NXBINS,NXHIST,PSYM=10,YTITLE='!17# PTS',XTITLE='!17Nx'
PLOT,NYBINS,NYHIST,PSYM=10,YTITLE='!17# PTS',XTITLE='!17Ny'
PLOT,NZBINS,NZHIST,PSYM=10,YTITLE='!17# PTS',XTITLE='!17Nz'
XYOUTS,.5,.4,/NORMAL,'!17HISTOGRAM OF COMPONENTS OF THE SHOCK NORMAL'
XYOUTS,.5,.38,/NORMAL,'!17DETERMINED FROM A STATISTICAL ENSEMBLE OF'
XYOUTS,.5,.36,/NORMAL,'!17MAGNETIC FIELD DATA IN GSE COORDINATES.'
XYOUTS,.5,.32,/NORMAL,'!17!Sn!R!A^!N = ('+SNORM(0)+' !9+!17'+SDNORM(0)+ $
', '+SNORM(1)+' !9+ !17'+SDNORM(1)+', '+ $
SNORM(2)+' !9+!17'+SDNORM(2)+' )'

WINDOW,1
!P.MULTI(0)=0
PLOT,BXBINS,BXHIST,PSYM=10,YTITLE='!17# PTS',XTITLE='!17Bx!D1!N (NT)'
PLOT,BYBINS,BYHIST,PSYM=10,YTITLE='!17# PTS',XTITLE='!17By!D1!N (NT)'
PLOT,BZBINS,BZHIST,PSYM=10,YTITLE='!17# PTS',XTITLE='!17Bz!D1!N (NT)'
XYOUTS,.5,.4,/NORMAL,'!17HISTOGRAM OF COMPONENTS OF THE UPSTREAM MAGNETIC'
XYOUTS,.5,.38,/NORMAL,'!17FIELD DETERMINED FROM A STATISTICAL ENSEMBLE OF'
XYOUTS,.5,.36,/NORMAL,'!17MAGNETIC FIELD DATA IN GSE COORDINATES.'
XYOUTS,.5,.32,/NORMAL,'!17!SB!R!A!96!17!N = ('+SBVECT(0)+ $
' !9+!17'+SDBVECT(0)+ $
', '+SBVECT(1)+' !9+ !17'+SDBVECT(1)+', '+ $
SBVECT(2)+' !9+!17'+SDBVECT(2)+' )'

WINDOW,2
!P.MULTI(0)=0
PLOT,BBINS,BHIST,PSYM=10,YTITLE='!17# PTS',XTITLE='!17B!D1!N (NT)'
PLOT,THBNBINS*180D0/!PI,THBNHIST,PSYM=10,YTITLE='!17# PTS' $
,XTITLE='!7H!17!DBn 1!N'
XYOUTS,.5,.4,/NORMAL,'!17HISTOGRAM OF THE UPSTREAM MAGNETIC FIELD AND'+ $
' !7H!17!DBn 1!N '
XYOUTS,.5,.38,/NORMAL,'!17DETERMINED FROM A STATISTICAL ENSEMBLE OF'
XYOUTS,.5,.36,/NORMAL,'!17MAGNETIC FIELD DATA IN GSE COORDINATES.'
XYOUTS,.5,.32,/NORMAL,'!10#!17B!10#!17 = '+SB+' !9+!17'+SDB
XYOUTS,.5,.28,/NORMAL,'!7H!17!DBn 1!N = '+STHBN+' !9+!17'+SDTHBN

!P.MULTI=0

RETURN
END
;=============================================================================
PRO MOM
COMMON HISTDAT,NXHIST,NXBINS,NYHIST,NYBINS,NZHIST,NZBINS,BXHIST,BXBINS $
,BYHIST,BYBINS,BZHIST,BZBINS,BHIST,BBINS,THBNHIST,THBNBINS
COMMON MOMDAT,NORM,DNORM,BUVECT,DBUVECT,BUAVG,DBU,THBNAVG,DTHBN $
,LVECT,DLVECT,MVECT,DMVECT
COMMON REPBDAT,NX,NY,NZ,LX,LY,LZ,MX,MY,MZ,BXU,BYU,BZU,BU,THBNU

NXMOM = MOMENT(NX)
NYMOM = MOMENT(NY)
NZMOM= MOMENT(NZ)

LXMOM = MOMENT(LX)
LYMOM = MOMENT(LY)
LZMOM= MOMENT(LZ)

MXMOM = MOMENT(MX)
MYMOM = MOMENT(MY)
MZMOM= MOMENT(MZ)

BXMOM = MOMENT(BXU)
BYMOM = MOMENT(BYU)
BZMOM= MOMENT(BZU)
BUMOM= MOMENT(BU)

THBNMOM= MOMENT(THBNU)

NXAVG = NXMOM(0)
NYAVG = NYMOM(0)
NZAVG = NZMOM(0)

DNX = SQRT(NXMOM(1))
DNY = SQRT(NYMOM(1))
DNZ = SQRT(NZMOM(1))

LXAVG = LXMOM(0)
LYAVG = LYMOM(0)
LZAVG = LZMOM(0)

DLX = SQRT(LXMOM(1))
DLY = SQRT(LYMOM(1))
DLZ = SQRT(LZMOM(1))

MXAVG = MXMOM(0)
MYAVG = MYMOM(0)
MZAVG = MZMOM(0)

DMX = SQRT(MXMOM(1))
DMY = SQRT(MYMOM(1))
DMZ = SQRT(MZMOM(1))

BXAVG = BXMOM(0)
BYAVG = BYMOM(0)
BZAVG = BZMOM(0)
BUAVG = BUMOM(0)

DBX = SQRT(BXMOM(1))
DBY = SQRT(BYMOM(1))
DBZ = SQRT(BZMOM(1))
DBU = SQRT(BUMOM(1))

THBNAVG = THBNMOM(0)
DTHBN = SQRT(THBNMOM(1))

NORM = [NXAVG,NYAVG,NZAVG]
DNORM = [DNX,DNY,DNZ]

LVECT = [LXAVG,LYAVG,LZAVG]
DLVECT = [DLX,DLY,DLZ]

MVECT = [MXAVG,MYAVG,MZAVG]
DMVECT = [DMX,DMY,DMZ]

BUVECT = [BXAVG,BYAVG,BZAVG]
DBUVECT = [DBX,DBY,DBZ]

RETURN
END
;===================================================================
PRO BNERR
COMMON EIGVECT,EVAL,EVECT1,EVECT2,EVECT3,EVECT
COMMON ASYMPT,U1,U2,D1,D2,BUX,BUY,BUZ,BDX,BDY,BDZ
COMMON MOMDAT,NORM,DNORM,BUVECT,DBUVECT,BUAVG,DBU,THBNAVG,DTHBN $
,LVECT,DLVECT,MVECT,DMVECT
COMMON BNERRDAT,DBNU,DBND

BUXROT = EVECT1(0)*BUX + EVECT1(1)*BUY + EVECT1(2)*BUZ
BUYROT = EVECT2(0)*BUX + EVECT2(1)*BUY + EVECT2(2)*BUZ
BUZROT = EVECT3(0)*BUX + EVECT3(1)*BUY + EVECT3(2)*BUZ

BDXROT = EVECT1(0)*BDX + EVECT1(1)*BDY + EVECT1(2)*BDZ
BDYROT = EVECT2(0)*BDX + EVECT2(1)*BDY + EVECT2(2)*BDZ
BDZROT = EVECT3(0)*BDX + EVECT3(1)*BDY + EVECT3(2)*BDZ

BUYAVG = (MOMENT(BUYROT))(0)
BUZAVG = (MOMENT(BUZROT))(0)

BDYAVG = (MOMENT(BDYROT))(0)
BDZAVG = (MOMENT(BDZROT))(0)

NPTS = N_ELEMENTS(BUX)
DLAMBDA = EVAL(0)*SQRT(2D0/NPTS)
DBNU = SQRT(EVAL(0)/NPTS + BUYAVG^2*DLAMBDA/(EVAL(1) - EVAL(0)) $
+ BUZAVG^2*DLAMBDA/(EVAL(2) - EVAL(0)))

NPTS = N_ELEMENTS(BDX)
DLAMBDA = EVAL(0)*SQRT(2D0/NPTS)
DBND = SQRT(EVAL(0)/NPTS + BDYAVG^2*DLAMBDA/(EVAL(1) - EVAL(0)) $
+ BDZAVG^2*DLAMBDA/(EVAL(2) - EVAL(0)))

RETURN
END
;=============================================================================
PRO THETBN
COMMON MOMDAT,NORM,DNORM,BUVECT,DBUVECT,BUAVG,DBU,THBNAVG,DTHBN $
,LVECT,DLVECT,MVECT,DMVECT
COMMON THETDAT,THBN1,DTHBN1,BN1,DBN1,B1,DB1
;ALTERNATIVE CALCULATION OF THETABN
B1 = MAG(BUVECT)
DB1 = TOTAL(ABS(BUVECT*DBUVECT))/B1

BN1 = BUVECT(0)*NORM(0) + BUVECT(1)*NORM(1) + BUVECT(2)*NORM(2)

DBN1 = TOTAL(ABS(DBUVECT*NORM)) + TOTAL(ABS(BUVECT*DNORM))

THBN1 = ACOS(BN1/B1)
IF(THBN1 GT !PI/2D0) THEN THBN1 = !PI - THBN1

DB1BN1 = B1/BN1*SQRT((DBN1/BN1)^2 + (DB1/B1)^2)

DTHBN1 = DB1BN1/(B1/BN1*SQRT((B1/BN1)^2 - 1D0))

PRINT,'THBN1 = ',THBN1*180D0/!PI
PRINT,'DTHBN1 = ',DTHBN1*180D0/!PI

PRINT,'BN1 = ',BN1
PRINT,'DBN1 = ',DBN1
PRINT,'B1 = ',B1
PRINT,'DB1 = ',DB1

RETURN
END
;=============================================================================
; MAIN PROGRAM
;=============================================================================
COMMON INITDAT,FILE,T1,T2,NSEQ
COMMON TIMESELDAT,T_U1,T_U2,T_D1,T_D2,T_L1,T_L2
COMMON FIELD,NPTS,TBI,BX,BY,BZ,B
COMMON BINTVALDAT,BXU1U2,BYU1U2,BZU1U2,BXD1D2,BYD1D2,BZD1D2,BXU2L1 $
,BYU2L1,BZU2L1,BXL2D1,BYL2D1,BZL2D1,NPTS_AVG,NPTS_U2L1,NPTS_L2D1
COMMON SAMPDAT,BXSAMP,BYSAMP,BZSAMP,BXUSAMP,BYUSAMP,BZUSAMP
COMMON REPBDAT,NX,NY,NZ,LX,LY,LZ,MX,MY,MZ,BXU,BYU,BZU,BU,THBNU
COMMON EIGVECT,EVAL,EVECT1,EVECT2,EVECT3,EVECT
COMMON ERRDAT,DBN
COMMON HISTDAT,NXHIST,NXBINS,NYHIST,NYBINS,NZHIST,NZBINS,BXHIST,BXBINS $
,BYHIST,BYBINS,BZHIST,BZBINS,BHIST,BBINS,THBNHIST,THBNBINS
COMMON MOMDAT,NORM,DNORM,BUVECT,DBUVECT,BUAVG,DBU,THBNAVG,DTHBN $
,LVECT,DLVECT,MVECT,DMVECT
COMMON THETDAT,THBN1,DTHBN1,BN1,DBN1,B1,DB1

INITIALIZE
MAGDAT
PRINT,'OUT MAGDAT'
BINTVALS
REPB
HISTREPB
MOM
PLOTHIST
THETBN
END
