PRO histogram_jds, array, x, y, binsize=binsize, min=min, max=max, $
                   residual=residual, binnum=binnum
;+
; NAME:
;      HISTOGRAM_RDH
;
; PURPOSE:
;      Create a 1-D histogram and provide x,y values for plotting.
;
; CATEGORY:
;      Histogram
;
;
; CALLING SEQUENCE:
;      histogram_rdh, array, x, y, binsize=binsize, min=min, max=max, $
;                   residual=residual
;
; INPUTS:
;      array: vector to be histogrammed.
;
; OPTIONAL INPUTS:
;      None.
;
;
; KEYWORD PARAMETERS:
;      BINSIZE: The binsize for the histogram.  (Defaults to 1.)
;      MIN, MAX: The minimum and maximum values for the histogram.
;                If not set they default to the limits of the data
;                with some effort made to have reasonable bin intervals.
;      RESIDUAL: An Output array giving the number of hits outsize the
;                minimum, maximum interval: [# < MIN, # > MAX].
;      BINNUM: The vector that gives the bin number of each input point.
;
; OUTPUTS:
;      x, y: The values to plot w/psym=10
;      NOTE that an additional bucket is placed at the beginning and
;                the end of x (with y=0) to clean-up the plot.
;
; OPTIONAL OUTPUTS:
;      None.
;
; COMMON BLOCKS:
;      None.
;
; SIDE EFFECTS:
;      None.
;
; RESTRICTIONS:
;      None.
;
; PROCEDURE:
;      Histogram the array using the IDL routine histogram and then
;      prepare the x vector to reflect the bin centers.
;
; EXAMPLE:
;      histogram, array, x, y
;      plot, x, y, psym=10
;
; MODIFICATION HISTORY:
;
;      making the residual branch more informative

;      $Log: histogram_rdh.pro,v $
;      Revision 1.3  2000/10/06 19:55:23  rdh  fixed 10/7 jds
;
;      Added sanity-check for number of bins.
;
;      Revision 1.2  1999/04/09 22:37:47  rdh
;      Added binnum keyword to return the bin numbers of each input point.
;
;      Revision 1.1  1998/11/17 18:26:39  rdh
;      Initial revision
;
;-
;
IF n_params() EQ 0 THEN BEGIN
   print, 'USAGE: histogram_rdh, array, x, y, binsize=binsize, '+$
    'min=min, max=max, residual=residual'
   return
ENDIF
IF NOT keyword_set(binsize) THEN binsize = 1.0

IF n_elements(min) EQ 0 THEN min = floor(min(array)/binsize)*binsize
IF n_elements(max) EQ 0 THEN max = floor(max(array)/binsize+1)*binsize
;
;   Check number of bins
;
nbins=(max-min)/binsize
if nbins gt 2l^31-1 then begin
    binsize=(max-min)/(1000000.d0)
    message, 'Number of bins too large: Setting binsize to '+$
      strtrim(binsize,2), /cont
endif

h = histogram(array, min=min, max=max, binsize=binsize)
n = n_elements(h)+2
x = findgen(n)*binsize+min-0.5*binsize
y = lonarr(n)
y(0) = 0
y(n-1) = 0
y(1:n-2) = h

residual = replicate(0, 2)
r_hi = where(array GT max,npts_hi)
IF npts_hi gt 0 THEN begin
  residual(1) = npts_hi
  y(n-1)=npts_hi

  print, 'your histogram has highside outliers of number', npts_hi
  print, 'these represent', (1.0d2*npts_hi)/n_elements(array),'%'
  print, ' the range of these is', min(array(r_hi)), max(array(r_hi))
  endif
r_lo = where(array LT min and array gt -1.0d30,npts_lo)
IF npts_lo gt 0 THEN begin
  residual(0) = npts_lo
  y(0)=npts_lo
  print, 'your histogram has lowside outliers of number', npts_lo
  print, 'these represent', (1.0d2*npts_lo)/n_elements(array),'%'
  print, ' the range of these is', min(array(r_lo)), max(array(r_lo))
  endif
IF keyword_set(binnum) THEN BEGIN
   binnum = fix((array-x(0)+0.5*binsize)/binsize)
ENDIF



return
END
