; plot_block.pro
; 
; plots counts vs. phase angle and energy for each of 12 detectors.
;
; Keywords:
;   /gif          ; write <YYMMDD>.<HHMMSS><I|E>.gif
;   /display      ; display on CRT 
;   /ions         ; plot ions 
;   /electrons    ; plot electrons (default)
;   /batch        ; data file exists possibly containing multiple blocks
;   /buffer       ; plot whatever's in the buffer
;
; Usage:  
;    plot_block, '960713','10:00:00', /display  
;
; History:
;   7/22/96 Written, Jeremy Faden
;
; Notes:
;     How data is loaded:  
;        1) check if data file exists. 
;        2) check if data is in buffer
;        3) extract data by spawning extract program.


pro plot_block, date, timef, gif=gif, display=display,  $
                ions=ions, electrons=electrons, buffer=buffer

   common block_buffer, scan_len, species, ienergy, energy, phase, pitch, $
    counts, cdate, ctime, timeinfo

; plot specifications
   image_sizex= 11.75           ; inches
   image_sizey= 8.25            ; inches
   ppi=72                       ; pixels per inch
   
   sxplot= 2.0 *ppi             ; size in pixels
   syplot=   1.*ppi             ; pixels
   plot_space_x= 5.0  *ppi      ; spacing btw plots
   plot_space_y= 1.25 *ppi
   plot_offset_x= .75 *ppi      ; position of lower right corner
   plot_offset_y= .50 *ppi
   phase_pitch_space= .25 *ppi  ; space between phase and pitch plots

   sxbar=  .25  *ppi
   sybar=  2.   *ppi
   barposx= plot_space_x * 2 + plot_offset_x
   barposy= 0                + plot_offset_y

   sxave= 7                     ; pixels
   aveposx= phase_pitch_space / 2 - sxave / 2

   timeposx= barposx + .45*ppi 
   timeposy= 5.0*plot_space_y + plot_offset_y + syplot

   sxdetave= 10                 ; pixels
                                ; sydetave depends on number of
                                ; energies, num_en
   detaveposx= timeposx -.40*ppi

   pi=acos(-1.0)

; initialize plotting device
   set_plot, 'Z'
   device, set_resolution = [ image_sizex*ppi, image_sizey*ppi ]

; initialize color bar
   ncolors=62
   loadct_hydra, 18             ; gsfc color bar
   multi, 256/float(ncolors+1)
   tvlct, color_r, color_g, color_b, /get

; set non-color-bar colors
   white=ncolors
   color_r(white)=255
   color_g(white)=255
   color_b(white)=255
   black=0
   color_r(black)=0
   color_g(black)=0
   color_b(black)=0
   grey=ncolors+1
   color_r(grey)=120
   color_g(grey)=120
   color_b(grey)=120
   pink=ncolors+3
   color_r(pink)=180
   color_g(pink)=140
   color_b(pink)=140

   tvlct, color_r, color_g, color_b
   green= fix ( ncolors * (.50) )

   if keyword_set(buffer) then goto, plot

   time=strmid(timef,0,2)+strmid(timef,3,2)+strmid(timef,6,2)
   if (keyword_set(batch)) then begin
      unit=10
      openr, unit, date+'.'+time+'.dat'
   endif else begin
      if (keyword_set(cdate)) then begin ; a block is in memory
         if ((cdate eq date) and (ctime eq time)) then goto, plot
      endif 
      print, 'extracting data...'
      spawn, "/home/jbf/work/idl_util/block/extract "+date+" "+timef+" 0" 
                                ;['extract',date,time,'0'], unit=unit
      unit=10
      openr, unit, date+'.'+time+'.dat'
   endelse

   libmask=string(30)
   readf, unit, libmask         ; read in library and mask

readdata:
                                ; read one block from the data file 
   readf, unit, scan_len
   timeinfo='timeinfo is a string'
   readf, unit, timeinfo
   species=intarr(scan_len)
   energy=fltarr(scan_len)
   ienergy=intarr(scan_len)
   phase= fltarr(scan_len)
   counts= intarr(scan_len,12)
   pitch= fltarr(scan_len,12)
   tcounts= intarr(12)
   tpitch= fltarr(12)
   for i=0,scan_len-1 do begin
      readf, unit, tspecies, tienergy, tenergy, tphase, tcounts, tpitch
      species(i)=tspecies
      energy(i)=tenergy
      ienergy(i)=abs(tienergy)
      phase(i)=tphase
      counts(i,*)=tcounts
      pitch(i,*)=tpitch
   endfor
   phase=phase*180.0/pi
   pitch=pitch*180.0/pi

   cdate=date
   ctime=time

plot:  
                                ; now plot the one block  
   s=-1                         ;
   if ( keyword_set(ions) and not keyword_set(electrons) ) then s=1 
specloop:             
   if (s eq -1) then sp='Ele' else sp='Ions'

  if ( s gt 0 ) then wspecies = where(species gt 0) else $
     wspecies = where(species lt 0) 
  num_en= max( ienergy(wspecies) )  
  sydetave = 4*num_en  
  detaveposy = timeposy-sydetave

  if ( s eq -1) then  max_co= 500.  else  max_co= 100.
  min_co= 0.1

 ; plot the color bar
  bar=congrid( bindgen(1,ncolors), sxbar,sybar )
  ytickv =  [0.1, 1.0, 10.0, 100.0 ] 
  IF max_co EQ 500. THEN ytickv = [0.1, 1.0, 10.0, 100.0, 500.0 ]
  yticks = n_elements(ytickv)-1
  plot, bar, /nodata, /device, yrange=[ min_co,max_co ], /ylog,   $
      position=[ barposx, barposy, barposx+sxbar, barposy+sybar ], $
      xticks=1, xtickname= [" "," "], color=white, xstyle=1, ystyle=1, $
      title='counts', ytickv=ytickv, yticks=yticks, yticklen=-0.15, $
      yminor=9
  tv, bar(1:*,1:*), barposx+1, barposy+1

  ; print time, date, and species
  xyouts, timeposx, timeposy, strmid(timeinfo,6,8) + "  " +       $
      strmid(timeinfo,22,8) +"  "+sp, orientation=-90.,  $
      color=white, /device, size=1.3

  ; print psi status 
  psiposy= 2* plot_space_y + plot_offset_y + syplot
  if strmid(timeinfo,38,1) eq '1'  then    $
      xyouts, timeposx, psiposy, "PSI ON", orientation=-90.0,      $
         size=1.3, color=green, /device                           $
  else                                                             $
      xyouts, timeposx, psiposy, "PSI OFF", orientation=-90.0,     $
         size=1.3, color=white, /device                           

  ; print mode
  mode= fix( strmid(timeinfo,46,2) )
  if (mode eq 10) then modestr='3(1e1p)3(2e2p)' $
  else if (mode eq 22) then modestr='6(3e3p)' $
  else if (mode eq 23) then modestr='6(4e4p)' $
  else modestr=' '
  xyouts, detaveposx, psiposy, modestr, orientation=-90.0, size=1.0,  $
      color=white, /device
       
  tvphase= bytarr( sxplot, syplot, 12 )
  tvpitch= bytarr( sxplot, syplot, 12 )
  ave= intarr( num_en+1, 12 )         ; average count accumulator
  ave_c= intarr( num_en+1, 12 )       ; number of samples in average  

  tvphase( *,*,* ) = black
  tvpitch( *,*,* ) = black

 ; energy_index -> energy map
  energies=fltarr(110)
  FOR i=0, scan_len-1 DO BEGIN
     energies(ienergy(i))=energy(i)
  ENDFOR 

 ; set the energy labels
  ytickv_energy = long(indgen(4)*((num_en-1)/3.0))+1
  ytickname_energy = strtrim(fix(energies(ytickv_energy)),1)
  yticks_energy = 3

 ; define the common plot parameters for 24 energy plots
  common_par_en = { nodata:1, device:1, noerase:1, color:white,              $
                    yrange:[0,num_en], xstyle:1, ystyle:1,                   $
                    xticks:4, yticks:yticks_energy, ytickv:ytickv_energy }

  for j=0,11 do begin
      ave( *,j ) = 0.0
      ave_c( *,j) = 0

      for i=0,scan_len-1 do begin    ; now fill in boxes with data
	 if ( species(i)*s gt 0 ) then begin   ; sign(species(i))=sign(s)
            y=block_linscale( syplot, 0, num_en, ienergy(i) )
            tienergy = ienergy(i)
            xphase=block_linscale( sxplot, 0, 360, phase(i) ) 
            xpitch=block_linscale( sxplot, 0, 180, pitch(i,j) ) 
            if ( counts(i,j) gt 0 ) then begin
               z=block_logscale( ncolors-1, min_co, max_co, counts(i,j) ) 
               block_put_point, tvpitch,j,xpitch,y,z, sxplot, syplot
	       block_put_point, tvphase,j,xphase,y,z, sxplot, syplot
               ave(tienergy,j)= ave(tienergy,j)+ counts(i,j) 
               ave_c(tienergy,j)= ave_c(tienergy,j) + 1
            endif else if ( counts(i,j) eq 0 ) then begin
               block_put_point, tvpitch,j,xpitch,y, pink, sxplot, syplot
               block_put_point, tvphase,j,xphase,y, pink, sxplot, syplot
               ave_c(tienergy,j)= ave_c(tienergy,j) + 1
            endif else begin
               block_put_point, tvpitch,j,xpitch,y, grey, sxplot, syplot
	       block_put_point, tvphase,j,xphase,y, grey, sxplot, syplot
            endelse
         endif
      endfor

     ; calculate averages
      tvave= bytarr( sxave, syplot, 12 )
      tvave( *,*,j ) = black
      
      for en= 1,num_en do begin
         if ( ave_c(en,j) gt 0 ) then begin
            ave_counts= float(ave(en,j))/ave_c(en,j) 
            z = block_logscale( ncolors-1, min_co, max_co, ave_counts ) 
            y = block_linscale( syplot, 0, num_en, en )
            block_put_point, tvave, j, sxave*1/3, y, z, sxave, syplot 
            block_put_point, tvave, j, sxave*2/3+1, y, z, sxave, syplot
         endif
      endfor

      plotposy= ( 5 - j/2 ) * plot_space_y + plot_offset_y

     ; plot phase
      plotposx= (j mod 2) * plot_space_x + plot_offset_x
      tv, tvphase(*,*,j), plotposx, plotposy
      if  j lt 2  then title='s/c phase' else title=' '
      if (j mod 2) eq 0 then $      
         plot, tvphase(*,*,j), _extra=common_par_en, xrange=[0,360],          $
         position=[ plotposx, plotposy, plotposx+sxplot, plotposy+syplot ],   $
         xtickname=[' ',' ',' ',' ',' '], ytickname=[' ',' ',' ',' '],        $
         ytitle='Detector '+strtrim(string(j+1),1), title=title               $
      else                                                                    $
         plot, tvphase(*,*,j), _extra=common_par_en, xrange=[0,360],          $
         position=[ plotposx, plotposy, plotposx+sxplot, plotposy+syplot ],   $
         title=title, ytickname=ytickname_energy,                             $
         xtickname=[' ',' ',' ',' ',' ']        
      if  j gt 9  then        $       ; put labels on bottom plots
         plot, tvphase(*,*,j), _extra=common_par_en, xrange=[0,360],          $
         position=[ plotposx, plotposy, plotposx+sxplot, plotposy+syplot ],   $
         ytickname=[' ',' ',' ',' ']
      
     ; plot pitch
      if  j lt 2  then title='pitch angle' else title=' '
      plotposx= (j mod 2) * plot_space_x + $ 
         sxplot+ phase_pitch_space + plot_offset_x
      tv, tvpitch(*,*,j), plotposx, plotposy
      plot, tvpitch(*,*,j), _extra=common_par_en, xrange=[0,180],             $
         position=[ plotposx, plotposy, plotposx+sxplot, plotposy+syplot ],   $
         ytickname=[' ',' ',' ',' '], xtickname=[' ',' ',' ',' ',' '],        $
         title=title 
      if  j gt 9  then         $
         plot, tvpitch(*,*,j), _extra=common_par_en, xrange=[0,180],          $
         position=[ plotposx, plotposy, plotposx+sxplot, plotposy+syplot ],   $
         ytickname=[' ',' ',' ',' ']

     ; plot average
      plotposx= (j mod 2) * plot_space_x + sxplot + aveposx + plot_offset_x
      plot, tvave( *,*,j ), /nodata, /device, /noerase, color=white,     $
         position=[ plotposx, plotposy, plotposx+sxave, plotposy+syplot ],   $
         xticks=1, xtickname=[' ',' '], yticks=1, ytickname=[' ',' '],      $
         yrange=[0,10], xrange=[0,10]
      tv, tvave( 1:*, 1:*, j ), plotposx+1, plotposy+1
  endfor   

 ; calculate average over all detectors
  detave= lonarr( num_en+1 )      
  detave_c= intarr ( num_en+1 )
  tvdetave= bytarr( sxdetave, sydetave )

  FOR en= 1, num_en DO BEGIN 
     detave(en)= total( ave(en,*) )
     detave_c(en)= total( ave_c(en,*) )
     ave_counts= float( detave(en) ) / detave_c(en) 
     z= block_logscale( ncolors-1, min_co, max_co, ave_counts )   
     y = block_linscale( sydetave, 0, num_en, en )
     ly= (y-4 > 0)
     uy= (y < sydetave-1)
     tvdetave( *, ly:uy ) = z
  ENDFOR

 ; plot average over all detectors
  plot, tvdetave, /nodata, /device, /noerase, color=white,               $
      position=[ detaveposx, detaveposy,                                 $
             detaveposx+sxdetave, detaveposy+sydetave ],                 $
      xticks=1, xtickname=[' ',' '], yticklen=-0.15,                     $
      yrange=[0,num_en],    $
      yticks=yticks_energy, ytickname=ytickname_energy, ytickv=ytickv_energy,   $
      title='<Ci>'
  tv, tvdetave( 1:*, 1:* ), detaveposx+1, detaveposy+1

  d= tvrd()
  if keyword_set(gif) then begin   
      fname=   $
       strmid(timeinfo,28,2)+strmid(timeinfo,22,2)+strmid(timeinfo,25,2)+'.'+ $
       strmid(timeinfo,6,2) +strmid(timeinfo,9,2) +strmid(timeinfo,12,2) +  $
       strmid(sp,0,1) + '.gif'         
      write_gif, fname, d, color_r, color_g, color_b
  endif

  if keyword_set(display) then begin
      set_plot, 'x'
      tvlct, color_r, color_g, color_b
      window, 3, xsize=image_sizex*ppi, ysize=image_sizey*ppi
      tv, d
  endif

  if ( (s eq -1) and (keyword_set(ions)) ) then begin
      s=1
      wait, 1
      goto, specloop
  endif

if (keyword_set(batch)) then if (not eof(unit)) then goto, readdata 

if (keyword_set(batch)) then delvar, cdate               

if (keyword_set(unit)) then begin
  close, unit
  if (not keyword_set(batch)) then spawn, 'rm '+date+'.'+time+'.dat'  
endif

END
