; plot_block.pro
; 
; plots counts vs. phase angle and energy for each of 12 detectors.
;
; Keywords:
;   /gif
;   /display
;   /ions         ; plot ions 
;   /electrons    ; plot electrons (default)
;   /batch        ; data file exists possibly containing multiple blocks
;   /buffer       ; plot whatever's in the buffer
;
; Usage:  
;    .comp plot_block                           
;    plot_block, '960713','10:00:00', /display  
;
; History:
;   7/22/96 Written, Jeremy Faden
;
; Notes:
;     How data is loaded:  
;        1) check if data file exists. 
;        2) check if data is in buffer
;        3) extract data by spawning extract program.


pro plot_block, date, timef, gif=gif, display=display,  $
  ions=ions, electrons=electrons, buffer=buffer

common block_buffer, scan_len, species, energy, phase, counts, pitch, $
  cdate, ctime, timeinfo
   
; plot specifications
image_sizex= 11.75         ; inches
image_sizey= 8.25         ; inches
ppi=72                   ; pixels per inch
 
sxplot= 2.0 *ppi     ; size in pixels
syplot=   1.*ppi     ; pixels
plot_space_x= 5.0  *ppi  ; spacing btw plots
plot_space_y= 1.25 *ppi
plot_offset_x= .75 *ppi  ; position of lower right corner
plot_offset_y= .50 *ppi
phase_pitch_space= .25 *ppi  ; space between phase and pitch plots

sxbar=  .25  *ppi
sybar=  2.   *ppi
barposx= plot_space_x * 2 + plot_offset_x
barposy= 0                + plot_offset_y

sxave= 7 ; pixels
aveposx= phase_pitch_space / 2 - sxave / 2

timeposx= barposx + .45*ppi 
timeposy= 5.0*plot_space_y + plot_offset_y + syplot

sxdetave= 10 ; pixels
sydetave= syplot*3
detaveposx= timeposx -.40*ppi
detaveposy= timeposy - sydetave

pi=acos(-1.0)

; initialize plotting device
set_plot, 'Z'
device, set_resolution = [ image_sizex*ppi, image_sizey*ppi ]

; initialize color bar
  ncolors=128
  loadct_hydra, 18  ; gsfc color bar
  multi, 256/float(ncolors+1)
  tvlct, color_r, color_g, color_b, /get

; set non-color-bar colors
  black=0
  color_r(black)=0
  color_g(black)=0
  color_b(black)=0
  grey=1
  color_r(grey)=120
  color_g(grey)=120
  color_b(grey)=120
  white=2
  color_r(white)=255
  color_g(white)=255
  color_b(white)=255
  pink=3
  color_r(pink)=180
  color_g(pink)=140
  color_b(pink)=140

  tvlct, color_r, color_g, color_b
  green= fix ( ncolors * (.50) )

  if keyword_set(buffer) then goto, plot

  time=strmid(timef,0,2)+strmid(timef,3,2)+strmid(timef,6,2)
  if (keyword_set(batch)) then begin
     unit=10
     openr, unit, date+'.'+time+'.dat'
  endif else begin
     if (keyword_set(cdate)) then begin     ; a block is in memory
	if ((cdate eq date) and (ctime eq time)) then goto, plot
     endif 
     print, 'extracting data...'
     spawn, "/home/jbf/work/idl_util/block/extract "+date+" "+timef+" 0" 
                          ;['extract',date,time,'0'], unit=unit
     unit=10
     openr, unit, date+'.'+time+'.dat'
  endelse

  libmask=string(30)
  readf, unit, libmask              ; read in library and mask

readdata:
 ; read one block from the data file 
  readf, unit, scan_len
  timeinfo='timeinfo is a string'
  readf, unit, timeinfo
  species=intarr(scan_len)
  energy=fltarr(scan_len)
  phase= fltarr(scan_len)
  counts= intarr(scan_len,12)
  pitch= fltarr(scan_len,12)
  tcounts= intarr(12)
  tpitch= fltarr(12)
  for i=0,scan_len-1 do begin
     readf, unit, tspecies, tenergy, tphase, tcounts, tpitch
     species(i)=tspecies
     energy(i)=tenergy
     phase(i)=tphase
     counts(i,*)=tcounts
     pitch(i,*)=tpitch
  endfor
  phase=phase*180.0/pi
  pitch=pitch*180.0/pi

  cdate=date
  ctime=time

plot:  
 ; find the energy levels
 ; ienergy=

 ; now plot the one block  
  s=-1;
  if ( keyword_set(ions) and not keyword_set(electrons) ) then s=1 
specloop:             
  if (s eq -1) then sp='Ele' else sp='Ions'

  if ( s gt 0 ) then wspecies = where(species gt 0) else $
     wspecies = where(species lt 0) 
  max_en= max( energy(wspecies) )  
  min_en= 1.0

  if ( s eq -1) then  max_co= 500.  else  max_co= 100.
  min_co= 0.1

 ; plot the color bar
  bar=congrid( bindgen(1,ncolors-4)+3, sxbar,sybar )
  plot, bar, /nodata, /device, yrange=[ min_co,max_co ], /ylog,   $
      position=[ barposx, barposy, barposx+sxbar, barposy+sybar ], $
      xticks=1, xtickname= [" "," "], color=white, xstyle=1, ystyle=1, $
      title='counts', yticks=4
  tv, bar(1:*,1:*), barposx+1, barposy+1

  ; print time, date, and species
  xyouts, timeposx, timeposy, strmid(timeinfo,6,8) + "  " +       $
      strmid(timeinfo,22,8) +"  "+sp, orientation=-90.,  $
      color=white, /device, size=1.3

  ; print psi status 
  psiposy= 2* plot_space_y + plot_offset_y + syplot
  if strmid(timeinfo,38,1) eq '1'  then    $
      xyouts, timeposx, psiposy, "PSI ON", orientation=-90.0,      $
         size=1.3, color=green, /device                           $
  else                                                             $
      xyouts, timeposx, psiposy, "PSI OFF", orientation=-90.0,     $
         size=1.3, color=white, /device                           

  ; print mode
  mode= fix( strmid(timeinfo,46,2) )
  if (mode eq 10) then modestr='3(1e1p)' $
  else if (mode eq 22) then modestr='6(3e3p)' $
  else if (mode eq 23) then modestr='6(4e4p)' $
  else modestr=' '
  xyouts, detaveposx, psiposy, modestr, orientation=-90.0, size=1.3,  $
      color=white, /device
       
  tvphase= bytarr( sxplot, syplot, 12 )
  tvpitch= bytarr( sxplot, syplot, 12 )
  ave= intarr( syplot, 12 )         ; average count accumulator
  ave_c= bytarr( syplot, 12 )       ; number of samples in average  

  tvphase( *,*,* ) = black
  tvpitch( *,*,* ) = black

  for j=0,11 do begin
      ave( *,j ) = 0.0
      ave_c( *,j) = 0

      for i=0,scan_len-1 do begin    ; now fill in boxes with data
	 if ( species(i)*s gt 0 ) then begin   ; sign(species(i))=sign(s)
            y=block_logscale( syplot, min_en, max_en, energy(i) )
            xphase=block_linscale( sxplot, 0, 360, phase(i) ) 
            xpitch=block_linscale( sxplot, 0, 180, pitch(i,j) ) 
            if ( counts(i,j) gt 0 ) then begin
               z=block_logscale( ncolors-5, min_co, max_co, counts(i,j) ) + 4
               block_put_point, tvpitch,j,xpitch,y,z, sxplot, syplot
	       block_put_point, tvphase,j,xphase,y,z, sxplot, syplot
               ave(y,j)= ave(y,j)+ counts(i,j) 
               ave_c(y,j)= ave_c(y,j) + 1
            endif else if ( counts(i,j) eq 0 ) then begin
               block_put_point, tvpitch,j,xpitch,y,pink, sxplot, syplot 
               block_put_point, tvphase,j,xphase,y,pink, sxplot, syplot
               ave_c(y,j)= ave_c(y,j) + 1
            endif else begin
               block_put_point, tvpitch,j,xpitch,y,grey, sxplot, syplot
	       block_put_point, tvphase,j,xphase,y,grey, sxplot, syplot
            endelse
         endif
      endfor

     ; calculate averages
      tvave= bytarr( sxave, syplot, 12 )
      tvave( *,*,j ) = black
      
      for y= 0,syplot-1 do begin
         if ( ave_c(y,j) gt 0 ) then begin
            if ( ave(y,j) eq 0 ) then z=pink else begin
               ave_counts= float(ave(y,j))/ave_c(y,j) 
               z= block_logscale( ncolors-5, min_co, max_co, ave_counts ) + 4
            endelse
            block_put_point, tvave, j, sxave*1/3, y, z, sxave, syplot 
            block_put_point, tvave, j, sxave*2/3, y, z, sxave, syplot 	
         endif
      endfor

      plotposy= ( 5 - j/2 ) * plot_space_y + plot_offset_y

     ; plot phase
      plotposx= (j mod 2) * plot_space_x + plot_offset_x
      tv, tvphase(*,*,j), plotposx, plotposy
      if  j lt 2  then title='s/c phase' else title=' '
      if (j mod 2) eq 0 then $      
         plot, tvphase(*,*,j), /nodata, /device, /noerase,                    $
         xrange=[0,360], yrange=[min_en, max_en], /ylog,                      $
         position=[ plotposx, plotposy, plotposx+sxplot, plotposy+syplot ],   $
         color=white, xstyle=1, ystyle=1, xticks=4, yticks=1,                 $
         xtickname=[' ',' ',' ',' ',' '], ytickname=[' ',' '],                $
         ytitle='Detector '+strtrim(string(j+1),1), title=title               $
      else                                                                    $
         plot, tvphase(*,*,j), /nodata, /device, /noerase,                    $
         xrange=[0,360], yrange=[min_en, max_en], /ylog,                      $
         position=[ plotposx, plotposy, plotposx+sxplot, plotposy+syplot ],   $
         color=white, xstyle=1, ystyle=1, title=title,                        $
         xticks=4, xtickname=[' ',' ',' ',' ',' ']        
      if  j gt 9  then        $       ; put labels on bottom plots
         plot, tvphase(*,*,j), /nodata, /device, /noerase,                    $
         xrange=[0,360], yrange=[min_en, max_en], /ylog,                      $
         position=[ plotposx, plotposy, plotposx+sxplot, plotposy+syplot ],   $
         color=white, xstyle=1, ystyle=1, yticks=1, ytickname=[' ',' '],      $
         xticks=4
      
     ; plot pitch
      if  j lt 2  then title='pitch angle' else title=' '
      plotposx= (j mod 2) * plot_space_x + $ 
         sxplot+ phase_pitch_space + plot_offset_x
      tv, tvpitch(*,*,j), plotposx, plotposy
      plot, tvpitch(*,*,j), /nodata, /device, /noerase,                       $
         xrange=[0,180], yrange=[min_en, max_en], /ylog,                      $
         position=[ plotposx, plotposy, plotposx+sxplot, plotposy+syplot ],   $
         color=white, xstyle=1, ystyle=1, xticks=4, title=title,              $
         yticks=1, ytickname=[' ',' '], xtickname=[' ',' ',' ',' ',' ']
      if  j gt 9  then         $
         plot, tvpitch(*,*,j), /nodata, /device, /noerase,                    $
         xrange=[0,180], yrange=[min_en, max_en], /ylog,                      $
         position=[ plotposx, plotposy, plotposx+sxplot, plotposy+syplot ],   $
         color=white, xstyle=1, ystyle=1, xticks=4,                           $
         yticks=1, ytickname=[' ',' ']

     ; plot average
      plotposx= (j mod 2) * plot_space_x + sxplot + aveposx + plot_offset_x
      plot, tvave( *,*,j ), /nodata, /device, /noerase, color=white,     $
         position=[ plotposx, plotposy, plotposx+sxave, plotposy+syplot ],   $
         xticks=1, xtickname=[' ',' '], yticks=1, ytickname=[' ',' ']
      tv, tvave( 1:*, 1:*, j ), plotposx+1, plotposy+1
  endfor   

 ; calculate average over all detectors
  detave= intarr( syplot )      ; MAJOR PROBLEM: each energy level
                                  ; needs own pixel!
  detave_c= intarr ( syplot )
  tvdetave= bytarr( sxdetave, sydetave )

  for y= 0, syplot-1 do begin
      if ( total( ave_c(y,*) ) gt 0 ) then begin
         detave(y)= total( ave(y,*) )
         detave_c(y)= total( ave_c(y,*) )
         ave_counts= float( detave(y) ) / detave_c(y) 
	 if ( ave_counts gt 0.0 ) then   $
            z= block_logscale( ncolors-5, min_co, max_co, ave_counts ) + 4  $
         else  $
            z= pink  
         ly= (y*3-3 > 0)
         uy= (y*3+3 < sydetave-1)
	 tvdetave( *, ly:uy ) = z
      endif
  endfor
 ; plot average over all detectors
  plot, tvdetave, /nodata, /device, /noerase, color=white,            $
      position=[ detaveposx, detaveposy,                              $
             detaveposx+sxdetave, detaveposy+sydetave ],              $
      xticks=1, xtickname=[' ',' '],                                  $
      yrange=[min_en, max_en], /ylog, ystyle=1,                       $
      title='<Ci>'
  tv, tvdetave( 1:*, 1:* ), detaveposx+1, detaveposy+1
   
  d= tvrd()
  if keyword_set(gif) then begin   
      fname=   $
       '/net/hydra/export/data1/ns-home/docs/archive/plots/ddeis/special/'+ $
       strmid(datetime,0,6) + '.' +  $
       strmid(timeinfo,6,2) + strmid(timeinfo,9,2) + strmid(timeinfo,12,2) +  $
       strmid(sp,0,1) + '.gif'         
      write_gif, fname, d, color_r, color_g, color_b
  endif

  if keyword_set(display) then begin
      set_plot, 'x'
      tvlct, color_r, color_g, color_b
      window, 3, xsize=image_sizex*ppi, ysize=image_sizey*ppi
      tv, d
  endif

  if ( (s eq -1) and (keyword_set(ions)) ) then begin
      s=1
      wait, 1
      goto, specloop
  endif

if (keyword_set(batch)) then if (not eof(unit)) then goto, readdata 

if (keyword_set(batch)) then delvar, cdate               

if (keyword_set(unit)) then begin
  close, unit
  if (not keyword_set(batch)) then spawn, 'rm '+date+'.'+time+'.dat'  
endif

END
