; plot_beam.pro
; 
; plots counts vs. phase angle and energy for each of 12 detectors.
;
; History:
;   7/22/96 Written, Jeremy Faden

pro plot_beam, pitch=k_pitch, time=k_time, gif=gif, ps=ps, display=display 

date='960713'

forward_function logscale
forward_function linscale

; plot specifications
image_sizex= 11.0         ; inches
image_sizey= 11.0         ; inches
ppi=72                   ; pixels per inch
 
sxplot= 3.5 *ppi     ; size in pixels
syplot=   1.*ppi     ; pixels
plot_space_x= 4.5  *ppi  ; spacing btw plots
plot_space_y= 1.75  *ppi
plot_offset_x=1.00 *ppi  ; position of lower right corner
plot_offset_y= .50 *ppi

sxbar=  .25  *ppi
sybar=  2.   *ppi
barposx= plot_space_x*2 + plot_offset_x
barposy= 0 +              plot_offset_y

timeposx= barposx
timeposy= 5.0*plot_space_y + plot_offset_y

pi=acos(-1.0)

openr, 10, date+'beam.dat'

libmask=string(30)
readf, 10, libmask              ; read in library and mask

; initialize plotting device
set_plot, 'Z'
device, set_resolution = [ image_sizex*ppi, image_sizey*ppi ]

; initialize color bar
  ncolors=128
  loadct, 39, ncolors=ncolors
  tvlct, color_r, color_g, color_b, /get

;   crop off black part at bottom
  cropamt=fix(ncolors/7)  ;  that's about deep blue
  color_r= congrid( color_r( cropamt:ncolors ), ncolors, /interp )
  color_g= congrid( color_g( cropamt:ncolors ), ncolors, /interp )
  color_b= congrid( color_b( cropamt:ncolors ), ncolors, /interp )

;   set non-color-bar colors
  black=0
  color_r(black)=0
  color_g(black)=0
  color_b(black)=0
  grey=1
  color_r(grey)=120
  color_g(grey)=120
  color_b(grey)=120
  white=2
  color_r(white)=255
  color_g(white)=255
  color_b(white)=255
  tvlct, color_r, color_g, color_b


block=1
while (not eof(10)) do begin
  ; read one block from the data file 
   readf, 10, scan_len
   time='time is a string'
   readf, 10, time
   species=intarr(scan_len)
   energy=fltarr(scan_len)
   phase= fltarr(scan_len)
   counts= intarr(scan_len,12)
   pitch= fltarr(scan_len,12)
   tcounts= intarr(12)
   tpitch= fltarr(12)
   for i=0,scan_len-1 do begin
      readf, 10, tspecies, tenergy, tphase, tcounts, tpitch
      species(i)=tspecies
      energy(i)=tenergy
      phase(i)=tphase
      counts(i,*)=tcounts
      pitch(i,*)=tpitch
   endfor
   phase=phase*180.0/pi
   pitch=pitch*180.0/pi

  ; now plot the one block  
   s=-1 ;  electrons
specloop:             
   if (s eq -1) then sp='e' else sp='p'

   tvplot= bytarr( sxplot, syplot, 12 )

   if ( s gt 0 ) then wspecies = where(species gt 0) else $
      wspecies = where(species lt 0) 
   max_en= max( energy(wspecies) )  
   min_en= 1.0
   max_co= max( counts(wspecies,*) )  
   min_co= 1

  ; plot the color bar
   bar=congrid( bindgen(1,ncolors-4)+3, sxbar,sybar )
   plot, bar, /nodata, /device, yrange=[ min_co,max_co ], /ylog,   $
      position=[ barposx, barposy, barposx+sxbar, barposy+sybar ], $
      xticks=1, xtickname= [" "," "], color=white, xstyle=1, ystyle=1, $
      ytitle='counts', yticks=4
   tv, bar(1:*,1:*), barposx+1, barposy+1

   xyouts, timeposx, timeposy, time+"  Species: "+sp, orientation=-90., $
       color=white, /device

   for j=0,11 do begin
      tvplot( *,*,j ) = black
      for i=0,scan_len-1 do begin    ; now fill in boxes with data
	 if ( species(i)*s gt 0 ) then begin
            y=logscale( syplot, min_en, max_en, energy(i) )
            if keyword_set(k_pitch) then begin
               x=linscale( sxplot, 0, 180, pitch(i,j) ) 
	       xrange=[0,180]
               xtitle='pitch angle'
            endif $
            else if keyword_set(k_time) then begin
               x=linscale( sxplot, 0, scan_len-1, i ) 
	       xrange=[0,scan_len]
               xtitle='scan sequence'
            endif $
            else begin
               x=linscale( sxplot, 0, 360, phase(i) ) 
               xrange=[0,360]
               xtitle='s/c phase'
            endelse
            if ( counts(i,j) gt 0 ) then begin
               z=logscale( ncolors-4, min_co, max_co, counts(i,j) ) + 3
	       if ( counts(i,j) gt max_co ) then print,j,z, counts(i,j), max_co
               put_point, tvplot,j,x,y,z, sxplot, syplot
            endif else if ( counts(i,j) eq 0 ) then              $
               put_point, tvplot,j,x,y,3, sxplot, syplot         $
            else                                                 $
               put_point, tvplot,j,x,y,grey, sxplot, syplot
         endif
      endfor
      plotposx= (j mod 2) * plot_space_x + plot_offset_x
      plotposy= ( 5 - j/2 ) * plot_space_y + plot_offset_y

      tv, tvplot(*,*,j), plotposx, plotposy
      plot, tvplot(*,*,j), /nodata, /device, /noerase,                       $
         xrange=xrange, yrange=[min_en, max_en], /ylog,                      $
         position=[ plotposx, plotposy, plotposx+sxplot, plotposy+syplot ],  $
         color=white, xstyle=1, ystyle=1, xticks=4, xtitle=xtitle,           $
         ytitle='Energy, eV',  title='Detector '+strtrim(string(j+1),1)
   endfor   

   d= tvrd()
   if keyword_set(gif) then $      
      write_gif, 'block'+strtrim(string(block),1)+sp+'.gif', $
         d, color_r, color_g, color_b

   if keyword_set(display) then begin
      set_plot, 'x'
      tvlct, color_r, color_g, color_b
      window, 3, xsize=image_sizex*ppi, ysize=image_sizey*ppi
      tv, d
   endif

   if (s eq -1) then begin
      s=1
      goto, specloop
   endif

   block=block+1
endwhile
   close, 10

END


pro put_point, tvplot, j, x,y,z, sxplot, syplot
   lx= (x-1) > 0
   ux= (x+1) < fix(sxplot-1)
   ly= (y-1) > 0
   uy= (y+1) < fix(syplot-1)
   tvplot( lx:ux, ly:uy, j ) = z
end   

function logscale, length, min, max, val
 ; log scale returns int
  return, fix( (length-1) * alog10(val/min) / alog10(max/min) )
end

function linscale, length, min, max, val
 ; lin scale returns int
  return, fix( (length-1) * ( val-min ) / ( max-min ) )
end

