; plot_block.pro
; 
; plots counts vs. phase angle and energy for each of 12 detectors.
;
; Keywords:
;   /gif          ; write <YYMMDD>.<HHMMSS><I|E>.gif
;   /display      ; display on CRT 
;   /ions         ; plot ions 
;   /electrons    ; plot electrons (default)
;   /batch        ; data file exists possibly containing multiple blocks
;   /extend       ; use the extended color bar
;   /preliminary  ; print 'Preliminary' across the plot
;
;   /next         ; go to next block
;   /prev         ; go to prev block

; Usage:  
;    plot_block, '960713','10:00:00', /display  
;
; History:
;   7/22/96 plot_block Written, Jeremy Faden
;   4/10/96 Reads Level1 source, called plot_block_l1 
;
; Notes:

function block_linscale, length, min, max, val
 ; lin scale returns int
  return, fix( (length-1) * ( val-min ) / ( max-min ) )
end

function block_logscale, length, min, max, val
 ; log scale returns int
  if ( val lt min ) then return, 0         ; logscale is getting gross, huh?
  if ( val gt max ) then return, length+1  ; saturated
  return, fix( (length) * alog10(val/min) / alog10(max/min) )
end

pro block_put_point, tvplot, j, x,y,z, sxplot, syplot
   lx= (x-1) > 0
   ux= (x+1) < fix(sxplot-1)
   ly= (y-1) > 0
   uy= (y+1) < fix(syplot-1)
   tvplot( lx:ux, ly:uy, j ) = z
end   

pro bt_load_colors
   
   common blocktool_colors, ncolors, background, foreground, $
    black, white, gray, yellow, blue, red, color_r, color_g, color_b

   loadct_hydra, 18             ; gsfc
   multi,!d.n_colors/float(ncolors) ; 0 thru n_colors-1 is color bar
   tvlct, color_r, color_g, color_b, /get
   
   black = 0
   white = ncolors+0
   gray = ncolors+1
   yellow = ncolors+2
   blue = ncolors+3
   red = ncolors+4
   background = ncolors+5
   foreground = ncolors+6
   
   !p.background = background

   pcl = 0
   ps = 0

   if (pcl EQ 1 OR ps EQ 1) then begin
      color_r(background) = 255
      color_g(background) = 255
      color_b(background) = 255
      
      color_r(foreground) = 0
      color_g(foreground) = 0
      color_b(foreground) = 0
      
      color_r(blue) = 0
      color_g(blue) = 0
      color_b(blue) = 255
      
      orbit1 = blue
      orbit2 = red

   endif else begin
      color_r(background) = 0
      color_g(background) = 0
      color_b(background) = 0
      
      color_r(foreground) = 255
      color_g(foreground) = 255
      color_b(foreground) = 255
      
      color_r(blue) = 91
      color_g(blue) = 240
      color_b(blue) = 255

   endelse
   
   color_r(black) = 0
   color_g(black) = 0
   color_b(black) = 0
   
   color_r(gray) = 140
   color_g(gray) = 140
   color_b(gray) = 140
   
   color_r(white) = 255
   color_g(white) = 255
   color_b(white) = 255
   
   color_r(red) = 255
   color_g(red) = 0
   color_b(red) = 0

   color_r(yellow) = 255
   color_g(yellow) = 255
   color_b(yellow) = 0
      
   tvlct, color_r, color_g, color_b
   
   return
end

function index_scale,value, min, max, top, inverse

df = (max-min)/top

r = where(value gt max) 

if (r(0) ne -1) then value(r) = (top+1)*df+min
r = where(value lt min) 
if (r(0) ne -1) then value(r) = min


inverse = findgen(top+1)*df+min

return, byte((value-min)/df)
end


; ### block_tool_l1 begins ###
pro block_tool_l1, date, time, gif=gif, display=display,  $
                   ions=ions, electrons=electrons, buffer=buffer, $
                   extend=extend, stop=stop, raw=raw, $
                   preliminary=preliminary, $
                   next=next, prev=prev, curr=curr

   common blocktool_common, block_data, mode_info, recnum

   common blocktool_colors, ncolors, background, foreground, $
    black, white, gray, yellow, blue, red, color_r, color_g, color_b

   common L1_rectime, L1rectime

; robust I/O
   if n_params() eq 2 then $
     hydra_time_io, date, time

; keyword processing
   if not keyword_set(display) then display=1
   next= keyword_set(next) and not keyword_set(prev)
   prev= keyword_set(prev) and not next
   curr= keyword_set(curr) and not (next or prev)
   raw=keyword_set(raw)

; plot specifications
   image_sizex= 11.75           ; inches
   image_sizey= 8.25            ; inches
   ppi=72                       ; pixels per inch
   
   sxplot= 2.0 *ppi             ; size in pixels
   syplot=   1.*ppi             ; pixels
   plot_space_x= 5.0  *ppi      ; spacing btw plots
   plot_space_y= 1.25 *ppi
   plot_offset_x= .75 *ppi      ; position of lower right corner
   plot_offset_y= .50 *ppi
   phase_pitch_space= .25 *ppi  ; space between phase and pitch plots

   sxbar=  .25  *ppi
   sybar=  2.   *ppi
   barposx= plot_space_x * 2 + plot_offset_x + 0.125*ppi
   barposy= 0                + plot_offset_y

   sxave= 7                     ; pixels
   aveposx= phase_pitch_space / 2 - sxave / 2

   timeposx= barposx + .45*ppi 
   timeposy= 5.0*plot_space_y + plot_offset_y + syplot

   sxdetave= 10                 ; pixels
                                ; sydetave depends on number of
                                ; energies, num_en
   detaveposx= timeposx -.40*ppi

   pi=acos(-1.0)


; initialize color bar
   ncolors=62
   bt_load_colors

; set non-color-bar colors

   pink=blue     ; we won't use blue (the spectragrams do)
   color_r(pink)=180
   color_g(pink)=140
   color_b(pink)=140

   green= fix ( ncolors * (.50) )
   
; initialize plotting device
   set_plot, 'Z'
   device, set_resolution = [ image_sizex*ppi, image_sizey*ppi ]
   
; get the data
   readdata:
   if not (next or prev or curr) then begin
       init_level1, date, block_data, mode_info   
       read_level1, recnum, block_data, mode_info, rectime=time
   endif else begin
       if (next) then recnum=recnum+1
       if (prev) then recnum=recnum+1
       if not curr then read_level1, recnum, block_data, mode_info
   endelse
   
   if not curr then begin
       if not raw then $
         correct_interpolate, block_data, mode_info
       calculate_pitch, block_data, mode_info
   endif   
   cdf_varget, $
     mode_info.l1stat.calid, 'PSI_STATUS', rec_start=block_data.blocknum, psi_status
   cdf_varget, $
     mode_info.l1stat.calid, 'PHASE_START', rec_start=block_data.blocknum, phase_start
   cdf_varget, $
     mode_info.l1stat.calid, 'PHI_OFFSET', rec_start=block_data.mode-1, phi_offset

   hour= fix( block_data.time / 3600 )
   min= fix( block_data.time mod 3600 / 60 )
   sec= fix( block_data.time mod 60 )

   timestr= string( hour, min, sec, format="( i2.2, ':', i2.2, ':', i2.2 )" )
   cdf_epoch, mode_info.epoch, yr, mon, day, /break
   datestr= string( mon, day, yr mod 100, format="( i2.2, '/', i2.2, '/', i2.2 )" )
   scan_len= mode_info.scan_length
   
   species= (mode_info.energy gt 0)*2 -1
   energy= mode_info.nom_energy_ref(*,0)
   ienergy= abs(mode_info.nom_energy_index_ref)

   if (raw) then counts= block_data.counts else $
     counts= block_data.corrected_counts
   r=where( block_data.datamask eq 0 )
   if r(0) ne -1 then counts(r)=-99.0
   pitch= block_data.alpha
   phase= ( phase_start+phi_offset ) mod (2*!pi)

   phase=phase*180.0/pi
   pitch=pitch*180.0/pi

   cdate=date
   ctime=time

plot:  
   zeros = where( counts eq 0 )
   counts( zeros ) = 0.00001
   glint = where( counts eq -99.0 )
   counts( glint ) = 0.00001
   log_counts = alog10( counts )
   counts( zeros ) = 0
   counts( glint ) = -99.0

   s=-1                         ; species: s=1 ions, s=-1 ele
   if ( keyword_set(ions) and not keyword_set(electrons) ) then s=1 
specloop:             
   if (s eq -1) then sp='Ele' else sp='Ions'

   if ( s gt 0 ) then wspecies = where(species gt 0) else $
    wspecies = where(species lt 0) 
   num_en= max( ienergy(wspecies) )  
   sydetave = 4*num_en  
   detaveposy = timeposy-sydetave

   min_co= alog10(0.1)
   if ( s eq -1) then begin
       max_co= alog10(500.)
       if keyword_set(extend) then max_co = alog10(4000.)
   endif else begin
       max_co= alog10(100.)
   endelse
  
   bar_max = max_co
   bar_min = min_co
   
   if  bar_max lt alog10(101.) then ytickv=[0.1, 1.0, 10.0, 100.0] $
   else if  bar_max lt alog10(501.) then ytickv=[0.1, 1.0, 10.0, 100.0, 500.0] $
   else if  bar_max gt alog10(9999.) then $
     ytickv = [0.1, 1.0, 10.0, 100.0, 1000.0, 10000.0] $
   else ytickv = [0.1, 1.0, 10.0, 100.0, 1000.0]
   
   color_counts = bytarr( scan_len, 12 )
   for i=0,scan_len-1 do $
     color_counts(i,*) = index_scale( log_counts(i,*), min_co, max_co, ncolors-1 )
  
                                ; plot the color bar
   bar= findgen(sybar) * (bar_max-bar_min)/sybar + bar_min
   _color_bar = bytarr(sybar)
   _color_bar = index_scale( bar, min_co, max_co, ncolors-1 )

   color_bar = bytarr(sxbar,sybar)
   for i=0,sybar-1 do color_bar(*,i) = _color_bar(i)
   
   yticks = n_elements(ytickv)-1
   plot, bar, /nodata, /device, yrange=[ 10^bar_min,10^bar_max ], /ylog,   $
     position=[ barposx, barposy, barposx+sxbar, barposy+sybar ], $
     xticks=1, xtickname= [" "," "], color=white, xstyle=1, ystyle=1, $
     title='counts', ytickv=ytickv, yticks=yticks, yticklen=-0.15, $
     yminor=9
   tv, color_bar(1:*,1:*), barposx+1, barposy+1
   
  ; print time, date, and species
   xyouts, timeposx, timeposy, datestr+'   '+timestr+'   '+sp, orientation=-90.,  $
     color=white, /device, size=1.3

  ; print psi status 
  psiposy= 2* plot_space_y + plot_offset_y + syplot
  if psi_status eq 1  then    $
    xyouts, timeposx, psiposy, "PSI ON", orientation=-90.0,      $
    size=1.3, color=green, /device                           $
  else                                                             $
    xyouts, timeposx, psiposy, "PSI OFF", orientation=-90.0,     $
    size=1.3, color=white, /device                           

  ; print mode
  mode= block_data.mode

  modesize = 1.0
  if (mode eq 10) then  modestr='1x2x' $
  else if (mode eq 22) then modestr='6(3e3p)'  $
  else if (mode eq 23) then modestr='6(4e4p)'  $
  else modestr=' '
  xyouts, detaveposx, psiposy, modestr, orientation=-90.0, size=modesize,  $
    color=white, /device
       
  tvphase= bytarr( sxplot, syplot, 12 )
  tvpitch= bytarr( sxplot, syplot, 12 )

  tvphase( *,*,* ) = black
  tvpitch( *,*,* ) = black

 ; energy_index -> energy map
  energies=fltarr(110)
  FOR i=0, scan_len-1 DO BEGIN
     energies(ienergy(i))=energy(i)
  ENDFOR 

 ; set the energy labels
  ytickv_energy = long(indgen(4)*((num_en-1)/3.0))+1
  ytickname_energy = strtrim(fix(energies(ytickv_energy)),1)
  yticks_energy = 3

 ; define the common plot parameters for 24 energy plots
  common_par_en = { nodata:1, device:1, noerase:1, color:white,              $
                    yrange:[0,num_en], xstyle:1, ystyle:1,                   $
                    xticks:4, yticks:yticks_energy, ytickv:ytickv_energy }

  ave= lonarr( num_en+1, 12 )         ; average count accumulator
  ave_c= intarr( num_en+1, 12 )       ; number of samples in average  
  
  for j=0,11 do begin
      ave( *,j ) = 0.0
      ave_c( *,j) = 0

      for i=0,scan_len-1 do begin    ; now fill in boxes with data
	 if ( species(i)*s gt 0 ) then begin   ; sign(species(i))=sign(s)
            y=block_linscale( syplot, 0, num_en, ienergy(i) )
            tienergy = ienergy(i)
            xphase=block_linscale( sxplot, 0, 360, phase(i) ) 
            xpitch=block_linscale( sxplot, 0, 180, pitch(i,j) ) 
            if ( counts(i,j) gt 0 ) then begin
               z=color_counts(i,j)
               block_put_point, tvpitch,j,xpitch,y,z, sxplot, syplot
	       block_put_point, tvphase,j,xphase,y,z, sxplot, syplot
               ave(tienergy,j)= ave(tienergy,j)+ long(counts(i,j))
               ave_c(tienergy,j)= ave_c(tienergy,j) + 1
            endif else if ( counts(i,j) eq 0 ) then begin
               block_put_point, tvpitch,j,xpitch,y, pink, sxplot, syplot
               block_put_point, tvphase,j,xphase,y, pink, sxplot, syplot
               ave_c(tienergy,j)= ave_c(tienergy,j) + 1
            endif else begin
               block_put_point, tvpitch,j,xpitch,y, gray, sxplot, syplot
	       block_put_point, tvphase,j,xphase,y, gray, sxplot, syplot
            endelse
         endif
      endfor

     ; calculate averages
      tvave= bytarr( sxave, syplot, 12 )
      tvave( *,*,j ) = black
      
      for en= 1,num_en do begin
         if ( ave_c(en,j) gt 0 ) then begin
            log_ave_counts= alog10(float(ave(en,j))/ave_c(en,j) > 0.0001)
            z = index_scale( log_ave_counts, min_co, max_co, ncolors-1 )
            y = block_linscale( syplot, 0, num_en, en )
            block_put_point, tvave, j, sxave*1/3, y, z, sxave, syplot 
            block_put_point, tvave, j, sxave*2/3+1, y, z, sxave, syplot
         endif
      endfor

      plotposy= ( 5 - j/2 ) * plot_space_y + plot_offset_y

     ; plot phase
      plotposx= (j mod 2) * plot_space_x + plot_offset_x
      tv, tvphase(*,*,j), plotposx, plotposy
      if  j lt 2  then title='s/c phase' else title=' '
      if (j mod 2) eq 0 then $      
         plot, tvphase(*,*,j), _extra=common_par_en, xrange=[0,360],          $
         position=[ plotposx, plotposy, plotposx+sxplot, plotposy+syplot ],   $
         xtickname=[' ',' ',' ',' ',' '], ytickname=[' ',' ',' ',' '],        $
         ytitle='Detector '+strtrim(string(j+1),1), title=title               $
      else                                                                    $
         plot, tvphase(*,*,j), _extra=common_par_en, xrange=[0,360],          $
         position=[ plotposx, plotposy, plotposx+sxplot, plotposy+syplot ],   $
         title=title, ytickname=ytickname_energy,                             $
         xtickname=[' ',' ',' ',' ',' ']        
      if  j gt 9  then        $       ; put labels on bottom plots
         plot, tvphase(*,*,j), _extra=common_par_en, xrange=[0,360],          $
         position=[ plotposx, plotposy, plotposx+sxplot, plotposy+syplot ],   $
         ytickname=[' ',' ',' ',' ']
      
     ; plot pitch
      if  j lt 2  then title='pitch angle' else title=' '
      plotposx= (j mod 2) * plot_space_x + $ 
         sxplot+ phase_pitch_space + plot_offset_x
      tv, tvpitch(*,*,j), plotposx, plotposy
      plot, tvpitch(*,*,j), _extra=common_par_en, xrange=[0,180],             $
         position=[ plotposx, plotposy, plotposx+sxplot, plotposy+syplot ],   $
         ytickname=[' ',' ',' ',' '], xtickname=[' ',' ',' ',' ',' '],        $
         title=title 
      if  j gt 9  then         $
         plot, tvpitch(*,*,j), _extra=common_par_en, xrange=[0,180],          $
         position=[ plotposx, plotposy, plotposx+sxplot, plotposy+syplot ],   $
         ytickname=[' ',' ',' ',' ']

     ; plot average
      plotposx= (j mod 2) * plot_space_x + sxplot + aveposx + plot_offset_x
      plot, tvave( *,*,j ), /nodata, /device, /noerase, color=white,     $
         position=[ plotposx, plotposy, plotposx+sxave, plotposy+syplot ],   $
         xticks=1, xtickname=[' ',' '], yticks=1, ytickname=[' ',' '],      $
         yrange=[0,10], xrange=[0,10]
      tv, tvave( 1:*, 1:*, j ), plotposx+1, plotposy+1
  endfor   

 ; calculate average over all detectors
  detave= lonarr( num_en+1 )      
  detave_c= intarr ( num_en+1 )
  tvdetave= bytarr( sxdetave, sydetave )

  FOR en= 1, num_en DO BEGIN 
     detave(en)= total( ave(en,*) )
     detave_c(en)= total( ave_c(en,*) )
     log_ave_counts= alog10(float(detave(en))/detave_c(en) > 0.0001)
     z = index_scale( log_ave_counts, min_co, max_co, ncolors-1 )
     y = block_linscale( sydetave, 0, num_en, en )
     ly= (y-4 > 0)
     uy= (y < sydetave-1)
     tvdetave( *, ly:uy ) = z
  ENDFOR

 ; plot average over all detectors
  plot, tvdetave, /nodata, /device, /noerase, color=white,               $
      position=[ detaveposx, detaveposy,                                 $
             detaveposx+sxdetave, detaveposy+sydetave ],                 $
      xticks=1, xtickname=[' ',' '], yticklen=-0.15,                     $
      yrange=[0,num_en],    $
      yticks=yticks_energy, ytickname=ytickname_energy, ytickv=ytickv_energy,   $
      title='<Ci>'
  tv, tvdetave( 1:*, 1:* ), detaveposx+1, detaveposy+1

  if (keyword_set(preliminary)) then begin
      xyouts, 0.5,0.5,'Preliminary', /normal, charsize=7.5, $
        alignment= 0.5, color=foreground, charthick=5, $
        orientation=30
  ENDIF
    
  d= tvrd()
  if keyword_set(gif) then begin   
      fname=   $
       strmid(timeinfo,28,2)+strmid(timeinfo,22,2)+strmid(timeinfo,25,2)+'.'+ $
       strmid(timeinfo,6,2) +strmid(timeinfo,9,2) +strmid(timeinfo,12,2) +  $
       strmid(sp,0,1) + '.gif'         
      write_gif, fname, d, color_r, color_g, color_b
  endif

  if display gt 0 then begin
      set_plot, 'x'
      tvlct, color_r, color_g, color_b
      window, display, xsize=image_sizex*ppi, ysize=image_sizey*ppi
      tv, d
  endif

  if ( (s eq -1) and (keyword_set(ions)) ) then begin
      s=1
      wait, 1
      goto, specloop
  endif

if (keyword_set(batch)) then if (not eof(unit)) then goto, readdata 

if (keyword_set(batch)) then delvar, cdate               

if (keyword_set(unit)) then begin
  close, unit
  if (not keyword_set(batch)) then spawn, 'rm '+date+'.'+time+'.dat'  
endif

if (keyword_set(stop)) then stop

END
