;***************************************************************************
;***************************************************************************
;* FILE:	 
;*      entrybox.pro
;* 
;* DESCRIPTION:  
;*      A window containing a entry field and buttons is displayed modally.
;*      The index of the pressed button is returned
;*
;* OVERVIEW:
;*      PRO entryBox_event, event
;*	FUNCTION EntryBox, strArray, buttons, CENTER=center
;*	
;* NEEDS:
;*      papco_util
;*
;* COMMON VARIABLES:
;*        COMMON EntryBox, buttonSelected
;*	  	 this COMMON-block is used to return the selected button
;*		 from the event-handler to the entryBox-function
;*               It must not be used by other programs.
;*
;* MODIFICATION HISTORY:       
;*      january 1997, written by R, Friedel
;***************************************************************************
;***************************************************************************

;***************************************************************************
;* PROCEDURE:     
;*      entryBox_event
;* 
;* DESCRIPTION:  
;*      this is the event-handler. It is called by XMANAGER only
;* 
;* INPUTS:       
;*      event 	: the event to be handled
;* 
;* OUTPUT:	 
;*      none
;*
;* CALLING SEQUENCE:
;*      this procedure is to be called by XMANAGER only
;*
;* MODIFICATION HISTORY:       
;*      january 1997, written by R, Friedel
;***************************************************************************
PRO entryBox_event, event

  COMMON EntryBox, buttonSelected, newvalue, char_base

  WIDGET_CONTROL,Event.Id,GET_UVALUE=Ev

  ON_ERROR, 2
  WIDGET_CONTROL, Event.Id, GET_UVALUE=Ev

  IF ev eq 'char_base' then begin
     newvalue=event.value
  endif

; check, if the event was initiated by pressing a button
  IF strlen(ev) GT 3 THEN $
     IF strmid(ev, 0, 3) EQ 'pb_' THEN BEGIN
        nr=fix(strfrom(ev,3))
        buttonSelected=nr
        WIDGET_CONTROL, char_base, GET_VALUE=entrystr
        newvalue=float(entrystr)
        WIDGET_CONTROL, Event.top, /DESTROY
     ENDIF
END

;***************************************************************************
;* FUNCTION:     
;*       FUNCTION entryBox, strArray, buttons, CENTER=center
;* 
;* DESCRIPTION:  
;*      A window containing a entry field and buttons is displayed modally.
;*      The index of the pressed button is returned
;*
;* INPUTS:       
;*	strArray:   An array containing the text to be displayed
;*	buttons :   a string-array containing the buttonlabels
;*
;* KEYWORDS:
;*	CENTER  :   If set, the displayed text is horizontally centered           
;*      TITLE   :   The title of the entry-box
;*      DefVal  :   A default value for the entry field
;*
;* OUTPUT:	 
;*      -1	:   if no button was selected
;*      Integer :   if a button was selected, its index is returned 
;*		    the first button has index 0
;*
;* CALLING EXAMPLE:
;*      IF EntryBox(['Exit','Program'], ['Yes', 'No'],$
;*                      TITLE='Exit Entry', /CENTER) EQ 0 $
;*         THEN EXIT
;*
;* MODIFICATION HISTORY:       
;*
;***************************************************************************
FUNCTION entryBox, strArray, buttons, CENTER=center, TITLE=title, $
                     XPOS=xpos, YPOS=ypos, DEFVAL=defval

   COMMON EntryBox, buttonSelected, newvalue, char_base

   newvalue=defval

   IF N_ELEMENTS(center) EQ 0 THEN center=1
   IF NOT KEYWORD_SET(TITLE) THEN title='Entry'   

   base = WIDGET_BASE($
         COLUMN=1, $
         TITLE=title, $
         UVALUE='Entry Box')

   width=60 > max(strlen(strArray))   
   buttonAnzahl=N_ELEMENTS(buttons)
   buttonWidth=max(strlen(buttons))+2
   buttonWidthTotal=buttonAnzahl*(buttonWidth)
   IF buttonAnzahl GT 1 THEN $
      buttonGap=(width-buttonWidthTotal) / (buttonAnzahl-1) + 1 $
   ELSE BEGIN
      buttonGap=(width-buttonWidthTotal) / 2 
      width=width > (buttonWidthTotal+2*buttonGap)
   ENDELSE
   width=width > (buttonWidthTotal + buttonGap*(buttonAnzahl-1))

   IF KEYWORD_SET(CENTER) THEN BEGIN
      strings=strarr(N_ELEMENTS(strArray))
      FOR i=0, N_ELEMENTS(strArray)-1 DO $
	strings(i)=strcenter(strArray(i), width)
   ENDIF ELSE BEGIN
      strings=strArray
   ENDELSE
 
   str = WIDGET_TEXT(base, $
      VALUE=strings,$
      YSIZE=N_ELEMENTS(strings), $
      XSIZE=width)

   char_base=CW_FIELD(base, $
                      value=defval, $
                      TITLE='Charsize (float):', $
                      UVALUE='char_base', $
                      /FLOATING)

   butBase=WIDGET_BASE(base, /row)
   FOR i=0, buttonAnzahl-1 DO BEGIN
     IF i GT 0 OR buttonAnzahl EQ 1 THEN $
        dummy=WIDGET_LABEL(butBase, VALUE=strcenter('', buttonGap))

     dummy=WIDGET_BUTTON(butBase, $
	      VALUE=strcenter(buttons(i), buttonWidth), $
	      UVALUE='pb_'+strtrim(string(i), 2))
   ENDFOR
   IF buttonAnzahl EQ 1 THEN $
       dummy=WIDGET_LABEL(butBase, VALUE=strcenter('', buttonGap))

   hidden_base=widget_base(base, MAP=0)
   dummy_draw=WIDGET_DRAW(hidden_base, xsize=1, ysize=1)

   geo=WIDGET_INFO(base, /GEOMETRY)
   x=600 - geo.xsize/2
   y=300 - geo.ysize/2

   if n_elements(xpos) gt 0 then x=xpos
   if n_elements(ypos) gt 0 then y=ypos

   WIDGET_CONTROL, base, XOFFSET=x, YOFFSET=y

   WIDGET_CONTROL, base, /REALIZE

   widget_Control, dummy_draw, get_Value=draw_Index

   SET_PLOT, 'X'     
   wset, draw_Index
   tvcrs, 0, 0

   buttonSelected=-1
   XMANAGER, 'entryBox', base, /MODAL
   if buttonSelected eq 0 then buttonSelected=newvalue else buttonSelected=defval
   return, buttonSelected 
END
