function hyd_bin_average, x_in, y_in, $
                          min=min, max=max, binsize=binsize, $ 
                          bin_variance=bin_var, $
                          bin_names=bin_name, $
                          bin_hits=bin_hits, $
                          fill_value=fill, $
                          _extra=e
;+
; NAME: hyd_bin_average
;
; PURPOSE: return average of data points y(x) in bins defined by 
;    min and binsize
;
; CATEGORY: utility
;
; CALLING SEQUENCE: average= hyd_bin_average( x, y, $
;                                             min=bin0_left,
;                                             binsize=binsize )
;
; INPUTS: y   is the data to be average.
;         x   the location in bin space of y.
;
; KEYWORD PARAMETERS:
;         min, max and binsize define the bins as with histogram.
;         min=min            the location of the first bin lower bound.
;         binsize=binsize    the width of each bin
;         max=max            a point within the last bin.
;         bin_variance=v     the variance of each bin average.
;         bin_names=bin_name the location of the mid-point of each bin.
;         bin_hits=bin_hits  the number of data points in each bin.
;         fill_value=fill    data with this value is invalid and is omitted
;                              from averages.  Default is NaN.
;
; OUTPUTS:
;    returns  the bin averages.
;
; COMMON BLOCKS: none.
;
; RESTRICTIONS: There may not always be enough virtual memory for large
;    numbers of bins.
;
; PROCEDURE:  use histogram to locate which points are in each bin, then
;    loop over bin to get the averages.
;
; EXAMPLE: 
;    x= findgen(10000)
;    y= randomn(s,10000)
;    avg= hyd_bin_average( x, y, min=0, binsize=1000, $
;                          bin_variance=var, bin_name=x_avg )
;    plot, x, y, psym=1
;    p_color0= !p.color & !p.color= get_color_index('red')
;    oploterr, x_avg, avg, var, color=get_color_index('red')
;    !p.color= p_color0
;
; MODIFICATION HISTORY:
;    written, jbf, October 27, 1999.
;-
  x= x_in
  y= y_in

  if n_elements(min) eq 0 then min= min(x)
  if n_elements(max) eq 0 then max= max(x)
  if n_elements(binsize) eq 0 then binsize=1.
  if n_elements(fill) ne 0 then begin      
      r= where( x eq fill )
      if r(0) ne -1 then x(r)=-1e31
      r= where( y eq fill )
      if r(0) ne -1 then begin
          y(r)=!values.f_nan
          x(r)=-1e31
      endif
      r= where( x ne -1e31 )
      if r(0) ne -1 then begin
          x= x(r)
          y= y(r)
      endif else begin
          message, 'no valid data found.', /cont
      endelse
  endif else begin
      fill=!values.f_nan
  endelse
  
  h= histogram( x, _extra=e, $
                min=min, max=max, binsize=binsize, $
                reverse_indices= R, /nan )
  avg= make_array( size=size(h), /double, value=fill )
  var= make_array( size=size(h), /double, value=fill )
  for i=0,n_elements(h)-1 do begin
      Ni= h(i)
      if Ni gt 0 then begin
          avg(i)= total( y(R(R(i):R(i+1)-1)), /nan ) / Ni
          if Ni gt 1 then begin
              var(i)= sqrt( total( ( y(R(R(i):R(i+1)-1)) - avg(i) )^2,/nan)/ $
                            ( Ni-1 ) )
          endif
      endif
  endfor
  bin_var= var
  bin_name= ( findgen( floor( (max-min)/binsize )+1 ) + 0.5 ) * binsize 
  bin_hits= h

  return, avg
end
