pro hydra_colorhist, Z_in, Data_in, binsize=binsize, min=min, max=max, $
                     fill=fill_in, zrange=zrange_in, zlog=zlog, $
                     psym10=psym10_in, $
                     draw_colorbar=draw_colorbar, inset_colorbar=cbar_inset, $
                     _extra=e

;+
; NAME: hydra_colorhist
;
; PURPOSE: plot histogram color-coded by second parameter
;
; CATEGORY: plotting
;
; CALLING SEQUENCE: hydra_colorhist, Z, X
; 
; INPUTS:
;    Z,  the code for the color.   This parameter is color-coded.
;    X,  the location of the bar.  This parameter is histogrammed.
;
; KEYWORD PARAMETERS:
;   binsize=    bin size for X
;   zbinsize=   bin size for Z.  default is color quantization.  Use
;                  this only when zbinsize>>color quantization.
;   min=        min of bins.  Measurements outside bins are not counted.
;   max=        max of bins.
;   psym10=     draw bars. 1=draw bars (default). 0=draw connect-the-dots.
;   zrange=     range of z values.
;   fill=       fill bars. 1=fill bars (default). 0= leave empty.
;   /draw_colorbar   draw colorbar.
;   colorbar keywords  see hydra_colorbar.pro.  Prefix with "cbar_"
;   
; PROCEDURE: loop over color.  first histogram all points below entire
;   zrange, and loop down to bottom of zrange.  
;
; EXAMPLE:
;   x= randomn(s,10000)*2+8
;   z= (x-8)/10*1+randomn(s,10000)/10+1
;   hydra_colorhist, z, x, binsize=0.25, /fill
;
; MODIFICATION HISTORY:
;
;-

   if n_params() eq 1 then begin
       Data= Z_in
       Z= findgen(n_elements(Data))
   endif else begin
       Data= Data_in
       Z= Z_in
   endelse

   if n_elements(e) ne 0 then begin
       sub_struct, e, 'cbar_*', clip_tag=strlen('cbar_'), cbar_parm, e
   endif

   check_plotparms, _extra=e

   if n_elements(psym10_in) eq 0 then psym10_in=1
   psym10= keyword_set( psym10_in )

   if n_elements(fill_in) eq 0 then fill_in=1
   fill= keyword_set( fill_in )   

   hydra_setcolors, cs
   
   IF NOT keyword_set(binsize) THEN binsize = 1.0
   IF n_elements(min) EQ 0 THEN min = floor(min(data)/binsize)*binsize
   IF n_elements(max) EQ 0 THEN max = floor(max(data)/binsize+1)*binsize

   if n_elements( zrange_in ) eq 0 then $
     zrange= [ min(Z,max=ma), ma ] $
   else $
     zrange= zrange_in
   
   if keyword_set( zlog ) then begin
       Z= alog10(Z)
       zrange= alog10(zrange)
   endif

   h = histogram(data, min=min, max=max, binsize=binsize)
   n = n_elements(h)+2
   x = findgen(n)*binsize+min-0.5*binsize
   y = [ 0,h,0 ]

   plot, x, y, _extra=e, /nodata

   zcolor= bytscl( Z, top= cs.colortable(1)-cs.colortable(0)+1, $
                   min=zrange(0), max=zrange(1) ) + cs.colortable(0) 

   for color= cs.colortable(1), cs.colortable(0),-1 do begin
       r= where( Zcolor le color )
       if r(0) ne -1 then begin
           h = histogram(data(r), min=min, max=max, binsize=binsize)
           n = n_elements(h)+2
           if psym10 then begin
               x= fltarr(n*2)
               i= findgen(n)*2
               x(i)= findgen(n)*binsize+min-0.5*binsize
               x(i+1)= findgen(n)*binsize+min+0.5*binsize
               y= fltarr(n*2)
               y(i)= [0,h,0]
               y(i+1)= [0,h,0]
           endif else begin
               x = findgen(n)*binsize+min-0.5*binsize
               y = [ 0,h,0 ]
           endelse
           if keyword_set(fill) then begin
               polyfill, x, y, color=color
           endif else begin
               oplot, x, y, color=color, psym=0
           endelse
       endif
   endfor

;   colorbar code
   if n_elements( cbar_parm ) gt 0 and $
     n_elements( draw_colorbar ) eq 0 then draw_colorbar=1
   if keyword_set( draw_colorbar ) then begin
       hydra_colorbar, zrange=zrange_in, zlog=zlog, ctable=cs.colortable, $
         _extra=cbar_parm
   endif
;   end colorbar code

end
