pro hydra_colorscat, X_in, Y_in, Z_in, $
                     zrange=zrange, zlog=zlog, dx=dx, dy=dy, $
                     sigx=sigx, sigy=sigy, $
                     _extra=plot_par, cellsize=pixsize, $
                     no_average=noavg, animate=animate_, step=step, $
                     backcolor=backcolor, $
                     min_value=min_value, max_value=max_value, $
                     overplot=overplot, ctable=ctable, $
                     sortZ=sortZ, sortabsZ=sortabsZ, $
                     bipolar=bipolar, $
                     colorbar=colorbar, draw_colorbar=draw_colorbar, $
                     nodata=nodata
;+
; NAME: hydra_colorscat
;
;
;
; PURPOSE:
;    creates a color-coded scatter plot.  A dot color-coded by Zi is
;    located at (Xi,Yi).  The size of the dot is controllable.  By default,
;    regions with overlapping dots are coded by the average of overlapping
;    measurements, so that if many points are plotted the average dependence
;    of Z on X,Y is displayed.  One may also animate the dots as they are
;    placed, so that the time-evolution may be observed as well.
;
; CATEGORY:
;    color plotting
;
;
; CALLING SEQUENCE:
;   hydra_colorscat, x, y, z
;
; 
; INPUTS:
;   x, y, z  1D arrays of ordinates
;
;
; OPTIONAL INPUTS:
;
;
;	
; KEYWORD PARAMETERS:
;   /draw_colorbar, /colorbar  draw colorbar. (colorbar
;                                            obsolete. Don't use)
;   zrange=[min,max]
;   /zlog
;   ctable=[imin,imax]
;   sigx, sigy    width of the dots.
;   /draw_colorbar  draw colorbar off to side
;   /inset_colorbar draw colorbar within plot, upper right corner.
;
; OUTPUTS: (to graphics device)
;
; SIDE EFFECTS:
;   As with plot, also sets !z.crange and !z.s
;   Calls hydra_setcolors to get the colortable, which may affect 
;     the colortable.
;
; RESTRICTIONS:
;
; PROCEDURE:
;
;
;
; EXAMPLE:
;
; hydra_colorscat, X, Y, Z, $
;                  zrange=[0.1,1000], /zlog, dx=3, dy=3, $
;                  sigx=0.1, sigy=0.1, $
;                  cellsize=3, $ 
;                  /no_average, animate=50, /step, $
;                  backcolor=get_color_index('white'), $
;                  min_value=0.1, max_value=1e30, $
;                  /overplot, ctable=[1,255], $
;                  /sortZ, /sortabsZ, $
;                  /draw_colorbar, ztitle=ztitle, cbar_title=zword
;                  
; X, Y, Z     are the data
; zrange=[,]  define max and min for color bar
; dx, dy      give dot size in cells (1,3,5,...)
; sigx, sigy  give the dot size in data units.
; cellsize    gives cell size in pixels (72 dpi for postscript)
; min_value, max_value  work like plot keyword.  See hydra_plotcolor.pro
; /overplot   overlay data onto existing plot
; /sortZ, /sortabsZ   plot the data in increasing Z or abs(Z) order
;
; ; example 2
; x= randomn(s,1000)
; y= randomn(s,1000)
; z= atan(y,x)
; hydra_colorscat, x, y, z, zrange=[-!pi,!pi],
;     position=[0.1,0.1,0.8,0.9], /draw_colorbar
; hydra_colorscat, x, y, z, zrange=[-!pi,!pi],
;     position=[0.1,0.1,0.8,0.9], /draw_colorbar, /no_average
; 
; MODIFICATION HISTORY:
;    written, jbf, 1997
;-

                     
; print usage if no arguments are specified.
if n_params() eq 0 then begin
    print, 'USAGE:                                                    '    
    print, 'pro hydra_colorscat, X, Y, Z, zrange=zrange, zlog=zlog, '
    print, '          dx=dx, dy=dy, sigx=sigx, sigy=sigy, step=step, '
    print, '          _extra=plot_par, cellsize=cellsize, backcolor=backcolor '
    return
endif

if n_elements(x_in) eq 0 then begin
    message, 'X is undefined.', /cont
    stop
endif

X=X_in
Y=Y_in
Z=Z_in

bipolar= keyword_set( bipolar )

if keyword_set( draw_colorbar ) then colorbar=1

if keyword_set( sortZ ) then begin
    s= sort(Z)
    X= X(s)  &  Y= Y(s)  &  Z= Z(s)
endif

if keyword_set( sortabsZ ) then begin
    s= sort(abs(Z))
    X= X(s)  &  Y= Y(s)  &  Z= Z(s)
endif

if n_elements( min_value ) gt 0 then begin
    r= where( Z ge min_value ) 
    if r(0) eq -1 then r(0)=0
    X=X(r)
    Y=Y(r)
    Z=Z(r)
endif

if n_elements( max_value ) gt 0 then begin
    r= where( Z le max_value ) 
    if r(0) eq -1 then r(0)=0
    X=X(r)
    Y=Y(r)
    Z=Z(r)
endif

zlog=keyword_set(zlog)

if n_elements(zrange) eq 0 then begin
    if total( abs( !z.range ) ) gt 0. then begin
        zrange= !z.range
    endif else begin        
        if bipolar then begin
            if zlog then zmin=min(abs(Z)) else zmin=0.
            zmax=max(abs(Z))
            zrange=[ zmin,zmax ]
        endif else begin
            zrange=[ min(Z,max=max),max ]
        endelse
    endelse
endif

zmin=zrange(0)
zmax=zrange(1)

noavg=keyword_set(noavg)
step=keyword_set(step)
if keyword_set( animate_ ) then begin
    nsteps= animate_
    animate= 1
endif else begin
    nsteps= n_elements(Z)
endelse

overplot= keyword_set( overplot )

if (n_elements(dx) eq 0) then dx=10
if (n_elements(dy) eq 0) then dy=10

if (n_elements(pixsize) eq 0) then pixsize=1

dx=dx/2 & dy=dy/2  

if n_elements( ctable ) ne 2 then begin
    hydra_setcolors, cs, /no_papco
    if bipolar then begin
        top= cs.colortable2(1)
        bottom= cs.colortable2(0)
    endif else begin
        top= cs.colortable(1)
        bottom= cs.colortable(0)
    endelse
    grey= cs.grey
endif else begin
    top= ctable(1)
    bottom= ctable(0)
    grey= get_color_index('grey')
endelse

if n_elements(backcolor) eq 0 then backcolor=grey

; set the x and y ranges
;print, 'set ranges'
if not keyword_set( overplot ) then begin    
    plot, X, Y, /nodata, _extra=plot_par
    !p.multi=!p.multi+[1,0,0,0,0]
endif

xdata=!x.crange & ydata=!y.crange ; ranges in data space
xlog=!x.type & ylog=!y.type
lldata= [ xdata(0), ydata(0) ]
sizedata=[ xdata(1)-xdata(0), ydata(1)-ydata(0) ]

; get the device coordinate system
;print, 'get device coords'
if (xlog) then xdata=10^xdata
if (ylog) then ydata=10^ydata
ll=fix(convert_coord(xdata(0),ydata(0),/data,/to_device))
ur=fix(convert_coord(xdata(1),ydata(1),/data,/to_device))

if overplot and !d.name eq 'ps' then begin
    message, 'overplot not implemented for postscript device.', /cont
    plot, [0,1,0,1],[0,0,1,1]
    return
endif

if overplot then begin
    d= tvrd()
    oldtv= d(ll(0):(ur(0)-1),ll(1):(ur(1)-1))
endif

if !d.name eq 'PS' then begin
    sizedev=((ur-ll)(0:1))/float(pixsize)/35 ; 72 dpi
    dfactor=1./35               ; device factor
endif else begin
    sizedev=((ur-ll)(0:1))/float(pixsize)
    dfactor=1.
endelse

if n_elements( sigx ) gt 0 then begin
    dx= ( sigx / ( xdata(1) - xdata(0) )  * sizedev(0) / 2 ) > 1
endif

if n_elements( sigy ) gt 0 then begin
    dy= ( sigy / ( ydata(1) - ydata(0) )  * sizedev(1) / 2 ) > 1
endif

; calculate color map averages
;print, 'calculate averages'
Zavg=dblarr(sizedev(0),sizedev(1)) ; values for each pixel, contains Z averages
Zavgc=lonarr(sizedev(0),sizedev(1)) ; number of Z's averaged in each pixel
Zavg1=dblarr(sizedev(0),sizedev(1)) 
maxindex= fix(sizedev)-[1,1]
scalefac= fix(sizedev)/sizedata

dot=(convert_coord(x,y,/data,/to_device))(0:1,*)
dot(0,*)=( dot(0,*)-ll(0) ) * dfactor
dot(1,*)=( dot(1,*)-ll(1) ) * dfactor

lldot= [ (dot(0,*)-dx)>0,           (dot(1,*)-dy)>0           ]
urdot= [ (dot(0,*)+dx)<maxindex(0), (dot(1,*)+dy)<maxindex(1) ] 

r=where( lldot(0,*) gt maxindex(0) or lldot(1,*) gt maxindex(1) )
if r(0) ne -1 then lldot(0,r)=-1 
r=where( urdot(0,*) lt 0 or urdot(1,*) lt 0 )
if r(0) ne -1 then lldot(0,r)=-1 

ndata= n_elements(Z)

iishow= 0  ; first index of current frame
while iishow lt ndata do begin
;   get the indeces to draw in this frame
    rshow= where( indgen(nsteps)+iishow lt ndata ) + iishow
    rvis= where( lldot( 0, rshow ) ge 0) + iishow
    if rvis(0) ge iishow then rshow=rvis else goto, shown
    
    if not keyword_set( nodata ) then begin
        if (not keyword_set(noavg)) then begin
            for irshow=0L,n_elements(rshow)-1 do begin
                i=rshow(irshow)
                Zavg( lldot(0,i):urdot(0,i), lldot(1,i):urdot(1,i) ) = $
                  Zavg( lldot(0,i):urdot(0,i), lldot(1,i):urdot(1,i) ) + Z(i)
                Zavgc( lldot(0,i):urdot(0,i), lldot(1,i):urdot(1,i) ) = $
                  Zavgc( lldot(0,i):urdot(0,i), lldot(1,i):urdot(1,i) ) + 1
            endfor
        endif else begin
            for irshow=0L,n_elements(rshow)-1 do begin
                i=rshow(irshow)
                Zavg( lldot(0,i):urdot(0,i), lldot(1,i):urdot(1,i) ) = Z(i)
                Zavgc( lldot(0,i):urdot(0,i), lldot(1,i):urdot(1,i) ) = 1
            endfor
        endelse
    endif
    shown:
    iishow=iishow+nsteps

    r0= where(Zavgc eq 0)
    r1= where(Zavgc ne 0)

    Zcolor=bytarr(sizedev(0),sizedev(1))
    
    if r1(0) ne -1 then begin
        Zavg1(r1)=Zavg(r1)/Zavgc(r1)
    
; calculate color map
;print, 'calculate color map'
        if (zlog) then begin
            logZavg= make_array( size=size(Zavg1) )
            rpos= where(Zavg1(r1) gt 0)
            rzero= where(Zavg1(r1) eq 0)
            logzmax=alog10(zmax)
            logzmin=alog10(zmin)
            if rpos(0) ne -1 then logZavg(r1(rpos))=alog10(Zavg1(r1(rpos)))
            if rzero(0) ne -1 then logZavg(r1(rzero))=logzmin
            Zcolor(r1)=bytscl(logZavg(r1), max=logzmax, min=logzmin, $
                              top=top-bottom-1) + bottom
        endif else begin
            Zcolor(r1)=bytscl(Zavg1(r1), max=zmax, min=zmin, $
                              top=top-bottom-1) + bottom
        endelse
    endif

    if overplot then begin
        if (r0(0) ne -1) then Zcolor(r0)=oldtv(r0)
    endif else begin
        if (r0(0) ne -1) then Zcolor(r0)=backcolor
    endelse
    
; display the color map
;print, 'display color map'
    if !d.name eq 'PS' then begin
        lln= convert_coord( ll, /device, /to_normal )
        urn= convert_coord( ur, /device, /to_normal )
        xsizen= urn(0)-lln(0)
        ysizen= urn(1)-lln(1)
        tv, zcolor, lln(0), lln(1), xsize=xsizen, ysize=ysizen, /normal
    endif else begin
        sizedev=sizedev*pixsize    
        tv, congrid(Zcolor, sizedev(0), sizedev(1)), ll(0), ll(1)
    endelse
    if keyword_set(step) then stop, 'Type .cont for next frame'
    
endwhile

; redraw the axes at the same position
;print, 'redraw axes'

if not overplot then begin
    pold=!p
    plot, X, Y, /nodata, /noerase, _extra=plot_par
    !p=pold
    !p.multi=!p.multi-[1,0,0,0,0]
endif


if keyword_set( colorbar ) then begin
    hydra_colorbar_link, zrange=[zmin,zmax], zlog=zlog, ctable=[bottom,top]
endif
    
if zlog then begin
    !z.crange= alog10( [ zmin, zmax ] )
endif else begin
    !z.crange=[ zmin, zmax ]
endelse

return
end

