pro hydra_setcolors, cs, no_papco=no_papco, $
                     gsfc_colorbar=gsfc, rainbow_colorbar=rainbow, $
                     bi3polar_colorbar=bi3polar_colorbar, $
                     ncol=ncol, force=force
;+
; NAME: hydra_setcolors
;
;
;
; PURPOSE: fills the idl colortable with primary colors that can be
; accessed by name, and some number of colorbars.  Also, some
; colorplotting utilities sense that hydra_setcolors color system has
; been installed and will use the loaded colors.
;
;
;
; CATEGORY: color plotting
;
;
;
; CALLING SEQUENCE: hydra_setcolors, [name_structure]
;
;
; 
; INPUTS: none.
;
;
;	
; KEYWORD PARAMETERS: 
;    /no_papco   hydra_setcolors trys to sense if papco's color system
;                is installed, and if it is, use those colors instead
;                of resetting the colortable.  /no_papco specifies
;                that the colortable should be reset, regardless if
;                papco is running or not.  (The sense fails, and
;                incorrect colors are used.)
;    /gsfc_colorbar  use gsfc colorbar (default)
;    /rainbow_colorbar  use rainbow colorbar 
;    /bi3polar_colorbar
;
;
; OPTIONAL OUTPUTS:
;    name_structure    is a structure containing the names of the
;                      colors, and the location of the colorbars.
;                      Presently, it has the form:
;
;         cs={ white:0, black:0, yellow:0, magenta:0, cyan:0, $
;              red:0, green:0, blue:0, grey:0, colortable:intarr(2), $
;              colortable2:intarr(2), $
;              foreground:0, background:0 $
;            }
;
;                      So for example cs.red is the index of the color
;                      red.  If the name_structure isn't used, then
;                      the colors may be accessed through a common block.
;
;
; COMMON BLOCKS:
;   This common block is obsolete and name_structure should be used instead:
;     common hydra_setcolors_common, white, black, yellow, magenta, cyan, $
;        red, green, blue
;
;   This common block is sensed by colorplotting codes, and lets them
;   know where the colorbar resides:
;     common hydra_setcolors1_common, cs_common
;
;
;
; SIDE EFFECTS:
;   Loads colortable.
;  *new* If device is 'PS', then "device, /color" is executed.
;
; PROCEDURE:
;   Form the R, G, B arrays.  Call hydra_colors to get the two
;   colorbar tables.  Use congrid to shrink the colorbars.  Load the
;   composite colortable with tvlct. 
;
; EXAMPLE:
;   hydra_setcolors, cs
;   plot, [1,2,3], /nodata
;   plot, [1,2,3], color=cs.yellow
;   plot, [1,1], !y.crange, color=cs.red
;   hydra_plotcolor, dist(20), ctable=cs.colortable
; 
; NOTES:
;   
; SEE ALSO:
;   get_color_index
;   hydra_colors
;
; MODIFICATION HISTORY:
;   written, 1997, Jeremy Faden
;-

  if not keyword_set( rainbow ) then gsfc=1

  common hydra_setcolors_common, white, black, yellow, magenta, cyan, $
    red, green, blue

  common hydra_setcolors1_common, cs_common

  cs={ white:0, black:0, yellow:0, magenta:0, cyan:0, $
       red:0, green:0, blue:0, orange:0, grey:0, purple:0, aqua:0, $
       grey00:0, grey20:0, grey40:0, grey60:0, grey80:0, grey100:0, $
       colortable:intarr(2), ct_base:0, ct_ncol:0, $
       colortable2:intarr(2), ct2_base:0, ct2_ncol:0, $
       greytable:intarr(2), $
       foreground:0, background:0 $
     }

;  is the display set?
  if getenv('DISPLAY') eq '' and $
    !version.os_family eq 'unix' and !d.name eq 'X' then begin
      message, 'DISPLAY not set.  Aborting.', /cont
      return
  endif 
    
;  can we read this device?
  catch, err 
  if err eq 0 then begin
      tvlct, /get, r, g, b           
      tvlct, r, g, b           
  endif else begin
      message, 'Can''t read colors from this device.', /cont
      return
  endelse
  catch, /cancel

  if !d.table_size lt 100 then begin
      message, 'need 100 colors, '+strtrim(!d.table_size,2)+$
        ' found.  Aborting.', /cont
      return
  endif

  common papco_color, papco_color, r_papco, g_papco, b_papco 
  if n_elements( papco_color ) gt 0 then begin
      tvlct, /get, r, g, b
      if total( abs( r_papco - r ) ) gt 2. then begin ; 2 is some threshold
;          no_papco=1
          message, 'papco colorsystem detected but colors not loaded. ', /cont
          message, 'resetting color table.', /cont
          message, 'total( abs( r_papco - r ) )='+$
            strtrim(total( abs( r_papco - r ) ),2), /cont          
          message, 'Using papco color system anyway.', /cont
;          x= temporary( r_papco )
      endif
  endif
  
  if n_elements( papco_color ) gt 0 and not keyword_set( no_papco ) $
    and not keyword_set(force) then begin
      COMMON papco_color_names, black1, red1, green1, yellow1, $
        blue1, magenta1, cyan1, $
        white1, burgundy1, olive1, dark_green1, teal1, $
        royal_blue1, violet1, dark_grey1, grey1, $
        foreground1, background1, color_bar_names1
      cs.white= white1
      cs.black= black1
      cs.yellow= yellow1
      cs.magenta= magenta1
      cs.cyan= cyan1
      cs.red= red1
      cs.green= green1      
      cs.blue= blue1
      cs.grey= grey1
      cs.orange= get_color_index('orange')
      cs.purple= get_color_index('purple')
      cs.aqua= get_color_index(6,255,206)

      forward_function PAPCO_get_Colorindices
      color_range= PAPCO_get_Colorindices( 0 )
      cs.colortable= color_range
      cs.foreground= foreground1
      cs.background= background1
      return
  endif

  red_arr=bytarr(!d.table_size)
  green_arr=bytarr(!d.table_size)
  blue_arr=bytarr(!d.table_size)
  if n_elements(ncol) eq 0 then $    
    ncol= !d.table_size

  if keyword_set(gsfc) then begin
      hydra_colors, /gsfc, r1, g1, b1
  endif 

  if keyword_set(rainbow) then begin
      hydra_colors, /rainbow, r1, g1, b1 
  endif

  
  if keyword_set( bi3polar_colorbar ) then begin
      hydra_colors, /bi3polar, r1, g1, b1
  endif 

  hydra_colors, /bipolar, r2, g2, b2  
  
  n1= n_elements(r1)            ; unipolar color bar
  n2= n_elements(r2)            ; bipolar color bar
  n3= 18                        ; primary colors

  if ncol lt ( n1+n2+n3 ) then begin
      r1= congrid( r1, n1/2 )
      g1= congrid( g1, n1/2 )
      b1= congrid( b1, n1/2 )
      r2= congrid( r2, n2/2 )
      g2= congrid( g2, n2/2 )
      b2= congrid( b2, n2/2 )      
      n1= n_elements(r1)        ; unipolar color bar
      n2= n_elements(r2)        ; bipolar color bar      
  endif

  base= 14
  n= base+n2-1
  red_arr(base:n)= r2
  green_arr(base:n)= g2
  blue_arr(base:n)= b2
  cs.colortable2(0)=base
  cs.colortable2(1)=n            ; last color of colortable is white
  cs.ct2_base= base
  cs.ct2_ncol= n - base + 1

  base= base+n2
  n= base+n1-1
  red_arr(base:n)= r1
  green_arr(base:n)= g1
  blue_arr(base:n)= b1
  cs.colortable(0)=base
  cs.colortable(1)=n-1 
  cs.ct_base= base
  cs.ct_ncol= n - base

  grey_rgb=[ 20, 40, 60, 80, 100 ] * 255 / 100 
  named_red=   [ 0, grey_rgb, 0, 255,   0,   0,   0, 255, 255, 200 ]
  named_green= [ 0, grey_rgb, 0,   0, 255,   0, 255,   0, 255, 200 ]
  named_blue=  [ 0, grey_rgb, 0,   0,   0, 255, 255, 255,   0, 200 ]
  
  red_arr(0)=   named_red
  green_arr(0)= named_green
  blue_arr(0)=  named_blue

  cs.black=0 & cs.red=7 & cs.green=8 & cs.blue=9 
  cs.cyan=10 & cs.magenta=11 & cs.yellow=12
  cs.grey=13
  
  cs.grey00= 0
  cs.grey20= 1
  cs.grey40= 2
  cs.grey60= 3
  cs.grey80= 4
  cs.grey100= 5
  
  cs.greytable= [ 0, 5 ]

  black=0 & red=11 & green=12 & blue=13 
  cyan=14 & magenta=15 & yellow=16 
  
  lastcolor=!d.table_size-1
  red_arr(lastcolor)=255
  green_arr(lastcolor)=255
  blue_arr(lastcolor)=255
  white=lastcolor
  cs.white= lastcolor

  if !d.name eq 'PS' or !d.name eq 'PCL' then begin
      if !d.name eq 'PS' then device, /color
      cs.background= cs.white
      cs.foreground= cs.black
  endif else begin
      cs.background= cs.black
      cs.foreground= cs.white
  endelse
  
  if !d.name eq 'PCL' then begin
      !p.background= lastcolor
      !p.color= 0
  endif

  tvlct, red_arr, green_arr, blue_arr

  cs.orange=get_color_index('orange') ; orange is part of colorbar
  cs.purple= get_color_index('purple')
  cs.aqua= get_color_index(6,255,206)

  cs_common= cs
end

