pro hyper_xyouts, x_in, y_in, text_in, $
                  normal=normal, $
                  charsize=charsize, $
                  symsize=symsize, $
                  color=color, $
                  width=width, $
                  _extra=e
;+
; NAME: hyper_xyouts
;
; PURPOSE: extend xyouts' function by adding command keywords that
;    can change text color and insert psyms.
;
; CATEGORY: plotting
;
; CALLING SEQUENCE: hyper_xyouts, x, y, text
; 
; INPUTS: 
;    x, y   locate the text, as with xyouts.
;    text   text string to print, as with xyouts, but command keywords
;             are interpretted.  
;
; COMMAND_KEYWORDS:
;    <color=name>   change text color to crayola name (eg red, green, blue...) 
;    </color>       return to entry color
;    <red>, <green>, etc
;                   same as color=name for defined colors.
;    <psym=n>       insert symbol number n (see psym keyword)
;    <linestyle=n>  insert line of style n (see linestyle keyword)
;    <gt>           insert greater than (>) sign.
;    <lt>           insert less than (<) sign.
;
; KEYWORD PARAMETERS: same as xyouts.
;
; OUTPUTS: writes to graphics device.
;
; SIDE EFFECTS: none
;
; RESTRICTIONS:
;      note that greater thans and less thans are interpretted as
;      control keywords, so the string will parse improperly if
;      <gt> and <lt> keywords are not used.
;
; PROCEDURE: parse the string into fragments of text that are
;      located and properly colored, oplotting lines and psyms
;      as requested between the text fragments.
;
; EXAMPLE:
;      xyouts, 0.5, 0.5, /normal, $
;         '<red><psym=4></color> red diamond!c'+$
;         '<yellow><psym=6></color> yellow square'
;      hyper_xyouts, 0.5, 0.5, /normal, $
;         '<red><psym=4></color> red diamond!c'+$
;         '<yellow><psym=6></color> yellow square'
;
; MODIFICATION HISTORY:
;      
;-


  if not keyword_set( normal ) then begin
      r= convert_coord( x_in,y_in, /data, /to_normal )
      orig_x= r(0)              ; origin
      orig_y= r(1)
  endif else begin
      orig_x= x_in              ; origin
      orig_y= y_in
  endelse
  
  if n_elements( charsize ) eq 0 then charsize=!p.charsize
  if charsize eq 0 then charsize=1.
  if n_elements( symsize ) eq 0 then symsize=!p.symsize
  if symsize eq 0. then symsize=1.
  if n_elements( color ) eq 0 then color=!p.color

  char_height= !d.y_ch_size / float(!d.y_size) * abs(charsize)
  char_width= !d.x_ch_size / float(!d.x_size) * abs(charsize)
  
  ccolor= color
  entry_color= color
  width= 0.

  text= str_sep( text_in, '!c' )
  for i=0,n_elements(text)-1 do begin
;      parse for meta characters
      t1= text(i)
      s= str_sep( t1, '<' )
      final_text=''
      cp=0.                     ; carriage_position
      for j=0,n_elements(s)-1 do begin
          s2= str_sep( s(j), '>' )
          if n_elements( s2 ) gt 1 then begin
              meta= s2(0)
              text_frag= s2(1)
;              parse meta
              s4= str_sep( meta, '=' )
              cmd=strtrim(s4(0),2)
              if n_elements(s4) gt 1 then arg=strtrim(s4(1),2)
              case cmd of 
                  'gt': begin
                      text_frag= '>'+text_frag
                  end
                  'lt': begin
                      text_frag= '<'+text_frag
                  end
                  'psym': begin
                      fac= 0.25
                      psym=long(arg)
                      if psym le 0 then begin
                          x= orig_x+cp+[0,1]*(char_height*symsize^fac)/2
                          y= [1,1]*orig_y-char_height*(i-0.5)
                      endif else begin
                          x= orig_x+cp+(char_height*symsize^fac)/2
                          y= orig_y-char_height*(i-0.5)
                      endelse
                      r= convert_coord( x, y, /norm, /to_data )
                      if charsize ge 0 then $
                        oplot, [r(0,*)], [r(1,*)], $
                        psym=psym, color=ccolor, symsize=symsize, $
                        /noclip
                      cp=cp+char_height*symsize^fac
                  end
                  'linestyle': begin
                      linestyle=long(arg)
                      fac=2
                      x= orig_x+cp+[0,1]*char_height*fac
                      y= [1,1]*orig_y-char_height*(i-0.5)

                      r= convert_coord( x, y, /norm, /to_data )
                      if charsize ge 0 then $
                        oplot, [r(0,*)], [r(1,*)], linestyle=arg, color=ccolor
                      cp=cp+char_height*fac
                  end
                  'color': begin
                      ccolor=get_color_index(arg)
                  end
                  'red': ccolor=get_color_index(cmd)                      
                  'blue': ccolor=get_color_index(cmd)                      
                  'yellow': ccolor=get_color_index(cmd)                      
                  'green': ccolor=get_color_index(cmd)                      
                  'cyan': ccolor=get_color_index(cmd)                      
                  'magenta': ccolor=get_color_index(cmd)                      
                  '/color': begin
                      ccolor=entry_color
                  end
              endcase              
          endif else begin
              text_frag= s2(0)
          endelse
          xyouts, $
            orig_x + cp, $
            orig_y - char_height*i, /normal, $
            text_frag, width=w, color=ccolor, charsize=charsize
          cp= cp+w
      endfor
      width= cp>width
  endfor
end
