function hyd_filter_freq, time, signal, spin_period, $
                          plot=plot, label=label, $
                          verbose=verbose,$
                          acoeff=acoeff,$
                          bcoeff=bcoeff,$
                          phasor=phasor
;+
; NAME:
;       hyd_filter_freq
; PURPOSE:
;       Filter out noise in B field measurements due to spacecraft spin.
; CATEGORY:
;       
; CALLING SEQUENCE:
;       hyd_filter_freq, time, signal, spin_period, $
;                          plot=plot, label=label, $
;                          verbose=verbose,$
;                          acoeff=acoeff,$
;                          bcoeff=bcoeff,$
;                          phasor=phasor
; INPUTS:
;       time[,]:         time array
;       signal[,,]:       B field data in 3 dimensions
;       spin_period:     average spin period for time interval
; OPTIONAL INPUTS:
;	
; KEYWORD PARAMETERS:
;       plot:            plot data if desired
;       label:           plot title
;       verbose:         print variables to screen
; OUTPUTS:
;       acoeff:          Fourier cosine coeff. of spin tone component of data
;       bcoeff:          Fourier sine   coeff. of spin tone component of data
;       phasor:          [acoeff, bcoeff]
; OPTIONAL OUTPUTS:
;       None.
; COMMON BLOCKS:
;       None.
; SIDE EFFECTS:
;       None.
; RESTRICTIONS:
;       None.
; PROCEDURE:
;       Use Fourier series to describe the data component at the
;       spin frequency.  This component is then subtracted from the
;       input signal.
; EXAMPLE:
;
; MODIFICATION HISTORY:
;       Released:  JAS 3/5/02
;-
;
;   Filter a signal at a given frequency
;
;   Inputs:
;         time:         time values
;         signal:       signal values
;         spin_period:  frequency to be filtered (1/T)
;
;   Keywords:
;         /plot                 Plot or not
;         label=label           Label for y-axis of plots
;         /verbose              Opens print statements
;         phasor=[,]            output the integral in the complex plane.


  bx=signal

  if (n_elements(bx) ne n_elements(time)) then begin
      print, 'Vector sizes do not match'
      stop
  endif

; Determine maximum number of spins available for Fourier window
;  dt=time[1]-time[0]

  dt          = median( time(1:*) - time(0:*))

  boxcar      = long(2.0*spin_period/dt) + 1
  boxcar_half = long((boxcar - 1)/2 + 0.5)
  samples     = n_elements(bx)
  Nspin       = long((samples-boxcar)*dt/spin_period)
  if keyword_set(verbose) then begin
      print, 'dt = ',dt
      print, 'boxcar = ',boxcar
      print, 'spin_period = ',spin_period
      print, 'boxcar_half = ',boxcar_half
      print, 'Nspin       = ',Nspin
  endif

; Determine Fourier info.
  fourier_seconds     = Nspin*spin_period
  fourier_samples     = fourier_seconds/dt
  fourier_samples_int = long(fourier_samples)
  fourier_samples_rem = fourier_samples - fourier_samples_int
  if keyword_set(verbose) then begin
      print, 'fourier_seconds     = ',fourier_seconds
      print, 'fourier_samples_int = ',fourier_samples_int
      print, 'fourier_samples_rem = ',fourier_samples_rem
  endif

; Determine start and end array indices for fourier transform
  fourier_start       = boxcar_half
  if (fourier_samples_rem gt 0) then begin
      extra = 1
  endif else begin
      extra = 0
  endelse
  fourier_end = boxcar_half + fourier_samples_int + extra - 1
  if keyword_set(verbose) then begin
      print, 'fourier window = ',fourier_start,fourier_end
  endif

  if keyword_set(label) then vectorlabel=label
  vector     = bx
  smvector   = bx
  
  smoothvector = smooth(smvector,boxcar)
  t_arr        = time - time[0]

; Interpolate to obtain values of last partial value

  if (fourier_samples_rem gt 0) then begin

      t_arr(fourier_end)        = t_arr(fourier_end - 1) + $
        (t_arr(fourier_end) - t_arr(fourier_end - 1))*$
        fourier_samples_rem

      vector(fourier_end)       = vector(fourier_end - 1) + $
        (vector(fourier_end) - vector(fourier_end - 1))*$
        fourier_samples_rem

      smoothvector(fourier_end) = smoothvector(fourier_end - 1) + $
        (smoothvector(fourier_end) - smoothvector(fourier_end - 1))*$
        fourier_samples_rem
      
  endif

  if keyword_set(verbose) then begin
      print, 'index = ',fourier_end-2,fourier_end-1,fourier_end,fourier_end+1
      print, 't_arr = ',t_arr(fourier_end-2),t_arr(fourier_end-1),$
        t_arr(fourier_end),t_arr(fourier_end+1)
  endif

; Translate arrays down to zero array indices
  smoothvector = smoothvector[fourier_start:fourier_end]
  vector       = vector[fourier_start:fourier_end]
  function_less_lowfreq         = vector - smoothvector

  t_arr        = t_arr[fourier_start:fourier_end]

  Linverse = 2/fourier_seconds
  phi      = Linverse*Nspin*!pi*t_arr

; Find acoeff coefficient
  integrand     = function_less_lowfreq * cos(phi)
  integral = trapezoid_intg(t_arr,integrand)
  acoeff = Linverse*integral

  if keyword_set(verbose) then begin
      print, 'acoeff = ',acoeff
  endif
; Find bcoeff coefficient
  integrand     = function_less_lowfreq * sin(phi)
  integral = trapezoid_intg(t_arr,integrand)
  bcoeff = Linverse*integral
  if keyword_set(verbose) then begin
      print, 'bcoeff = ',bcoeff
  endif

; Compose spin tone element
  if keyword_set(plot) then begin
      spintone= acoeff*cos(phi) + bcoeff*sin(phi)
      
; Plot results
      arb_start   = 0
      num_sam_arb = 1000 < (n_elements(t_arr)-1 -arb_start)
      
      plot,  t_arr[arb_start:num_sam_arb],vector[arb_start:num_sam_arb],$
        psym=3,$
        min_val=-1E10,$
        xtitle='Time',$
        ytitle=vectorlabel
      
      oplot, t_arr[arb_start:num_sam_arb],$
        smoothvector[arb_start:num_sam_arb],$
        color = get_color_index('green')
      
      oplot, t_arr[arb_start:num_sam_arb],$
        vector[arb_start:num_sam_arb]-spintone[arb_start:num_sam_arb],$
        color = get_color_index('red')
      
      axis, yaxis=1, ytitle=systime(0)
  endif      

  phasor= [ acoeff, bcoeff ]

; Expand spintone to range over original time window
;  while accounting for offset caused if boxcar set arbitrarily
  vector   = signal
  phi      = Linverse*Nspin*!pi*(time-time(fourier_start))
  spintone = acoeff*cos(phi) + bcoeff*sin(phi)

  return, vector-spintone
  end





