pro create_moments_data_mask, cdfid, rec_start, rec_count, data_mask

;+
; NAME: create_moments_data_mask.pro
;
; PURPOSE: Given the moments CDF file id, rec_start and rec_count,
; extract the necessary status variables to create a data mask to
; apply to other moment variables extracted from the same CDF file.
; Invalid moments points will have a data mask value of zero, valid
; points will have a data mask value of 1.
;
; CATEGORY: Moments extraction tool.
;
; CALLING SEQUENCE: 
;    create_moments_data_mask, cdfid, rec_start, rec_count, data_mask
; 
; INPUTS:   cdfid      -  Moments CDF file ID # returned when opened
;           rec_start  -  First record to extract
;           rec_count  -  Number of records to extract
;
; OPTIONAL INPUTS:  None
;
; KEYWORD PARAMETERS:  None
;
; OUTPUTS:  data_mask  - 1-D array of dimension rec_count
;                        data_mask = 0 for invalid data, 1 for valid data
;
; OPTIONAL OUTPUTS:  None
;
; COMMON BLOCKS:  None
;
; MODIFICATION HISTORY: Written, 3/23/99, Pamela A. Puhl-Quinn
;-

if (n_params() eq 0) then begin
    doc_library, 'create_moments_data_mask'
    return
endif

data_mask = intarr(rec_count)

if (cdf_varexist(cdfid,'status_flag')) then begin
    
    message, 'Using "status_flag" for the moments data mask', /cont
    
    cdf_varget, cdfid, 'status_flag', status_flag, $
      rec_start=rec_start, rec_count=rec_count

    gd = where(status_flag eq 0)
    if (gd(0) ne -1) then data_mask(gd) = 1
    
endif else begin
    
    message, 'Using "b_status" and "status" for the moments data mask', /cont
    
    cdf_varget, cdfid, 'b_status', b_status, $
      rec_start=rec_start, rec_count=rec_count
    
    cdf_varget, cdfid, 'status', status, $
      rec_start=rec_start, rec_count=rec_count
    
    patch_flag = (status/10000) MOD 10 ; indicates the fifth digit of status
    
    gd = where(status GT 0 AND patch_flag NE 3 AND b_status EQ 1)
    
    if (gd(0) ne -1) then data_mask(gd) = 1
    
endelse

return
end


