pro deriv_lagrange4, zarr, varr_in, deriv, $
                     wrt_x=wrt_x, wrt_y=wrt_y, fill=fill_in
;+
; NAME: deriv_lagrange4.pro
;
; PURPOSE: Given a function Z(X) (OR Z(X,Y)) and X, this routine
;          computes dZ/dX (OR dZ/dY) by constructing the 
;          DERIVATIVE of the 4th Lagrange interpolating polynomial.
;          For dZ/dX at X(i), this method uses two adjacent points on
;          both sides of X(i) (i.e. X(i-2),X(i-1),X(i),X(i+1),X(i+2)).
;          Therefore Z must have at least 5 elements.  Because of
;          these boundary issues, dZ/dX will only be defined for
;          [X(2),X(3),...X(N-4),X(N-3)] where N is the number of
;          elements of Z.  dZ/dX at [X(0),X(1),X(N-2),X(N-1)] will
;          be filled with the standard fill value of -1.e+31, unless
;          the user specifies the fill to use with the fill keyword.
;
;          See: 'Numerical Analysis' by Burden and Faires, 
;                      Page. 98, 5th Edition
;
; CATEGORY: Numerical Analysis
;
; CALLING SEQUENCE: 1D case:  deriv_lagrange4, Z, V, deriv
;                   2D case:  deriv_lagrange4, Z, V, deriv, /wrt_x
;
; INPUTS:  Z        1D or 2D array
;
;          V        The variable you are taking the derivative
;                   with respect to.  It must be either of
;                   the same dimension as Z (for either 1D or
;                   2D case), or the same dimension as
;                   Z(*,0) (for the 2D case with the "wrt_x"
;                   keyword set) or Z(0,*) (for the 2D case with
;                   the "wrt_y" keyword_set).
;	
; KEYWORD PARAMETERS:  /wrt_x     If Z is 2D, derivative with respect
;                                 to the first argument is returned
;                                 (Ignored if Z is 1D)
;
;                      /wrt_y     If Z is 2D, derivative with respect
;                                 to the second argument is returned
;                                 (Ignored if Z is 1D)
;
;                      fill=F     F will be used as the fill value
;                                 (see above).  If not set, a fill
;                                 value of -1.e+31 is used.
;
; OUTPUTS:   deriv   dZ/dV; will have the same dimensions as Z
;
; COMMON BLOCKS: none
;
; RESTRICTIONS:  If Z is 2D, you MUST set either "wrt_x" or "wrt_y"
;
; EXAMPLES:
;
; Z is 2D, V is 2D, /wrt_x ---------------------------------------
; V = [1.,2.,3.,4.,5.,6.,7.,8.,9.,10.]#make_array(2,/float,value=1.)
; Z = .3452 + .2093*V + 12.*V^2
; nx = n_elements(V(*,0))
; ny = n_elements(V(0,*))
; deriv_lagrange4, Z, V, deriv, /wrt_x
; code = deriv(2:nx-3,*)
; theory = .2093 + 2.*12.*V(2:nx-3,*)
; print, 'Z is 2D, V is 2D, /wrt_x:'
; print, abs((theory-code)/theory)*100. ; % error
;
; Z is 2D, V is 1D, /wrt_y ---------------------------------------
; V = [1.,2.,3.,4.,5.,6.,7.,8.,9.,10.]
; Z = .3452 + .2093*V + 12.*V^2
; Z = make_array(2,/float,value=1.)#Z
; nx = n_elements(Z(*,0))
; ny = n_elements(Z(0,*))
; deriv_lagrange4, Z, V, deriv, /wrt_y
; code = deriv(*,2:ny-3)
; theory = .2093 + 2.*12.*make_array(2,/float,value=1.)#V(2:ny-3)
; print, 'Z is 2D, V is 1D, /wrt_y:'
; print, abs((theory-code)/theory)*100. ; % error
;
; Z is 1D, V is 1D         ---------------------------------------
; V = [1.,2.,3.,4.,5.,6.,7.,8.,9.,10.]
; Z = .3452 + .2093*V + 12.*V^2
; nx = n_elements(Z)
; deriv_lagrange4, Z, V, deriv
; code = deriv(2:nx-3)
; theory = .2093 + 2.*12.*V(2:nx-3)
; print, 'Z is 1D, V is 1D:'
; print, abs((theory-code)/theory)*100. ; % error
;
; MODIFICATION HISTORY: Written, 2/9/00, Pamela A. Puhl-Quinn
;                                        University of Iowa
;                                        Polar/Hydra Research Group
;-
;=============================================================================
if n_elements(fill_in) eq 0 then fill=-1.e+31 else fill=fill_in

if n_elements(zarr) lt 5 then message, 'Z must have at least 5 elements'

; 2-dimensional case-------------------------------
if ((size(zarr))(0) eq 2) then begin
    
    if (n_elements(wrt_x) eq 0 and n_elements(wrt_y) eq 0) then $
      message, $
      'Must set either "wrt_x" or "wrt_y" keyword for a 2_D array input'
    
    nxg = n_elements(zarr(*,0))
    nyg = n_elements(zarr(0,*))
    
    if keyword_set(wrt_x) then begin
        
; Derivatives in the X-direction, dz/dx
        
        if ((size(varr_in))(0) eq 2) then begin
            if (n_elements(varr_in(*,0)) ne nxg or $
                n_elements(varr_in(0,*)) ne nyg) then $
              message, 'V must have the same dimensions as Z'
            varr = varr_in
        endif else if ((size(varr_in))(0) eq 1) then begin
            if (n_elements(varr_in) ne nxg) then $
              message, 'V must have the same dimension as Z(*,0)'
            varr = varr_in#make_array(nyg,val=1)
        endif else begin
            message, 'V must have either the same dimensions as Z, or the same dimension as Z(*,0)', /cont
        endelse
        
        ix = lindgen(nxg-4)+2   ; [2,3,4,5,...,nxg-4,nxg-3]
        
        deriv = make_array(nxg,nyg,/float,value=0.)
        deriv([0,1,nxg-2,nxg-1],*) = fill
        deriv(ix,*) = $
          zarr(ix-2,*)*(varr(ix,*)-varr(ix,*))*(varr(ix,*)-varr(ix+1,*))*(varr(ix,*)-varr(ix+2,*))/(varr(ix-2,*)-varr(ix-1,*))/(varr(ix-2,*)-varr(ix,*))/(varr(ix-2,*)-varr(ix+1,*))/(varr(ix-2,*)-varr(ix+2,*))+ $
          zarr(ix-2,*)*(varr(ix,*)-varr(ix-1,*))*(varr(ix,*)-varr(ix+1,*))*(varr(ix,*)-varr(ix+2,*))/(varr(ix-2,*)-varr(ix-1,*))/(varr(ix-2,*)-varr(ix,*))/(varr(ix-2,*)-varr(ix+1,*))/(varr(ix-2,*)-varr(ix+2,*))+ $
          zarr(ix-2,*)*(varr(ix,*)-varr(ix-1,*))*(varr(ix,*)-varr(ix,*))*(varr(ix,*)-varr(ix+2,*))/(varr(ix-2,*)-varr(ix-1,*))/(varr(ix-2,*)-varr(ix,*))/(varr(ix-2,*)-varr(ix+1,*))/(varr(ix-2,*)-varr(ix+2,*))+ $
          zarr(ix-2,*)*(varr(ix,*)-varr(ix-1,*))*(varr(ix,*)-varr(ix,*))*(varr(ix,*)-varr(ix+1,*))/(varr(ix-2,*)-varr(ix-1,*))/(varr(ix-2,*)-varr(ix,*))/(varr(ix-2,*)-varr(ix+1,*))/(varr(ix-2,*)-varr(ix+2,*))+ $
          zarr(ix-1,*)*(varr(ix,*)-varr(ix,*))*(varr(ix,*)-varr(ix+1,*))*(varr(ix,*)-varr(ix+2,*))/(varr(ix-1,*)-varr(ix-2,*))/(varr(ix-1,*)-varr(ix,*))/(varr(ix-1,*)-varr(ix+1,*))/(varr(ix-1,*)-varr(ix+2,*))+ $
          zarr(ix-1,*)*(varr(ix,*)-varr(ix-2,*))*(varr(ix,*)-varr(ix+1,*))*(varr(ix,*)-varr(ix+2,*))/(varr(ix-1,*)-varr(ix-2,*))/(varr(ix-1,*)-varr(ix,*))/(varr(ix-1,*)-varr(ix+1,*))/(varr(ix-1,*)-varr(ix+2,*))+ $
          zarr(ix-1,*)*(varr(ix,*)-varr(ix-2,*))*(varr(ix,*)-varr(ix,*))*(varr(ix,*)-varr(ix+2,*))/(varr(ix-1,*)-varr(ix-2,*))/(varr(ix-1,*)-varr(ix,*))/(varr(ix-1,*)-varr(ix+1,*))/(varr(ix-1,*)-varr(ix+2,*))+ $
          zarr(ix-1,*)*(varr(ix,*)-varr(ix-2,*))*(varr(ix,*)-varr(ix,*))*(varr(ix,*)-varr(ix+1,*))/(varr(ix-1,*)-varr(ix-2,*))/(varr(ix-1,*)-varr(ix,*))/(varr(ix-1,*)-varr(ix+1,*))/(varr(ix-1,*)-varr(ix+2,*))+ $
          zarr(ix,*)*(varr(ix,*)-varr(ix-1,*))*(varr(ix,*)-varr(ix+1,*))*(varr(ix,*)-varr(ix+2,*))/(varr(ix,*)-varr(ix-2,*))/(varr(ix,*)-varr(ix-1,*))/(varr(ix,*)-varr(ix+1,*))/(varr(ix,*)-varr(ix+2,*))+ $
          zarr(ix,*)*(varr(ix,*)-varr(ix-2,*))*(varr(ix,*)-varr(ix+1,*))*(varr(ix,*)-varr(ix+2,*))/(varr(ix,*)-varr(ix-2,*))/(varr(ix,*)-varr(ix-1,*))/(varr(ix,*)-varr(ix+1,*))/(varr(ix,*)-varr(ix+2,*))+ $
          zarr(ix,*)*(varr(ix,*)-varr(ix-2,*))*(varr(ix,*)-varr(ix-1,*))*(varr(ix,*)-varr(ix+2,*))/(varr(ix,*)-varr(ix-2,*))/(varr(ix,*)-varr(ix-1,*))/(varr(ix,*)-varr(ix+1,*))/(varr(ix,*)-varr(ix+2,*))+ $
          zarr(ix,*)*(varr(ix,*)-varr(ix-2,*))*(varr(ix,*)-varr(ix-1,*))*(varr(ix,*)-varr(ix+1,*))/(varr(ix,*)-varr(ix-2,*))/(varr(ix,*)-varr(ix-1,*))/(varr(ix,*)-varr(ix+1,*))/(varr(ix,*)-varr(ix+2,*))+ $
          zarr(ix+1,*)*(varr(ix,*)-varr(ix-1,*))*(varr(ix,*)-varr(ix,*))*(varr(ix,*)-varr(ix+2,*))/(varr(ix+1,*)-varr(ix-2,*))/(varr(ix+1,*)-varr(ix-1,*))/(varr(ix+1,*)-varr(ix,*))/(varr(ix+1,*)-varr(ix+2,*))+ $
          zarr(ix+1,*)*(varr(ix,*)-varr(ix-2,*))*(varr(ix,*)-varr(ix,*))*(varr(ix,*)-varr(ix+2,*))/(varr(ix+1,*)-varr(ix-2,*))/(varr(ix+1,*)-varr(ix-1,*))/(varr(ix+1,*)-varr(ix,*))/(varr(ix+1,*)-varr(ix+2,*))+ $
          zarr(ix+1,*)*(varr(ix,*)-varr(ix-2,*))*(varr(ix,*)-varr(ix-1,*))*(varr(ix,*)-varr(ix+2,*))/(varr(ix+1,*)-varr(ix-2,*))/(varr(ix+1,*)-varr(ix-1,*))/(varr(ix+1,*)-varr(ix,*))/(varr(ix+1,*)-varr(ix+2,*))+ $
          zarr(ix+1,*)*(varr(ix,*)-varr(ix-2,*))*(varr(ix,*)-varr(ix-1,*))*(varr(ix,*)-varr(ix,*))/(varr(ix+1,*)-varr(ix-2,*))/(varr(ix+1,*)-varr(ix-1,*))/(varr(ix+1,*)-varr(ix,*))/(varr(ix+1,*)-varr(ix+2,*))+ $
          zarr(ix+2,*)*(varr(ix,*)-varr(ix-1,*))*(varr(ix,*)-varr(ix,*))*(varr(ix,*)-varr(ix+1,*))/(varr(ix+2,*)-varr(ix-2,*))/(varr(ix+2,*)-varr(ix-1,*))/(varr(ix+2,*)-varr(ix,*))/(varr(ix+2,*)-varr(ix+1,*))+ $
          zarr(ix+2,*)*(varr(ix,*)-varr(ix-2,*))*(varr(ix,*)-varr(ix,*))*(varr(ix,*)-varr(ix+1,*))/(varr(ix+2,*)-varr(ix-2,*))/(varr(ix+2,*)-varr(ix-1,*))/(varr(ix+2,*)-varr(ix,*))/(varr(ix+2,*)-varr(ix+1,*))+ $
          zarr(ix+2,*)*(varr(ix,*)-varr(ix-2,*))*(varr(ix,*)-varr(ix-1,*))*(varr(ix,*)-varr(ix+1,*))/(varr(ix+2,*)-varr(ix-2,*))/(varr(ix+2,*)-varr(ix-1,*))/(varr(ix+2,*)-varr(ix,*))/(varr(ix+2,*)-varr(ix+1,*))+ $
          zarr(ix+2,*)*(varr(ix,*)-varr(ix-2,*))*(varr(ix,*)-varr(ix-1,*))*(varr(ix,*)-varr(ix,*))/(varr(ix+2,*)-varr(ix-2,*))/(varr(ix+2,*)-varr(ix-1,*))/(varr(ix+2,*)-varr(ix,*))/(varr(ix+2,*)-varr(ix+1,*))
        
    endif else begin
        
; Derivatives in the Y-direction, dz/dy

        if ((size(varr_in))(0) eq 2) then begin
            if (n_elements(varr_in(*,0)) ne nxg or $
                n_elements(varr_in(0,*)) ne nyg) then $
              message, 'V must have the same dimensions as Z'
            varr = varr_in
        endif else if ((size(varr_in))(0) eq 1) then begin
            if (n_elements(varr_in) ne nyg) then $
              message, 'V must have the same dimension as Z(0,*)'
            varr = make_array(nxg,val=1)#varr_in
        endif else begin
            message, 'V must have either the same dimensions as Z, or the same dimension as Z(0,*)', /cont
        endelse

        iy = lindgen(nyg-4)+2   ; [2,3,4,5,...,nyg-4,nyg-3]
        
        deriv = make_array(nxg,nyg,/float,value=0.)
        deriv(*,[0,1,nyg-2,nyg-1]) = fill
        deriv(*,iy) = $
          zarr(*,iy-2)*(varr(*,iy)-varr(*,iy))*(varr(*,iy)-varr(*,iy+1))*(varr(*,iy)-varr(*,iy+2))/(varr(*,iy-2)-varr(*,iy-1))/(varr(*,iy-2)-varr(*,iy))/(varr(*,iy-2)-varr(*,iy+1))/(varr(*,iy-2)-varr(*,iy+2))+ $
          zarr(*,iy-2)*(varr(*,iy)-varr(*,iy-1))*(varr(*,iy)-varr(*,iy+1))*(varr(*,iy)-varr(*,iy+2))/(varr(*,iy-2)-varr(*,iy-1))/(varr(*,iy-2)-varr(*,iy))/(varr(*,iy-2)-varr(*,iy+1))/(varr(*,iy-2)-varr(*,iy+2))+ $
          zarr(*,iy-2)*(varr(*,iy)-varr(*,iy-1))*(varr(*,iy)-varr(*,iy))*(varr(*,iy)-varr(*,iy+2))/(varr(*,iy-2)-varr(*,iy-1))/(varr(*,iy-2)-varr(*,iy))/(varr(*,iy-2)-varr(*,iy+1))/(varr(*,iy-2)-varr(*,iy+2))+ $
          zarr(*,iy-2)*(varr(*,iy)-varr(*,iy-1))*(varr(*,iy)-varr(*,iy))*(varr(*,iy)-varr(*,iy+1))/(varr(*,iy-2)-varr(*,iy-1))/(varr(*,iy-2)-varr(*,iy))/(varr(*,iy-2)-varr(*,iy+1))/(varr(*,iy-2)-varr(*,iy+2))+ $
          zarr(*,iy-1)*(varr(*,iy)-varr(*,iy))*(varr(*,iy)-varr(*,iy+1))*(varr(*,iy)-varr(*,iy+2))/(varr(*,iy-1)-varr(*,iy-2))/(varr(*,iy-1)-varr(*,iy))/(varr(*,iy-1)-varr(*,iy+1))/(varr(*,iy-1)-varr(*,iy+2))+ $
          zarr(*,iy-1)*(varr(*,iy)-varr(*,iy-2))*(varr(*,iy)-varr(*,iy+1))*(varr(*,iy)-varr(*,iy+2))/(varr(*,iy-1)-varr(*,iy-2))/(varr(*,iy-1)-varr(*,iy))/(varr(*,iy-1)-varr(*,iy+1))/(varr(*,iy-1)-varr(*,iy+2))+ $
          zarr(*,iy-1)*(varr(*,iy)-varr(*,iy-2))*(varr(*,iy)-varr(*,iy))*(varr(*,iy)-varr(*,iy+2))/(varr(*,iy-1)-varr(*,iy-2))/(varr(*,iy-1)-varr(*,iy))/(varr(*,iy-1)-varr(*,iy+1))/(varr(*,iy-1)-varr(*,iy+2))+ $
          zarr(*,iy-1)*(varr(*,iy)-varr(*,iy-2))*(varr(*,iy)-varr(*,iy))*(varr(*,iy)-varr(*,iy+1))/(varr(*,iy-1)-varr(*,iy-2))/(varr(*,iy-1)-varr(*,iy))/(varr(*,iy-1)-varr(*,iy+1))/(varr(*,iy-1)-varr(*,iy+2))+ $
          zarr(*,iy)*(varr(*,iy)-varr(*,iy-1))*(varr(*,iy)-varr(*,iy+1))*(varr(*,iy)-varr(*,iy+2))/(varr(*,iy)-varr(*,iy-2))/(varr(*,iy)-varr(*,iy-1))/(varr(*,iy)-varr(*,iy+1))/(varr(*,iy)-varr(*,iy+2))+ $
          zarr(*,iy)*(varr(*,iy)-varr(*,iy-2))*(varr(*,iy)-varr(*,iy+1))*(varr(*,iy)-varr(*,iy+2))/(varr(*,iy)-varr(*,iy-2))/(varr(*,iy)-varr(*,iy-1))/(varr(*,iy)-varr(*,iy+1))/(varr(*,iy)-varr(*,iy+2))+ $
          zarr(*,iy)*(varr(*,iy)-varr(*,iy-2))*(varr(*,iy)-varr(*,iy-1))*(varr(*,iy)-varr(*,iy+2))/(varr(*,iy)-varr(*,iy-2))/(varr(*,iy)-varr(*,iy-1))/(varr(*,iy)-varr(*,iy+1))/(varr(*,iy)-varr(*,iy+2))+ $
          zarr(*,iy)*(varr(*,iy)-varr(*,iy-2))*(varr(*,iy)-varr(*,iy-1))*(varr(*,iy)-varr(*,iy+1))/(varr(*,iy)-varr(*,iy-2))/(varr(*,iy)-varr(*,iy-1))/(varr(*,iy)-varr(*,iy+1))/(varr(*,iy)-varr(*,iy+2))+ $
          zarr(*,iy+1)*(varr(*,iy)-varr(*,iy-1))*(varr(*,iy)-varr(*,iy))*(varr(*,iy)-varr(*,iy+2))/(varr(*,iy+1)-varr(*,iy-2))/(varr(*,iy+1)-varr(*,iy-1))/(varr(*,iy+1)-varr(*,iy))/(varr(*,iy+1)-varr(*,iy+2))+ $
          zarr(*,iy+1)*(varr(*,iy)-varr(*,iy-2))*(varr(*,iy)-varr(*,iy))*(varr(*,iy)-varr(*,iy+2))/(varr(*,iy+1)-varr(*,iy-2))/(varr(*,iy+1)-varr(*,iy-1))/(varr(*,iy+1)-varr(*,iy))/(varr(*,iy+1)-varr(*,iy+2))+ $
          zarr(*,iy+1)*(varr(*,iy)-varr(*,iy-2))*(varr(*,iy)-varr(*,iy-1))*(varr(*,iy)-varr(*,iy+2))/(varr(*,iy+1)-varr(*,iy-2))/(varr(*,iy+1)-varr(*,iy-1))/(varr(*,iy+1)-varr(*,iy))/(varr(*,iy+1)-varr(*,iy+2))+ $
          zarr(*,iy+1)*(varr(*,iy)-varr(*,iy-2))*(varr(*,iy)-varr(*,iy-1))*(varr(*,iy)-varr(*,iy))/(varr(*,iy+1)-varr(*,iy-2))/(varr(*,iy+1)-varr(*,iy-1))/(varr(*,iy+1)-varr(*,iy))/(varr(*,iy+1)-varr(*,iy+2))+ $
          zarr(*,iy+2)*(varr(*,iy)-varr(*,iy-1))*(varr(*,iy)-varr(*,iy))*(varr(*,iy)-varr(*,iy+1))/(varr(*,iy+2)-varr(*,iy-2))/(varr(*,iy+2)-varr(*,iy-1))/(varr(*,iy+2)-varr(*,iy))/(varr(*,iy+2)-varr(*,iy+1))+ $
          zarr(*,iy+2)*(varr(*,iy)-varr(*,iy-2))*(varr(*,iy)-varr(*,iy))*(varr(*,iy)-varr(*,iy+1))/(varr(*,iy+2)-varr(*,iy-2))/(varr(*,iy+2)-varr(*,iy-1))/(varr(*,iy+2)-varr(*,iy))/(varr(*,iy+2)-varr(*,iy+1))+ $
          zarr(*,iy+2)*(varr(*,iy)-varr(*,iy-2))*(varr(*,iy)-varr(*,iy-1))*(varr(*,iy)-varr(*,iy+1))/(varr(*,iy+2)-varr(*,iy-2))/(varr(*,iy+2)-varr(*,iy-1))/(varr(*,iy+2)-varr(*,iy))/(varr(*,iy+2)-varr(*,iy+1))+ $
          zarr(*,iy+2)*(varr(*,iy)-varr(*,iy-2))*(varr(*,iy)-varr(*,iy-1))*(varr(*,iy)-varr(*,iy))/(varr(*,iy+2)-varr(*,iy-2))/(varr(*,iy+2)-varr(*,iy-1))/(varr(*,iy+2)-varr(*,iy))/(varr(*,iy+2)-varr(*,iy+1))


    endelse

endif else if ((size(zarr))(0) eq 1) then begin

    nxg = n_elements(zarr)
    if (n_elements(varr_in) ne nxg) then $
      message, 'V must have the same dimension as Z'
    varr = varr_in
    
; Derivatives in the X-direction, dz/dx
    ix = lindgen(nxg-4)+2       ; [2,3,4,5,...,nxg-4,nxg-3]

    deriv = make_array(nxg,/float,value=0.)
    deriv([0,1,nxg-2,nxg-1]) = fill
    deriv(ix) = $
      zarr(ix-2)*(varr(ix)-varr(ix))*(varr(ix)-varr(ix+1))*(varr(ix)-varr(ix+2))/(varr(ix-2)-varr(ix-1))/(varr(ix-2)-varr(ix))/(varr(ix-2)-varr(ix+1))/(varr(ix-2)-varr(ix+2))+ $
      zarr(ix-2)*(varr(ix)-varr(ix-1))*(varr(ix)-varr(ix+1))*(varr(ix)-varr(ix+2))/(varr(ix-2)-varr(ix-1))/(varr(ix-2)-varr(ix))/(varr(ix-2)-varr(ix+1))/(varr(ix-2)-varr(ix+2))+ $
          zarr(ix-2)*(varr(ix)-varr(ix-1))*(varr(ix)-varr(ix))*(varr(ix)-varr(ix+2))/(varr(ix-2)-varr(ix-1))/(varr(ix-2)-varr(ix))/(varr(ix-2)-varr(ix+1))/(varr(ix-2)-varr(ix+2))+ $
      zarr(ix-2)*(varr(ix)-varr(ix-1))*(varr(ix)-varr(ix))*(varr(ix)-varr(ix+1))/(varr(ix-2)-varr(ix-1))/(varr(ix-2)-varr(ix))/(varr(ix-2)-varr(ix+1))/(varr(ix-2)-varr(ix+2))+ $
      zarr(ix-1)*(varr(ix)-varr(ix))*(varr(ix)-varr(ix+1))*(varr(ix)-varr(ix+2))/(varr(ix-1)-varr(ix-2))/(varr(ix-1)-varr(ix))/(varr(ix-1)-varr(ix+1))/(varr(ix-1)-varr(ix+2))+ $
      zarr(ix-1)*(varr(ix)-varr(ix-2))*(varr(ix)-varr(ix+1))*(varr(ix)-varr(ix+2))/(varr(ix-1)-varr(ix-2))/(varr(ix-1)-varr(ix))/(varr(ix-1)-varr(ix+1))/(varr(ix-1)-varr(ix+2))+ $
      zarr(ix-1)*(varr(ix)-varr(ix-2))*(varr(ix)-varr(ix))*(varr(ix)-varr(ix+2))/(varr(ix-1)-varr(ix-2))/(varr(ix-1)-varr(ix))/(varr(ix-1)-varr(ix+1))/(varr(ix-1)-varr(ix+2))+ $
      zarr(ix-1)*(varr(ix)-varr(ix-2))*(varr(ix)-varr(ix))*(varr(ix)-varr(ix+1))/(varr(ix-1)-varr(ix-2))/(varr(ix-1)-varr(ix))/(varr(ix-1)-varr(ix+1))/(varr(ix-1)-varr(ix+2))+ $
      zarr(ix)*(varr(ix)-varr(ix-1))*(varr(ix)-varr(ix+1))*(varr(ix)-varr(ix+2))/(varr(ix)-varr(ix-2))/(varr(ix)-varr(ix-1))/(varr(ix)-varr(ix+1))/(varr(ix)-varr(ix+2))+ $
      zarr(ix)*(varr(ix)-varr(ix-2))*(varr(ix)-varr(ix+1))*(varr(ix)-varr(ix+2))/(varr(ix)-varr(ix-2))/(varr(ix)-varr(ix-1))/(varr(ix)-varr(ix+1))/(varr(ix)-varr(ix+2))+ $
      zarr(ix)*(varr(ix)-varr(ix-2))*(varr(ix)-varr(ix-1))*(varr(ix)-varr(ix+2))/(varr(ix)-varr(ix-2))/(varr(ix)-varr(ix-1))/(varr(ix)-varr(ix+1))/(varr(ix)-varr(ix+2))+ $
      zarr(ix)*(varr(ix)-varr(ix-2))*(varr(ix)-varr(ix-1))*(varr(ix)-varr(ix+1))/(varr(ix)-varr(ix-2))/(varr(ix)-varr(ix-1))/(varr(ix)-varr(ix+1))/(varr(ix)-varr(ix+2))+ $
      zarr(ix+1)*(varr(ix)-varr(ix-1))*(varr(ix)-varr(ix))*(varr(ix)-varr(ix+2))/(varr(ix+1)-varr(ix-2))/(varr(ix+1)-varr(ix-1))/(varr(ix+1)-varr(ix))/(varr(ix+1)-varr(ix+2))+ $
      zarr(ix+1)*(varr(ix)-varr(ix-2))*(varr(ix)-varr(ix))*(varr(ix)-varr(ix+2))/(varr(ix+1)-varr(ix-2))/(varr(ix+1)-varr(ix-1))/(varr(ix+1)-varr(ix))/(varr(ix+1)-varr(ix+2))+ $
      zarr(ix+1)*(varr(ix)-varr(ix-2))*(varr(ix)-varr(ix-1))*(varr(ix)-varr(ix+2))/(varr(ix+1)-varr(ix-2))/(varr(ix+1)-varr(ix-1))/(varr(ix+1)-varr(ix))/(varr(ix+1)-varr(ix+2))+ $
      zarr(ix+1)*(varr(ix)-varr(ix-2))*(varr(ix)-varr(ix-1))*(varr(ix)-varr(ix))/(varr(ix+1)-varr(ix-2))/(varr(ix+1)-varr(ix-1))/(varr(ix+1)-varr(ix))/(varr(ix+1)-varr(ix+2))+ $
      zarr(ix+2)*(varr(ix)-varr(ix-1))*(varr(ix)-varr(ix))*(varr(ix)-varr(ix+1))/(varr(ix+2)-varr(ix-2))/(varr(ix+2)-varr(ix-1))/(varr(ix+2)-varr(ix))/(varr(ix+2)-varr(ix+1))+ $
      zarr(ix+2)*(varr(ix)-varr(ix-2))*(varr(ix)-varr(ix))*(varr(ix)-varr(ix+1))/(varr(ix+2)-varr(ix-2))/(varr(ix+2)-varr(ix-1))/(varr(ix+2)-varr(ix))/(varr(ix+2)-varr(ix+1))+ $
      zarr(ix+2)*(varr(ix)-varr(ix-2))*(varr(ix)-varr(ix-1))*(varr(ix)-varr(ix+1))/(varr(ix+2)-varr(ix-2))/(varr(ix+2)-varr(ix-1))/(varr(ix+2)-varr(ix))/(varr(ix+2)-varr(ix+1))+ $
      zarr(ix+2)*(varr(ix)-varr(ix-2))*(varr(ix)-varr(ix-1))*(varr(ix)-varr(ix))/(varr(ix+2)-varr(ix-2))/(varr(ix+2)-varr(ix-1))/(varr(ix+2)-varr(ix))/(varr(ix+2)-varr(ix+1))
    
endif else begin
    
    message, 'Z can have only 1 or 2 dimensions at this time'

endelse

return
end
