PRO find_linreg_sigma, nparams, mm, theta_p, error_b
;+
; NAME: find_linreg_sigma.pro
; WRITTEN: 10/8/98, Pamela A. Puhl-Quinn, ppq@space-theory.physics.uiowa.edu
; PURPOSE:  Find the errors associated with linear regression fit
;           parameters.  The linear system is:
;                    mm * b = a
;           where mm(nparams,nparams) and a(nparams) are known, and
;           b(nparams) is solved for using a linear system solver.  This
;           routine finds the errors in b, error_b.
;
; USAGE:    find_linreg_sigma, nparams, mm, theta_p, error_b
;
; INPUT:    nparams             # of parameters in the fit
;           mm(nparams,nparams) Normal Equations Matrix, mm(nparams,nparams)
;           theta_p             = nparams*red_chi_sq for a particular
;                                 confidence level (see Bevington)
;                    (e.g. theta_p = 2.3 for nparams=2, 68% confidence,
;                          theta_p = 3.5 for nparams=3, 68% confidence,
;                          theta_p = 7.81 for nparams=3, 95% confidence,
;                          theta_p = 1.65 for nparams=4, 20% confidence,
;                          etc...)
; OUTPUT:   error_b(nparams)    Errors in b

; NOTE: If error_b(i) = -1.e+31, the matrix inversion failed.
;
;-
;*************** EVERYTHING IS DOUBLE PRECISION ********************

   nparams = long(nparams)
   mm = double(mm)
   theta_p = double(theta_p)

   error_b = dblarr(nparams)
   nn = dblarr(nparams,nparams)
   nn_temp = dblarr(nparams,nparams)
   imatrix = dblarr(nparams,nparams)
   kd = lonarr(nparams,nparams)
   tau = dblarr(nparams)
   sv = dblarr(nparams)
   i = long(0)
   j = long(0)
   k = long(0)
   iparam = long(0)
   iretcode = long(0)
   ii = long(0)
   sum = double(0)
   sum0 = double(0)

; Calculate the Kronecker delta
   FOR i = 0,nparams-1 DO BEGIN 
      FOR j = 0,nparams-1 DO BEGIN 
         kd(i,j)=0
      ENDFOR 
      kd(i,i)=1
   ENDFOR 
   
; Choose a parameter
   FOR iparam = 0,nparams-1 DO BEGIN 

; Calculate the nn matrix
      FOR i=0,nparams-1 DO BEGIN 
         FOR j=0,nparams-1 DO BEGIN 
            
;            nn(i,j)=mm(i,j)*(1-kd(i,iparam)-kd(j,iparam)) $
;             +(1.+mm(iparam,iparam))*kd(i,j)*kd(i,iparam)*kd(j,iparam)
;
;           Recoded 9/6/00 for cancellation
;
            nn(i,j)=(mm(i,j)*(1-kd(i,iparam)-kd(j,iparam)) $
             +mm(iparam,iparam)*kd(i,j)*kd(i,iparam)*kd(j,iparam)) $
             +kd(i,j)*kd(i,iparam)*kd(j,iparam)

            nn_temp(i,j)=nn(i,j)
            
         ENDFOR 
      ENDFOR 
      
; Invert the nn matrix
      imatrix = invert(nn_temp,iretcode,/double)
      IF (iretcode NE 0) THEN begin
          error_b(iparam) = -1.e+31
          goto, nextparam
      endif
      
; Check the inverse:
;      message, 'Check inverse: ', /cont
      FOR i=0,nparams-1 DO BEGIN 
         FOR j=0,nparams-1 DO BEGIN 
            sum=0
            FOR k=0,nparams-1 DO BEGIN 
               sum=sum + nn(i,k)*imatrix(k,j)
            ENDFOR 
;            print,i,j,sum
         ENDFOR 
      ENDFOR 
      
; Calculate the tau vector
;      message, 'tau: ', /cont
      FOR i=0,nparams-1 DO BEGIN 
         tau(i)=mm(i,iparam)*(kd(i,iparam)-1)+kd(i,iparam)
;         print, i,tau(i)
      ENDFOR 
      
; Calculate the solution vector
;      message, 'solution vector: ', /cont
      FOR i=0,nparams-1 DO BEGIN 
         sum=0.
         FOR j=0,nparams-1 DO BEGIN 
            sum=sum + imatrix(i,j)*tau(j)
         ENDFOR 
         sv(i) = sum
;         print,i,sv(i)
      ENDFOR 
      
; Calculate the error for this parameter
      sum=0.d0
      FOR i=0,nparams-1 DO BEGIN 
         sum0 = 0.d0
         FOR j=0,nparams-1 DO BEGIN 
            sum0 = sum0 + mm(i,j)*sv(j)
         ENDFOR 
         sum = sum + sum0*sv(i)
      ENDFOR 
;      message, 'sum: ', /cont
;      print, sum
      
      error_b(iparam)=sqrt(theta_p/sum)

      nextparam:
      
;      message, 'iparam, error: ', /cont
;      print, iparam, error_b(iparam)
      
   ENDFOR
   
   return
END 

            
      

