PRO generalized_least_squares_fit, g, sigma_g, nparams, $
                                   f0, f1, f2, f3, f4, $
                                   f5, f6, f7, f8, f9, $
                                   b=b, sigma_b=sigma_b, rchi2=rchi2, $
                                   istat=istat, $
                                   inv_accuracy=inv_accuracy
;+
; NAME: generalized_least_squares_fit.pro
; WRITTEN: 10/8/98, Pamela A. Puhl-Quinn, ppq@space-theory.physics.uiowa.edu
;          10/13/98, sigma_b and inv_accuracy must be PREDEFINED to be
;                    returned
;          10/16/98, Refined the logic regarding determining errors
;                      when reduced chisq is < 1.  Renormalize the chisq
;                      surface so that the minimum value is on the
;                      order of the # of degrees of freedom, THEN
;                      find errors on the fit parameters from this temporary
;                      chisq surface.  This logic applies to the
;                      determination of sigma_b.
;
; PURPOSE: Perform a generalized, LINEAR least-squares fit to data which 
;          consists of nparams basis functions (f_0, f_1, f_2, etc.), 
;          and one dependent variable (g).  Errors in g (sigma_g) 
;          are also used to weight the problem.  The general form of
;          the problem to solve is:
;
;    g = b(0)*f_0 + b(1)*f_1 + b(2)*f_2 + ... + b(nparams-1)*f_(nparams-1)
;
; Notice that the problem is LINEAR in the fit coefficients b(0:nparams-1)
;
; USAGE:   sigma_b = 1 ; MUST BE PREDEFINED in order to be returned
;          inv_accuracy = 1 ;  MUST BE PREDEFINED in order to be returned
;          generalized_least_squares_fit, g, sigma_g, nparams, $
;                                         f_0, f_1, f_2, ... , f_(nparams-1), $
;                                         b=b, sigma_b=sigma_b, rchi2=rchi2, $
;                                         istat=istat, $
;                                         inv_accuracy=inv_accuracy
;
; MANDATORY ARGUMENTS:
;       g(npts), sigma_g(npts)            Dependent variable, error
;       nparams                           # of free parameters in the fit
;                                         (also the # of basis functions)
;                                         Note: nparams cannot exceed 10
;       f_0(npts), f_1(npts), etc.        The basis functions (at least
;                                         one and not to exceed 10)
;
;
; OUTPUT KEYWORDS:
;       b=b                               Fit coefficients, b(nparams)
;       rchi2=rchi2                       Reduced chi-squared of the fit
;       istat=istat                       Status code
;                                          = 0, fit successful
;                                          = 1, singular matrix, fatal error
;                                          = 2, small pivot, fatal error
;
; NOTE: The following keywords must be PREDEFINED IN THE CALLING ROUTINE
;       (any nonzero scalar will do) in order to be returned
;
;       sigma_b =sigma_b                  Errors in the fit coefficients
;                                         (70% confidence level)
;       inv_accuracy=inv_accuracy         How accurate is the inverse?
;                                         (in # of significant digits)
; SHORT EXAMPLE CODES: ======================================================
;   MODEL: Third order polynomial in f_1 
;                      (i.e. g = b(0) + b(1)*f_1 + b(2)*f_1^2 + b(3)*f_1^3)
;   f_0 = dblarr(100) + 1.d0
;   f_1 = dindgen(100)/10.
;   f_2 = f_1^2
;   f_3 = f_1^3
;   seed = -2938785
;   g = 68.29 + 3.76*f_1 + 1.4*f_2 + 0.26*f_3 + 5.*randomn(seed,100)
;   sigma_g = dblarr(100) + 5.
;   sigma_b = 1 ; Must be PREDEFINED to be returned
;   inv_accuracy = 1 ; Must be PREDEFINED to be returned
;   generalized_least_squares_fit, g, sigma_g, 4, f_0, f_1, f_2, f_3, $
;    b=b, sigma_b=sigma_b, rchi2=rchi2, inv_accuracy=inv_accuracy, istat=istat
;   print, b
;   print, sigma_b
;   print, rchi2
;   print, inv_accuracy
;   print, istat
;   !p.multi = [0,1,1]
;   plot, f_1, g, psym=2, xtitle='f_1', ytitle='g'
;   errplot, f_1, g-sigma_g, g+sigma_g
;   oplot, f_1, f_3*b(3) + f_2*b(2) + f_1*b(1) + f_0*b(0), line=3
;
;   MODEL: Linear in f_0, intercept=0
;                      (i.e. g = b(0)*f_0)
;   f_0 = dindgen(100)
;   seed = -2938785
;   g = f_0*3.76 + 20.*randomn(seed,100)
;   sigma_g = dblarr(100) + 20.
;   sigma_b = 1 ; Must be PREDEFINED to be returned
;   inv_accuracy = 1 ; Must be PREDEFINED to be returned
;   generalized_least_squares_fit, g, sigma_g, 1, f_0, $
;    b=b, sigma_b=sigma_b, rchi2=rchi2, inv_accuracy=inv_accuracy, istat=istat
;   print, b
;   print, sigma_b
;   print, rchi2
;   print, inv_accuracy
;   print, istat
;   !p.multi = [0,1,1]
;   plot, f_0, g, psym=2, ytitle='g', xtitle='f_0'
;   errplot, f_0, g-sigma_g, g+sigma_g
;   oplot, f_0, f_0*b, line=3
;-
;=======================================================
; EVERYTHING IN THIS CODE IS DONE IN DOUBLE PRECISION
;=======================================================
   
   IF (n_params() EQ 0) THEN BEGIN
      doc_library, 'generalized_least_squares_fit'
      return
   ENDIF 

   nbasis = n_params() - 3
   IF (nbasis NE nparams) THEN message, 'nparams is not equal to the number of basis functions supplied'
   
   istat = 0
   ndata = n_elements(g)
   g = double(g)
   sigma_g = double(sigma_g)

; Calculate the weight
   nezero = where(sigma_g NE 0.)
   weight = dblarr(ndata)
   weight(nezero) = 1./sigma_g(nezero)^2
   
   f = dblarr(nparams,ndata)
   f(0,*) = double(f0)
   IF (nparams EQ 1) THEN BEGIN ; Slope only
      b = total( g*f(0,*)*weight, /double) / total( f(0,*)^2*weight, /double )
      gmodel = b*reform(f(0,*))
      rchi2 = total( weight*(g - gmodel)^2 , /double )
      rchi2 = rchi2/float(ndata - nparams - 1)
      IF (rchi2 LT 1) THEN renorm = rchi2 ELSE renorm = 1.
      IF (keyword_set(sigma_b)) THEN sigma_b = $
       1.d0/sqrt( total( f(0,*)^2*weight, /double )/renorm )
      IF (keyword_set(inv_accuracy)) THEN inv_accuracy = 100
      return
   ENDIF 

   IF (nparams GT 1) THEN f(1,*) = double(f1)
   IF (nparams GT 2) THEN f(2,*) = double(f2)
   IF (nparams GT 3) THEN f(3,*) = double(f3)
   IF (nparams GT 4) THEN f(4,*) = double(f4)
   IF (nparams GT 5) THEN f(5,*) = double(f5)
   IF (nparams GT 6) THEN f(6,*) = double(f6)
   IF (nparams GT 7) THEN f(7,*) = double(f7)
   IF (nparams GT 8) THEN f(8,*) = double(f8)
   IF (nparams GT 9) THEN f(9,*) = double(f9)
            
; Calculate the normal equations matrix and inhomogeneous vector
   matrix = dblarr(nparams,nparams)
   FOR irow=0,nparams-1 DO BEGIN
      FOR icol=0,nparams-1 DO BEGIN
         matrix(irow,icol) = total( f(irow,*)*f(icol,*)*weight, /double )
      ENDFOR
   ENDFOR
   vec = dblarr(nparams)
   FOR irow=0,nparams-1 DO BEGIN
      vec(irow) = total( f(irow,*)*g*weight, /double )
   ENDFOR

; Invert the matrix
   inverse_matrix = invert(matrix, inverse_status, /double)
   IF (inverse_status EQ 1) THEN BEGIN
      istat = 1
      message, 'Singular matrix, inversion invalid', /cont
      return
   ENDIF ELSE IF (inverse_status EQ 2) THEN BEGIN
      istat = 2
      message, 'Small pivot used in matrix inversion, accuracy lost', /cont
      return
   ENDIF 
   
   IF (keyword_set(inv_accuracy)) THEN BEGIN 
; To how many significant digits is the inverse accurate?
      imatrix = dblarr(nparams,nparams)
      FOR irow=0,nparams-1 DO BEGIN
         FOR icol=0,nparams-1 DO BEGIN
            imatrix(irow,icol) = 0.d0
         ENDFOR
         imatrix(irow,irow) = 1.d0
      ENDFOR 
      diff = 0.d0
      inv_accuracy = -100
      FOR irow=0,nparams-1 DO BEGIN
         FOR icol=0,nparams-1 DO BEGIN
            diff = abs( total( matrix(irow,*)*inverse_matrix(*,icol), /double ) $
                        - imatrix(irow,icol) )
            IF (diff NE 0.0) THEN BEGIN
               power = long(alog10(diff))
               inv_accuracy = max([inv_accuracy,power])
            ENDIF 
         ENDFOR
      ENDFOR
      inv_accuracy = abs(inv_accuracy)
   ENDIF

; Calculate the solution vector
   b = dblarr(nparams)
   FOR irow=0,nparams-1 DO BEGIN
      b(irow) = total( inverse_matrix(irow,*)*vec, /double )
   ENDFOR

; Calculate the reduced chi-squared of the fit
   gmodel = dblarr(ndata)
   FOR ii=0,nparams-1 DO BEGIN
      gmodel =  gmodel + b(ii)*reform(f(ii,*))
   ENDFOR 
   rchi2 = total( weight*(g - gmodel)^2 , /double )
   rchi2 = rchi2/float(ndata - nparams - 1)

   IF (keyword_set(sigma_b)) THEN BEGIN 
; Find sigma_b from the curvature matrix
; theta_p for 70% confidence:
      theta_p = [ 1.074*1., $
                  1.204*2., $
                  1.222*3., $
                  1.220*4., $
                  1.213*5., $
                  1.205*6., $
                  1.198*7., $
                  1.191*8., $
                  1.184*9., $
                  1.178*10. ]   ; Bevington, Table C-4
; Renormalize the curvature matrix if rchi2 < 1
      IF (rchi2 LT 1) THEN matrix_temp = matrix/rchi2 ELSE matrix_temp = matrix
      find_linreg_sigma, nparams, matrix_temp, theta_p(nparams-1), sigma_b
   ENDIF
   
   

   return
END





   
