;+
;NAME: get_tpphr_cdfdata_access.pro
;PURPOSE: Return the 't'en 'p'arameter 'p'atch 'h'igh 'r'esolution
;moments OR the 's'ix 'p'arameter 'p'atch 'h'igh 'r'esolution moments!!!
;
;AUTHOR: Pamela A. Puhl-Quinn
;USAGE: get_tpphr_cdfdata_access,'YYYYMMDD','tpphr_ele_time_S',varout,$
;                                path=path, vers=vers,$
;                                dstring=dstring,$
;                                error_num=error_num,$
;                                error_msg=error_msg,$
;                                silent=silent,$
;                                file_opened=file_opened
;
;       where the 'S' in the second argument identifies the number of
;       sweeps that are ganged together for the moment calculation.
;       '1' is the highest resolution (1.15s) and '6' is just block
;       resolution (13.8). Other possibilities are '2' and '3'.
;
;EXAMPLE: get_tpphr_cdfdata_access,'19960529','tpphr_ele_time_2',varout
;
;EXAMPLE2: get_tpphr_cdfdata_access,'19960529','spphr_ele_time_2',varout
; (YES!  This code handles the spphr data extraction too!!!)
;
; NOTES: If path and vers are not set, you'll get the highest version
; CDF in the default data area.  The default file opened will have
; a filename of the form:
;        DEFAULT_PATH/YYYYMMDD_hyd_tpphr_S_vN.NN.cdf
; where 'S' denotes nsweep, which can be 1, 2, 3 or 6
;
; The keyword 'dstring' allows cusp study data to be returned, which
; is generated using 54Hz lag-corrected bfield data rather than the 
; sweep-averaged 8Hz bfield data.
;        dstring = 'YYYYMMDD_HH_YYYYMMDD_HH'
; will open the file
; /opt/hydra/scratch1/quinn/YYYYMMDD_HH_YYYYMMDD_HH_hyd_tpphr_54hz_S_vN.NN.cdf
;
;-

pro get_tpphr_cdfdata_access, date_in, vstring_in, v_out, $
                              path=path_in, vers=vers_in, dstring=dstring, $
                              error_num=error_num, error_msg=error_msg, $
                              silent=silent, file_opened=file_opened

if (n_params() eq 0) then begin
    doc_library, 'get_tpphr_cdfdata_access'
    return
endif

if n_elements(v_out) ne 0 then dum = temporary(v_out)
if n_elements(file_opened) ne 0 then dum = temporary(file_opened)

silent = keyword_set(silent)
msghead = '% GET_TPPHR_CDFDATA_ACCESS: '

fp = [msghead]
fp = [fp, ' > '+snapshot_idl_common_ppq(/identify)]

error_msg = msghead+'Successful'
error_num = 0

@hd_include

;NOTE: This code also handles 'spphr_ele_time_2', etc...

;tpphr_ele_time_2          seconds since midnight
;tpphr_ele_time_doy96_2    decimal days since 00:00:00 12/31/1995
;tpphr_ele_time_epoch_2    milliseconds since 00:00:00 01/01/0000  (1 B.C.)
;tpphr_ele_dt_2            seconds

;tpphr_ele_patch_status_2
;tpphr_ele_npatch_2
;tpphr_ele_rchi2_2
;tpphr_ele_rchi2_all_2
;tpphr_ele_min_energy_2
;tpphr_ele_max_energy_2

;tpphr_ele_bhat_2
;tpphr_ele_bmag_2
;tpphr_ele_b_status_2
;tpphr_ele_bhat_errorcone_angle_2    degrees
;tpphr_ele_angle_b_2                 degrees

;tpphr_ele_density_2

;tpphr_ele_u_2
;tpphr_ele_uparl_2
;tpphr_ele_uperp_2

;tpphr_ele_kt_2
;tpphr_ele_ktparl_2
;tpphr_ele_ktperp_2       <----- avg(ktperps)
;tpphr_ele_ktperp1_2
;tpphr_ele_ktperp2_2

;tpphr_ele_gyrotropy_2    <----- abs(ktperp1-ktperp2)/avg(ktperps)
;tpphr_ele_anisotropy_2   <----- ktparl/avg(ktperps)
;tpphr_ele_beta_2         <----- nkT/(B^2/8pi)

; What is the input state?
ident0 = strmid(vstring_in,0,5)
spec_in = strmid(vstring_in,6,3)
nlen = strlen(vstring_in)
nsweep_in = strmid(vstring_in,nlen-1,1)
varonly_in = strmid(vstring_in,10,nlen-12)
default_path = getenv('HYDRA_TPPHR_DATA_PATH')
if n_elements(path_in) ne 0 then path=path_in+'/' else path=default_path
if n_elements(vers_in) ne 0 then vers=string(vers_in,'(f4.2)') else vers='?.??'

fp = [fp,' > Variable requested: '+vstring_in]

;---------------------------------------------------
; Find the CDF file
if (n_elements(dstring) ne 0) then begin
    fp = [fp,' > Date requested: '+dstring]
    file ='/opt/hydra/scratch1/quinn/'+dstring+'_hyd_'+ident0+'_54hz_'
endif else begin
    fp = [fp,' > Date requested: '+date_in]
    file = path+date_in+'_hyd_'+ident0+'_'
endelse

file = file+nsweep_in+'_v'+vers+'.cdf' 
f = findfile(file)
if f(0) eq '' then begin
    error_msg = msghead+'File not found: '+file
    error_num = 1
    if (not silent) then print, error_msg
    hd_add_footprint, error_msg
    return
endif else begin
    if (n_elements(f) ge 2) then begin
        nlen = strlen(f(0))
        fvers = float(strmid(f,nlen-8,4))
        maxv = max(fvers,imax)
        file = f(imax)
    endif else begin
        file = f(0)
    endelse
endelse

if (not keyword_set(silent)) then message, 'Opening: '+file, /cont
file_opened = file
id = cdf_open(file)

fp = [fp,' > File opened: '+file]

cdf_attget, id, 'RCS Snapshot Name',0,tpphr_code_version
fp = [fp,' > TPPHR f77 release code dir: /home/ppq/bigdisk/patching/ten_param_patch/hydra_highres/release', $
      ' > TPPHR f77 release code snapshot: '+tpphr_code_version]

cdf_varget, id, 'nrec', nrec, rec_start=0, rec_count=1

IF (nrec lt 1) THEN BEGIN
    error_msg = msghead+'No data found in CDF'
    error_num = 2
    if (not silent) then print, error_msg
    hd_add_footprint, error_msg
    return
ENDIF
rec_start = 0
rec_count = nrec

case varonly_in of
    'time':begin
        cdf_varget, id, spec_in+'_epoch', epoch, $
          rec_start=rec_start, rec_count=rec_count
        cdf_epoch, epoch(0), iyear,imonth,iday,ihour,iminute,isecond,imsec,$
          /breakdown_epoch
        cdf_epoch, mnight_epoch, iyear,imonth,iday,0,0,0,0,/compute_epoch
        var_out = (epoch - mnight_epoch)/1000.d0 ; ssm
    end
    'time_doy96':begin
        cdf_varget, id, spec_in+'_epoch', epoch, $
          rec_start=rec_start, rec_count=rec_count
        cdf_epoch, epoch(0), iyear,imonth,iday,ihour,iminute,isecond,imsec, $
          /breakdown_epoch
        dstr = string(iyear,'(i4.4)')+string(imonth,'(i2.2)')+$
          string(iday,'(i2.2)')
        total_time = double(ihour)*3600.d0 + double(iminute)*60.d0 + $
          double(isecond) + double(imsec)/1000.d0 ; seconds
        firstrec_doy96 = gen_time(date=dstr,time=total_time,/seconds)
        time_mssfr = epoch - epoch(0)   ; milliseconds since first record
        var_out = time_mssfr/1000.d0/60.d0/60.d0/24.d0 + firstrec_doy96 ; doy96
    end
    'time_epoch':cdf_varget, id, spec_in+'_epoch', var_out, $
      rec_start=rec_start, rec_count=rec_count
    'dt':begin
        case nsweep_in of
            '1':var_out=make_array(rec_count,/float, value=1.15)
            '2':var_out=make_array(rec_count,/float, value=3.45)
            '3':var_out=make_array(rec_count,/float, value=5.75)
            '6':var_out=make_array(rec_count,/float, value=12.65)
            else:message, 'nsweep value not recognized: '+nsweep_in
        endcase
    end
    'npatch':cdf_varget, id, spec_in+'_npatch', var_out, $
      rec_start=rec_start, rec_count=rec_count
    'rchi2':cdf_varget, id, spec_in+'_rchi2', var_out, $
      rec_start=rec_start, rec_count=rec_count
    'rchi2_all':cdf_varget, id, spec_in+'_rchi2_all', var_out, $
      rec_start=rec_start, rec_count=rec_count
    'bhat':begin
        cdf_varget, id, spec_in+'_bhat', var_out, $
          rec_start=rec_start, rec_count=rec_count
        var_out = transpose(var_out)
    end
    'bmag':cdf_varget, id, spec_in+'_bmag', var_out, $
      rec_start=rec_start, rec_count=rec_count
    'b_status':cdf_varget, id, spec_in+'_b_status', var_out, $
      rec_start=rec_start, rec_count=rec_count
    'bhat_errorcone_angle':cdf_varget, id, spec_in+'_bhat_errorcone_angle', $
      var_out, rec_start=rec_start, rec_count=rec_count
    'min_energy':cdf_varget, id, spec_in+'_min_energy', var_out, $
      rec_start=rec_start, rec_count=rec_count
    'max_energy':cdf_varget, id, spec_in+'_max_energy', var_out, $
      rec_start=rec_start, rec_count=rec_count
    'density':cdf_varget, id, spec_in+'_density', var_out, $
      rec_start=rec_start, rec_count=rec_count
    'kt':cdf_varget, id, spec_in+'_kt', var_out, $
      rec_start=rec_start, rec_count=rec_count
    'u':begin
        cdf_varget, id, spec_in+'_u', var_out, $
          rec_start=rec_start, rec_count=rec_count
        var_out = transpose(var_out)
    end
    'uperp':begin
        cdf_varget, id, spec_in+'_bhat', bhat, $
          rec_start=rec_start, rec_count=rec_count
        cdf_varget, id, spec_in+'_u', u, $
          rec_start=rec_start, rec_count=rec_count
        var_out = transpose(u) - $
          hd_svmult(hd_dot(transpose(u),transpose(bhat)),transpose(bhat))
        u = 0                   ; de-allocate
        bhat = 0                ; de-allocate
        cdf_varget, id, spec_in+'_b_status', b_status, $
          rec_start=rec_start, rec_count=rec_count
        ngd = where(b_status ne 1)
        b_status = 0            ; de-allocate
        if ngd(0) ne -1 then var_out(ngd,*) = hd_fill
    end
    'uparl':begin
        cdf_varget, id, spec_in+'_bhat', bhat, $
          rec_start=rec_start, rec_count=rec_count
        cdf_varget, id, spec_in+'_u', u, $
          rec_start=rec_start, rec_count=rec_count
        var_out = hd_dot(transpose(u),transpose(bhat))
        u = 0                   ; de-allocate
        bhat = 0                ; de-allocate
        cdf_varget, id, spec_in+'_b_status', b_status, $
          rec_start=rec_start, rec_count=rec_count
        ngd = where(b_status ne 1)
        b_status = 0            ; de-allocate
        if ngd(0) ne -1 then var_out(ngd) = hd_fill
    end
    'angle_b':cdf_varget, id, spec_in+'_angle_b', var_out, $
      rec_start=rec_start, rec_count=rec_count
    'ktparl':cdf_varget, id, spec_in+'_ktparl', var_out, $
      rec_start=rec_start, rec_count=rec_count
    'ktperp':begin
        cdf_varget, id, spec_in+'_ktperp1', ktp1, $
          rec_start=rec_start, rec_count=rec_count
        ktp1 = reform(ktp1)
        cdf_varget, id, spec_in+'_ktperp1', ktp2, $
          rec_start=rec_start, rec_count=rec_count
        ktp2 = reform(ktp2)
        var_out = make_array(rec_count,/float, value=hd_fill)
        gd = where(ktp1 ne hd_fill and ktp2 ne hd_fill)
        if (gd(0) ne -1) then var_out(gd) = (ktp1(gd)+ktp2(gd))/2.d0        
    end
    'ktperp1':cdf_varget, id, spec_in+'_ktperp1', var_out, $
      rec_start=rec_start, rec_count=rec_count
    'ktperp2':cdf_varget, id, spec_in+'_ktperp2', var_out, $
      rec_start=rec_start, rec_count=rec_count
    'gyrotropy':cdf_varget, id, spec_in+'_gyrotropy', var_out, $
      rec_start=rec_start, rec_count=rec_count
    'anisotropy':cdf_varget, id, spec_in+'_anisotropy', var_out, $
      rec_start=rec_start, rec_count=rec_count
    'beta':begin
        cdf_varget, id, spec_in+'_density', n, $
          rec_start=rec_start, rec_count=rec_count
        n = reform(n)
        cdf_varget, id, spec_in+'_kt', kt, $
          rec_start=rec_start, rec_count=rec_count
        kt = reform(kt)
        p = hd_ssmult(n,hd_svmult(1.6d-12,kt))
        n = 0                   ; de-allocate
        kt = 0                  ; de-allocate
        cdf_varget, id, spec_in+'_bmag', bmag, $
          rec_start=rec_start, rec_count=rec_count
        bmag = reform(bmag)
        bmag = hd_svmult(1.d-9*1.d+4,bmag)
        var_out = hd_ssdivide(p,hd_ssmult(hd_svdivide(8.d0*!pi,bmag),bmag))
        p = 0
        bmag = 0
        cdf_varget, id, spec_in+'_b_status', b_status, $
          rec_start=rec_start, rec_count=rec_count
        ngd = where(b_status ne 1)
        b_status = 0            ; de-allocate
        if ngd(0) ne -1 then var_out(ngd) = hd_fill
    end
    'patch_status':cdf_varget, id, spec_in+'_patch_status', var_out, $
      rec_start=rec_start, rec_count=rec_count
endcase
if n_elements(var_out) gt 1 then var_out = reform(var_out)
cdf_varget, id, spec_in+'_patch_status', patch_status, $
  rec_start=rec_start, rec_count=rec_count
if n_elements(patch_status) gt 1 then patch_status = reform(patch_status)
cdf_close, id

; Return only those fits that were successful
gdstr = ['(gd)','(gd,*)','(gd,*,*)','(gd,*,*,*)']
gd = where(patch_status eq 0)
if (gd(0) eq -1) then begin
    error_msg = msghead+'No data passed the patch_status=0 filter'
    error_num = 3
    if (not silent) then print, error_msg
    hd_add_footprint, error_msg
    return
endif
sz = size(var_out)
istat = execute('var_out = var_out'+gdstr(sz(0)-1))

if (not silent) then print, error_msg

hd_add_footprint, fp

v_out = var_out

return
end
