function hyd_coord_transform, date, time, vec_in, $
                              ttype=ttype_in, $
                              tmode=tmode_in, $
                              istat=istat_out, $
                              rotmatricies=rm_out, $
                              scvelocities=sc_out
;+
; NAME: hyd_coord_transform.pro
;
; PURPOSE: Perform a coordinate transformation on a 
;          vector (or series of vectors) from:
;                     Polar fixed payload to GSE
;                     GSE to Polar fixed payload
;                     Polar fixed payload to GSM
;                     GSM to Polar fixed payload
;
; CALLING SEQUENCE EXAMPLE:
;
;    vec_out = hyd_coord_transform(date, time, vec_in, ttype='pay2gse')
;
; INPUTS:
;    date - 'YYYYMMDD'
;    time - The time (scalar or 1-D array) in seconds since midnight
;    vec_in - Array of size 3 (or 3 by n_elements(time) for a
;             non-scalar time) containing the vectors to be
;             transformed
; RESULT: 
;    The transformed vector, having the same dimensions as vec_in.
;	
; KEYWORD PARAMETERS:
;
;    ttype = 'pay2gsm'   Polar fixed payload to GSM (default)
;          = 'pay2gse'   Polar fixed payload to GSE
;          = 'gsm2pay'   GSM to Polar fixed payload
;          = 'gse2pay'   GSE to Polar fixed payload
;
;    tmode = 0  rotation only (default)
;               (e.g. magnetic fields, position vectors)
;          = 1  rotate, then add S/C velocity 
;               (e.g. flow speeds for pay2gse or pay2gsm)
;          = 2  subtract S/C velocity, then rotate 
;               (e.g. flow speeds for gse2pay or gsm2pay)
;
;          NOTE: FOR TMODE=1 AND TMODE=2, VEC_IN MUST BE IN KM/S!!!
;
;    istat=istat_out       Output status of the rotation (1 for success)
;
;    rotmatricies=rm_out   Rotation matricies used
;
;    scvelocities=sc_out   S/C velocities used (if tmode=1 or tmode=2)
;
; RESTRICTIONS: If setting tmode=1 or tmode=2, vec_in must be in km/s.
;
; MODIFICATION HISTORY:
;    Written 6/2000 from hydra_payload_to_gse.pro
;    Pamela A. Puhl-Quinn, ppq@space-theory.physics.uiowa.edu
;
; NOTES:  Please note that the algorithm used for both the GSE and GSM
; rotations are exactly the same.
; The crucial difference is the spin vector used, either GSE or GSM.
; The fact that these two rotations use the same algorithm was 
; thoroughly researched and tested.  PPQ
;
;-

if n_elements(istat_out) ne 0 then dum = temporary(istat_out)
if n_elements(rm_out) ne 0 then dum = temporary(rm_out)
if n_elements(sc_out) ne 0 then dum = temporary(sc_out)
if n_elements(tmode_in) ne 0 then tmode = tmode_in else tmode = 0
if n_elements(ttype_in) ne 0 then ttype = ttype_in else ttype = 'pay2gsm'

case ttype of
    'pay2gsm':begin
        reverse=0
        pfix='GSM'
    end
    'pay2gse':begin
        reverse=0
        pfix='GSE'
    end
    'gse2pay':begin
        reverse=1
        pfix='GSE'
    end
    'gsm2pay':begin
        reverse=1
        pfix='GSM'
    end
    else:begin
        message, 'Transformation type not defined', /cont
        istat_out=0
        return, ''
    end
endcase

myname = '% HYD_COORD_TRANSFORM: '

; Get the ephemeris cdfid
f= hydra_findfile( date, /ddcal, /silent)
if (f NE '') then begin
    cdfid = cdf_open(f)
endif else BEGIN
    message, 'DDCAL file not found', /cont
    istat_out = 0
    return, ''
ENDELSE

; First get the gse/gsm spin-vector info
; NOTE: Don't jump off a bridge because 'GSM_EPHEM_TIME_AT' is used
;       below even if pfix='GSE'...This attitude time is valid for 
;       both GSM and GSE...For some reason it is labelled with a 'GSM'
;       in the ddcal file.
att_res = 600.                  ; Attitude variables resolution in seconds
cdf_control, cdfid, var='GSM_EPHEM_TIME_AT', get_var_info=r ; Same for GSE!
cdf_varget, cdfid, 'GSM_EPHEM_TIME_AT', att_time_all, rec_start=0, $
  rec_count=r.maxrec+1
dum = min(abs(att_time_all-time(0)),i1)
dum = min(abs(att_time_all-time(n_elements(time)-1)),i2)
rec_start = i1
rec_count = i2-i1+1
cdf_varget, cdfid, 'GSM_EPHEM_TIME_AT', att_time, rec_start=rec_start, $
  rec_count=rec_count
cdf_varget, cdfid, pfix+'_R_ASCENSION_AT', rascen, rec_start=rec_start, $
  rec_count=rec_count
cdf_varget, cdfid, pfix+'_DECLINATION_AT', declin, rec_start=rec_start, $
  rec_count=rec_count

if (tmode eq 1 or tmode eq 2) then begin
    orb_res = 60.               ; Orbit variables resolution in seconds
    cdf_control, cdfid, var='ORBIT_TIME_OR', get_var_info=r
    cdf_varget, cdfid, 'ORBIT_TIME_OR', orb_time_all, rec_start=0, $
      rec_count=r.maxrec+1
    dum = min(abs(orb_time_all-time(0)),i1)
    dum = min(abs(orb_time_all-time(n_elements(time)-1)),i2)
    rec_start = i1
    rec_count = i2-i1+1
    cdf_varget, cdfid, 'ORBIT_TIME_OR', orb_time, rec_start=rec_start, $
      rec_count=rec_count
    cdf_varget, cdfid, pfix+'_VEL_OR', sc_vel, rec_start=rec_start, $
      rec_count=rec_count       ; sc_vel is in km/s
    sc_vel = reform(sc_vel)
endif

cdf_close, cdfid

; Create the matrices needed on this crude time-scale (minimize the
; number of trig functions)

rmat = make_array(3,3,n_elements(att_time),value=-1.e+31,/double)
rmat_stat = make_array(n_elements(att_time),value=0,/long)

FOR itime=0L,n_elements(att_time)-1 DO BEGIN 
    
    RAC = double(rascen(itime))
    DEC = double(declin(itime))
;
    EPX = COS(DEC) * COS(RAC)
    EPY = COS(DEC) * SIN(RAC)
    EPZ = SIN(DEC)
;
;-- COMPUTE SINE VALUES AND COSINE VALUES OF ANGLES A AND D
;
    COSA = SQRT(EPY*EPY + EPZ*EPZ)
    SINA = EPX
    COSD = EPZ/COSA
    SIND = EPY/COSA
;
;-- CONSTRUCT THE TRANSFORMATION MATRIX
;
    rmat(0,0,itime) = COSA      ; At this point rmat is payload-to-pfix
    rmat(0,1,itime) = 0.
    rmat(0,2,itime) = SINA
    rmat(1,0,itime) = -SINA * SIND
    rmat(1,1,itime) = COSD
    rmat(1,2,itime) = COSA * SIND
    rmat(2,0,itime) = -SINA * COSD
    rmat(2,1,itime) = -SIND
    rmat(2,2,itime) =  COSA * COSD

    rmat_stat(itime) = 1
    
    if (reverse) then begin
        dum = invert(rmat(*,*,itime),invertstat,/double)
        if (invertstat eq 0) then begin
            rmat(*,*,itime) = dum
        endif else begin
            rmat(*,*,itime) = -1.e+31
            rmat_stat(itime) = 0
        endelse
    endif
    
ENDFOR

; Now loop over the time series of vectors to be transformed.

vec_out = make_array(size=size(vec_in), value=-1.e+31)
rm_out = make_array(3,3,n_elements(time), /double, value=-1.e+31)
if (tmode eq 1 or tmode eq 2) then sc_out = $
  make_array(3,n_elements(time), /double, value=-1.e+31)

for n=0L, n_elements(time)-1 do BEGIN
;   Which rotation matrix should I use?
    near = min(abs(att_time - time(n)),jj)
    if (near gt att_res/2.) then message, 'Rotation matrix not found'
    
    if rmat_stat(jj) then begin
;       Vector to be coord_transformd
        v1 = vec_in(0:2,n)
        if (tmode eq 1 or tmode eq 2) then begin
;       Which S/C velocity should I use?
            near = min(abs(orb_time - time(n)),kk)
            if (near gt orb_res/2.) then message, 'S/C velocity not found'
            sc_out(*,n) = sc_vel(*,kk)
;       Subtract off sc_vel first before rotation if desired
            if (tmode eq 2) then v1 = v1 - sc_vel(0:2,kk)
        endif
;       Perform the rotation
        vec_out(0,n) = v1(0)*rmat(0,0,jj)+$
          v1(1)*rmat(0,1,jj)+v1(2)*rmat(0,2,jj)
        vec_out(1,n) = v1(0)*rmat(1,0,jj)+$
          v1(1)*rmat(1,1,jj)+v1(2)*rmat(1,2,jj)
        vec_out(2,n) = v1(0)*rmat(2,0,jj)+$
          v1(1)*rmat(2,1,jj)+v1(2)*rmat(2,2,jj)
        rm_out(*,*,n) = rmat(*,*,jj)        
;       Add the sc_vel after rotation if desired
        if (tmode eq 1) then vec_out(0:2,n) = vec_out(0:2,n) + sc_vel(0:2,kk)
    endif
    
ENDFOR

istat_out = 1
return, vec_out

END




