PRO hydra_payload_to_gse, date, time, vec_pay, vec_gse, istat, rate=rate, $
                          rotmatricies=rm_out
;+
; NAME: hydra_payload_to_gse.pro
;
; PURPOSE: Rotate a vector (or series of vectors) from fixed payload
; coordinates to GSE coordinates
;
; CALLING SEQUENCE: 
;    hydra_payload_to_gse, date, time, vec_pay, vec_gse, istat, rate=rate
;
; INPUTS:
;    date - 'YYYYMMDD'
;    time - The time (scalar or 1-D array) in seconds since midnight
;    vec_pay - Array of size 3 (or 3 by n_elements(time) for a
;              non-scalar time) containing the payload vectors to be rotated
;	
; KEYWORD PARAMETERS:
;    /rate - If set, assumes vec_pay is a velocity and will correct
;            the rotation by the space craft velocity.  vec_pay MUST
;            BE IN KM/S IN ORDER FOR THIS TO WORK CORRECTLY!
;
; OUTPUTS:
;    vec_gse - Array of size 4 (or 4 by n_elements(time) for a
;              non-scalar time) containing the gse vectors, the 4th
;              component being the magnitude.
;    istat - Status of rotation, 1 for success, 0 for failure
;
; COMMON BLOCKS: get_error
;
; RESTRICTIONS: If using the /rate keyword, vec_pay must be in km/s.
;
; MODIFICATION HISTORY:
;    Written 1997, Pamela A. Puhl-Quinn,
;    ppq@space-theory.physics.uiowa.edu
;
; NOTES:  1) Adapted from the routine ICSS_PAYLOAD_TO_GSE.F
;         2) Will NOT handle day boundaries (need to rewrite...)
;-

   COMMON get_error, get_err_no, get_err_msg

   istat =  1

   vec_gse = fltarr(4,n_elements(time))
   rm_out = dblarr(3,3,n_elements(time))

   myname = 'hydra_payload_to_gse: '

   print, myname+'Rotating to GSE...'

; Get the ephemeris cdfid
   f= hydra_findfile( date, /ddcal )
   if (f NE '') then begin
      print, myname+'Opening '+f+'...'
      cdfid = cdf_open(f)
   endif else BEGIN
      print, myname+'Findfile error...'
      istat = 0
      return                    ;hydra_findfile sets the error message
   ENDELSE

; First get the gse spin-vector info
   att_res = 600.               ; Attitude variables resolution in seconds
   orb_res = 60.                ; Orbit variables resolution in seconds

   rec_start = 0
   rec_count_att = 144
   rec_count_orb = 1440

   cdf_varget, cdfid, 'GSM_EPHEM_TIME_AT', att_time, rec_start=rec_start, $
     rec_count=rec_count_att    ; THIS TIME IS SAME FOR GSE!!!
   cdf_varget, cdfid, 'GSE_R_ASCENSION_AT', gse_ra, rec_start=rec_start, $
    rec_count=rec_count_att
   cdf_varget, cdfid, 'GSE_DECLINATION_AT', gse_dec, rec_start=rec_start, $
    rec_count=rec_count_att

   cdf_varget, cdfid, 'ORBIT_TIME_OR', orb_time, rec_start=rec_start, $
    rec_count=rec_count_orb
   cdf_varget, cdfid, 'GSE_VEL_OR', gse_vel, rec_start=rec_start, $
    rec_count=rec_count_orb     ;gse_vel is in km/s

   cdf_close,cdfid

   att_time = reform(att_time)
   gse_ra = reform(gse_ra)
   gse_dec = reform(gse_dec)

; Create the matrices needed on this crude time-scale (minimize the
; number of trig functions)

   pay2gse = dblarr(3,3,n_elements(att_time))

   FOR itime=0,n_elements(att_time)-1 DO BEGIN 

      RAC = double(gse_ra(itime))
      DEC = double(gse_dec(itime))
;
      EPX = COS(DEC) * COS(RAC)
      EPY = COS(DEC) * SIN(RAC)
      EPZ = SIN(DEC)
;
;-- COMPUTE SINE VALUES AND COSINE VALUES OF ANGLES A AND D
;
      COSA = SQRT(EPY*EPY + EPZ*EPZ)
      SINA = EPX
      COSD = EPZ/COSA
      SIND = EPY/COSA
;
;-- CONSTRUCT THE TRANSFORMATION MATRIX
;
      pay2gse(0,0,itime) = COSA
      pay2gse(0,1,itime) = 0.
      pay2gse(0,2,itime) = SINA
      pay2gse(1,0,itime) = -SINA * SIND
      pay2gse(1,1,itime) = COSD
      pay2gse(1,2,itime) = COSA * SIND
      pay2gse(2,0,itime) = -SINA * COSD
      pay2gse(2,1,itime) = -SIND
      pay2gse(2,2,itime) =  COSA * COSD

   ENDFOR


; Now loop over blocks
   for n=0L, n_elements(time)-1 do BEGIN
      
; Which transformation matrix should I use??
      
      near = where(att_time GE time(n)-att_res/2.)
      IF (near(0) NE -1) THEN BEGIN
         jj = near(0)
      ENDIF ELSE BEGIN
         jj = rec_count_att - 1
      ENDELSE
      
      vec_gse(0,n) = vec_pay(0,n)*pay2gse(0,0,jj)+vec_pay(1,n)*pay2gse(0,1,jj)+vec_pay(2,n)*pay2gse(0,2,jj)
      vec_gse(1,n) = vec_pay(0,n)*pay2gse(1,0,jj)+vec_pay(1,n)*pay2gse(1,1,jj)+vec_pay(2,n)*pay2gse(1,2,jj)
      vec_gse(2,n) = vec_pay(0,n)*pay2gse(2,0,jj)+vec_pay(1,n)*pay2gse(2,1,jj)+vec_pay(2,n)*pay2gse(2,2,jj)
      vec_gse(3,n) = sqrt(vec_gse(0,n)^2+vec_gse(1,n)^2+vec_gse(2,n)^2)

      rm_out(*,*,n) = pay2gse(*,*,jj)

      IF keyword_set(rate) THEN BEGIN      ;Correct for the S/C velocity
; Which spacecraft velocity should I use??
         near = where(orb_time GE time(n)-orb_res/2.)
         IF (near(0) NE -1) THEN BEGIN
            jj = near(0)
         ENDIF ELSE BEGIN
            jj = rec_count_orb - 1
         ENDELSE

         FOR icomp= 0,2 DO BEGIN
            vec_gse(icomp,n) = vec_gse(icomp,n) + gse_vel(icomp,jj)
         ENDFOR
      ENDIF

      
   ENDFOR

END
   



