function xticks, axis,index,value
   
   day = long(value/86400.) + 1
   hour = long( (value-(day-1)*86400)/3600. )
   min = long ( value - (day-1)*86400 - 3600*hour ) /60.
   sec = (value - (day-1)*86400) mod 60
   
   return, string(day,hour,min,format="(i2.2,'_',i2.2,':',i2.2)")
end

;+
;================================================================
; NAME: cdhf_one.pro
;
; PURPOSE: Look at one day's worth of Goddard key parameters
;
; AUTHOR: Pamela A. Puhl-Quinn, ppq@space-theory.physics.uiowa.edu
;
; HISTORY: 
; 1st version: 5/16/97 Written, Pamela Puhl-Quinn
;   7/24/98, Added 'ver' keyword, usage help, ppq
;
; USAGE:                             
;    cdhf_one, 960529, ver='06',/ps
;
; MANDATORY ARGUMENTS:
;    1st argument:  YYMMDD or YYYYMMDD
;
; OPTIONAL KEYWORDS:
;    nwin=N    - Number of output window
;    ver='NN'  - Version number string of CDHF KP file
;    /print
;    /ps
;    /pcl
;    /gif
;================================================================ 
;-

PRO cdhf_one, date_in, ver=ver, nwin=nwin, print=print, ps=ps, pcl=pcl, gif=gif,data=data
   
   common cdhf_one_blocks, cdate, cver, time, density, averageE, $
    nfit, post_gap, dqf, instr_mode, sc_pos_gsm, max_energy, min_energy, $
    kplib_version, kp_data_version

; Usage help
   IF (n_params() EQ 0) THEN BEGIN
      doc_library, 'cdhf_one'
      return
   ENDIF
   
   IF (keyword_set(ver)) THEN ver = ver ELSE ver = ''
   IF (keyword_set(nwin)) THEN nwin = nwin ELSE nwin = 0

; Convert the date----------------------
   hydra_time_io,date_in,0,fyear,fmonth,fday,fhour,fminute,fsecond
   date =  strmid(date_in,0,8)
   
   if (keyword_set(cdate)) then begin  
      
      if (cdate eq date AND cver EQ ver) then begin
         read=0 
         print, 'data already in buffer'
      endif else begin
         read=1
         print, 'data being read from file'
      endelse
      
   endif else begin
      print, 'data being read from file'
      read=1
   endelse
   
   if (keyword_set(print)) then print=1 else print = 0
   if (keyword_set(ps)) then ps = 1 else ps = 0
   if (keyword_set(pcl)) then pcl = 1 else pcl = 0
   if (keyword_set(gif)) then gif = 1 else gif = 0
   
   IF (print EQ 1 OR ps EQ 1) THEN set_ps,orient='L',/color
   
   if (read eq 1) then begin
      
;=========READ KEY PARAMETERS CDF FILE==========
       if (keyword_set(ver)) then f = hydra_findfile(date,/CDHF,getversion=ver) else f = hydra_findfile(date,/CDHF)
       
       IF f EQ '' THEN BEGIN
           print, 'No CDHF Key Parameter Data for this day'
           return
       ENDIF ELSE BEGIN
           fn_cdf = f
       ENDELSE
       
       print, 'Opening file ',fn_cdf
       cdfid = CDF_OPEN(fn_cdf)
      print, 'cdfid: ',cdfid
      
      CDF_CONTROL, cdfid, var='Epoch',get_var_info = vinfo
;vinfo.maxrec contains the maximum record number in the file
      cdfnrec = vinfo.maxrec+1
      
      IF (cdfnrec EQ 1) THEN BEGIN
         CDF_ATTGET, cdfid, 'VAR_NOTES','POST_GAP',pg_description
         CDF_VARGET, cdfid, 'POST_GAP', post_gap, rec_count=cdfnrec
         message, 'Only one record in CDF', /cont
         message, 'POST GAP FLAG: '+string(post_gap,'(i2)'),/cont
         message, 'POST GAP FLAG DESCRIPTION: '+pg_description
         CDF_CLOSE, cdfid
         return
      ENDIF

      CDF_VARGET, cdfid, 'Time_PB5', time_pb5, rec_count = cdfnrec
      CDF_VARGET, cdfid, 'ELE_MEAN_ENERGY', averageE, rec_count = cdfnrec
      CDF_VARGET, cdfid, 'ELE_DENSITY', density, rec_count = cdfnrec
      CDF_VARGET, cdfid, 'fit_params',fit_params,rec_count=cdfnrec
      
      
      CDF_ATTGET, cdfid, 'kplib_version',0,kplib_version
      CDF_ATTGET, cdfid, 'kp_data_version',0,kp_data_version
      CDF_ATTGET, cdfid, 'whos_sc_pot',0,whos_sc_pot
      CDF_ATTGET, cdfid, 'scan_interval',0,scan_interval
      
      CDF_VARGET, cdfid, 'POST_GAP', post_gap, rec_count=cdfnrec
      CDF_VARGET, cdfid, 'DQF', dqf, rec_count=cdfnrec
      CDF_VARGET, cdfid, 'INSTR_MODE', instr_mode, rec_count=cdfnrec
      CDF_VARGET, cdfid, 'SC_pos_gsm', sc_pos_gsm, rec_count=cdfnrec
      CDF_VARGET, cdfid, 'MAX_ENERGY', max_energy, rec_count=cdfnrec
      CDF_VARGET, cdfid, 'MIN_ENERGY', min_energy, rec_count=cdfnrec
      
      CDF_CLOSE, cdfid
      
; The time is measured in seconds from midnight
      
      time = time_pb5(2,*)/1000.
      nfit =  fit_params(0,*)
      
      cdate =  date
      cver = ver
   endif                        ; General data read
;---------------------------------------------------------------   
; Create an 11-element color table
; color = 0 is black
; color = 1 is white
; color = 2 is grey
; color = 3 is green
; color = 4 is dark blue
; color = 5 is red
; color = 6 is purple
; color = 7 is orange
; color = 8 is rust
; color = 9 is hunter green
; color = 10 is hot pink
   
   red=  [0, 255, 153, 0,   0,   255, 88,  255, 131, 0,   243]
   green=[0, 255, 153, 255, 0,   0,   0,   178, 20 , 147, 0  ]
   blue= [0, 255, 153, 0,   255, 0,   132, 0,   46 , 37,  191]
   tvlct,red,green,blue
   
   !p.background=1
   !p.color=0
   !p.multi =  [0,1,7]
   
   xmin =  0
   xmax =  24
   IF (keyword_set(print) OR keyword_set(ps) OR keyword_set(pcl)) THEN $
    charsize = .5 ELSE charsize = 1.5
   cdhf = {psym:1,symsize:.5,xrange:[xmin,xmax],charsize:charsize}
   two = {psym:1,symsize:.5}
   
   IF (print NE 1 AND ps NE 1) THEN window,nwin,xsize=850,ysize=800
   good1 =  where(nfit EQ 1)
   good2 =  where(nfit EQ 2)
   good3 =  where(nfit EQ 3)
   
   plot, time/3600.,density,/ylog,color=0,_extra=cdhf,ytitle='density',/nodata,title=date
   oplot, time/3600.,density,_extra=two,color=0
   yy = fltarr(n_elements(density)) + 10.
   oplot, time/3600.,yy,line=3
   
   plot, time/3600.,averageE,/ylog,color=0,_extra=cdhf,ytitle='kT',/nodata
   oplot, time/3600.,averageE,_extra=two,color=0
   
   plot, time/3600.,post_gap,color=0,_extra=cdhf,yrange=[-1,10],ytitle='post_gap'
   plot, time/3600.,dqf,color=0,_extra=cdhf,yrange=[-2,2],ytitle='dqf'
   
   plot, time/3600.,instr_mode(0,*),color=0,_extra=cdhf,yrange=[1,100],/ylog,ytitle='TM and BLK Mode'
   oplot,time/3600.,instr_mode(1,*),color=0,_extra=two
   
   radius =  sqrt(sc_pos_gsm(0,*)^2+sc_pos_gsm(1,*)^2+sc_pos_gsm(2,*)^2)
   plot, time/3600.,radius,color=0,_extra=cdhf,/ylog,ytitle='radius'
   
   plot, time/3600., max_energy,color=0,_extra=cdhf,yrange=[1,20000],/ylog,ytitle='Max and Min Energies',xtitle='UT [hr]'
   oplot, time/3600.,min_energy,color=0,_extra=two
   
   slab = '  kplib: '+string(kplib_version,'(f5.2)')+' ,kp_data_version: '+string(kp_data_version,'(f5.2)')
   
   label_page,side_label= slab
   
   
;===================================================================
   
   if (gif eq 1) then begin
      write_gif,'idl.gif',tvrd(),red,green,blue
   ENDIF

   IF keyword_set(data) THEN BEGIN
      nrec = n_elements(time)
      data = {time:fltarr(nrec), $
              density:fltarr(nrec), $
              averageE:fltarr(nrec), $
              nfit:intarr(nrec), $
              post_gap:intarr(nrec), $
              dqf:intarr(nrec), $
              instr_mode:intarr(2,nrec), $
              sc_pos_gsm:fltarr(3,nrec), $
              max_energy:fltarr(nrec), $
              min_energy:fltarr(nrec)}
      data.time = time
      data.density = density
      data.averageE = averageE
      data.nfit = nfit
      data.post_gap = post_gap
      data.dqf = dqf
      data.instr_mode = instr_mode
      data.sc_pos_gsm = sc_pos_gsm
      data.max_energy = max_energy
      data.min_energy = min_energy
   ENDIF

   
   if (print EQ 1) then begin
      end_of_prog, /print
   endif else if (ps eq 1) then begin
      end_of_prog
   endif else if (pcl eq 1) then begin
      device,/close
      spawn, 'lp -d cis_color idl.pcl'
      print, 'spawning lp -d cis_color idl.pcl'
      set_plot,'x'
   endif
   
   
END

