function xticks, axis,index,value

day = long(value/86400.) + 1
hour = long( (value-(day-1)*86400)/3600. )
min = long ( value - (day-1)*86400 - 3600*hour ) /60.
sec = (value - (day-1)*86400) mod 60

return, string(day,hour,min,format="(i2.2,'_',i2.2,':',i2.2)")
end
;+
;================================================================
; NAME: cdhf_stack.pro
;
; PURPOSE: Look at one month's worth of Goddard key parameters,
; 'stacked' according to the 17.5 hour period of POLAR's orbit.
;
; AUTHOR: Pamela A. Puhl-Quinn, ppq@space-theory.physics.uiowa.edu
;
; HISTORY: 
; 1st version: 2/14/96 Written, Pamela Puhl-Quinn
;   7/24/98, Usage help, ppq
;
; USAGE:                             
;    cdhf_stack, '199605', /ps
;
; MANDATORY ARGUMENTS:
;    1st argument:  'YYYYMM'
;
; OPTIONAL KEYWORDS:
;    nwin=N    - Number of output window
;    /print
;    /ps
;    /pcl
;    /gif
;================================================================ 
;-
PRO cdhf_stack, date, print=print, ps=ps, pcl=pcl, gif=gif

common cdhf_stack_blocks, cdate, time, density, averageE, icounter, $
  trange_gap, cal_version, day_of_month

; Usage help
IF (n_params() EQ 0) THEN BEGIN
    doc_library, 'cdhf_stack'
    return
ENDIF

if (keyword_set(cdate)) then begin  
    if (cdate eq date) then begin
        read=0 
        print, 'data already in buffer'
    endif else begin
        read=1
        print, 'data being read from file'
    endelse
endif else begin
    print, 'data being read from file'
    read=1
endelse

if (keyword_set(print)) then print=1 else print = 0
if (keyword_set(ps)) then ps = 1 else ps = 0
if (keyword_set(pcl)) then pcl = 1 else pcl = 0
if (keyword_set(gif)) then gif = 1 else gif = 0

year = long(strmid(date,0,4))
month = long(strmid(date,4,2))

julday0 = julday(1,1,year,0,0,0)
julday1 = julday(month,1,year,0,0,0)
if (month ne 12) then julday2 = julday(month+1,1,year,0,0,0) else $
  julday2 = julday(1,1,year+1,0,0,0)
doy1 = long(julday1 - julday0 + 1) ; Day-of-year of the 1st of this month
ndays = long(julday2 - julday1) ; Days in this month
juldays = findgen(ndays) + julday1
caldat, juldays, calmo, calda, calyr, calhr, calmi, calse
caldays = $                     ; List-of-days for this month (strings)
  string(calyr,'(i4.4)')+string(calmo,'(i2.2)')+string(calda,'(i2.2)')

if (read eq 1) then BEGIN
    
;=========READ KEY PARAMETERS CDF FILE==========
    icount_max=100000
    time = fltarr(icount_max)
    density = fltarr(icount_max)
    averageE = fltarr(icount_max)
    trange_gap = fltarr(2,icount_max)
    day_of_month = intarr(icount_max)
    cal_version = fltarr(31) - 99
    
    icounter = long(-1)
    igap = long(-1)
    
    for icday = 0, n_elements(caldays)-1 do begin
        
        f = hydra_findfile(caldays(icday),/cdhf,/silent)
        if (f eq '') then begin
            message, 'No CDHF file found for '+caldays(icday), /cont
            goto, no_data
        endif
        
        message, 'Opening file '+f, /cont
        cdfid = CDF_OPEN(f)
        
        iday = long(strmid(caldays(icday),6,2))
        
        CDF_CONTROL, cdfid, var='Epoch',get_var_info = vinfo
; vinfo.maxrec contains the maximum record number in the CDF file
        cdfnrec = vinfo.maxrec+1
        
        if (cdfnrec le 1) then begin
            message, 'No data in '+f, /cont
            CDF_CLOSE, cdfid
            goto, no_data
        endif
        
        time_pb5 = lonarr(3,cdfnrec)
        averageE_ele = fltarr(cdfnrec)
        density_ele = fltarr(cdfnrec) 
        CDF_VARGET, cdfid, 'Time_PB5', time_pb5, rec_count = cdfnrec
        CDF_VARGET, cdfid, 'ELE_MEAN_ENERGY', averageE_ele, rec_count = cdfnrec
        CDF_VARGET, cdfid, 'ELE_DENSITY', density_ele, rec_count = cdfnrec
        CDF_VARGET, cdfid, 'POST_GAP', post_gap, rec_count=cdfnrec
        CDF_VARGET, cdfid, 'FP_CALIBRATION_VERSION', calibration_version, rec_count=1
        cal_version(iday-1) = calibration_version
        
;Diagnostic reads (not required for plot)
;log_file = string('                                  ')
;epoch = dblarr(cdfnrec)
;post_gap = lonarr(cdfnrec)
;dqf = lonarr(cdfnrec)
;instr_mode = lonarr(2,cdfnrec)
;sc_pos_gsm = fltarr(3,cdfnrec)
;max_energy = fltarr(cdfnrec)
;min_energy = fltarr(cdfnrec)
;CDF_ATTGET, cdfid, 'log_filename',0,log_file
;CDF_ATTGET, cdfid, 'quality_url',0,log_file
;CDF_ATTGET, cdfid, 'CATDESC','POST_GAP',log_file
;CDF_VARGET, cdfid, 'Epoch', epoch, rec_count=cdfnrec
;CDF_VARGET, cdfid, 'POST_GAP', post_gap, rec_count=cdfnrec
;CDF_VARGET, cdfid, 'DQF', dqf, rec_count=cdfnrec
;CDF_VARGET, cdfid, 'INSTR_MODE', instr_mode, rec_count=cdfnrec
;CDF_VARGET, cdfid, 'SC_pos_gsm', sc_pos_gsm, rec_count=cdfnrec
;CDF_VARGET, cdfid, 'MAX_ENERGY', max_energy, rec_count=cdfnrec
;CDF_VARGET, cdfid, 'MIN_ENERGY', min_energy, rec_count=cdfnrec
        
        CDF_CLOSE, cdfid
        
                                ; The time is measured in seconds from doy1:00:00:00
        for i=0,cdfnrec-1 do begin
            icounter=icounter+1
            time(icounter) = (time_pb5(1,i)-doy1)*24*3600. +  $
              time_pb5(2,i)/1000.
            day_of_month(icounter) = long(time(icounter)/86400.) + 1
            density(icounter) = density_ele(i)
            averageE(icounter) = averageE_ele(i)
            IF (post_gap(i) EQ 4) THEN BEGIN
                igap = igap+1
                trange_gap(0,igap) = time(icounter-1)
                trange_gap(1,igap) = time(icounter)
            ENDIF
            
        endfor
        
        no_data:
        
    endfor
    
    if (icounter eq -1) then begin
        message, 'No CDHF KP data found for this month', /cont
        return
    endif

    time = time(0:icounter)
    density = density(0:icounter)
    averageE = averageE(0:icounter)
    day_of_month = day_of_month(0:icounter)
    trange_gap = trange_gap(0:1,0:igap)
    
;----------------------
    cdate = date
    
endif                           ; General data read

;  Find orbit_frac based on the start_time and period
;start_time = float(hour_start*3600.)   ; hour_start no longer an input
start_time = time(0)
period = float(17.5*3600.)
orbit_frac = fltarr(icounter+1)
for jj = 0L, icounter do begin
    orbit_frac(jj) = time(jj)/period + $
      (1.-(start_time+period)/period)
endfor
orbit_frac = orbit_frac(0:icounter)

;======================PLOTTING STATEMENTS=========================
if (print eq 1) then set_ps,orient='L'

IF (print EQ 0) THEN BEGIN
    window,0,xsize=900,ysize=800
    pltsize = 2.0
ENDIF ELSE BEGIN
    pltsize =  .75
ENDELSE

loadct, 39
!p.background = 99
!p.color = 0
!p.font = -1
!p.noclip = 0
font = '!5'
!p.multi = [0,2,4,0,0]

for i = 0, long(max(orbit_frac)) do begin
    
    orbit = where (orbit_frac ge i and orbit_frac le i+1)
    xmin = i*period + start_time
    xmax = (i+1)*period + start_time
    
    if (orbit(0) eq -1) then begin
        plot,	time,density,/nodata,$
          /ylog,xrange=[xmin,xmax],xstyle=1,$
          ytitle = 'density',min_value = .0001,$ 
          xtitle = 'UT',xtickformat = 'xticks',charsize=pltsize, $
          title = date+' CDHF Key Parameters'
        xyouts,.5*(xmin+xmax),3,'KP gap here'
                                ;psym=2, symsize=.5
        oplot, [xmin,xmax],[10,10],line=3
        
        plot,	time,averageE,/nodata,$
          /ylog,xrange=[xmin,xmax],xstyle=1,$
          ytitle = 'averageE',ystyle=1,min_value = .0001,$
          xtitle='UT',xtickformat = 'xticks',charsize=pltsize
        xyouts,.5*(xmin+xmax),50,'KP gap here'
                                ;psym=2, symsize=.5
    endif else begin
        plot,	time(orbit),density(orbit),$
          /ylog,xrange=[xmin,xmax],xstyle=1,$
          ytitle = 'density',min_value = .0001,$ 
          xtitle = 'UT',xtickformat = 'xticks',charsize=pltsize, $
          title = date+' CDHF Key Parameters'
                                ;psym=2, symsize=.5
        
        inside = where(trange_gap(0,*) GE xmin AND trange_gap(1,*) LE xmax)
        IF (inside(0) NE -1) THEN BEGIN 
            FOR iii=0,n_elements(inside)-1 DO BEGIN
                ii = inside(iii)
                polyfill, [trange_gap(0,ii),trange_gap(0,ii),trange_gap(1,ii),trange_gap(1,ii)],10^([!y.crange(0),!y.crange(1),!y.crange(1),!y.crange(0)]),color=80
            ENDFOR 
        ENDIF
        
        oplot, time(orbit), density(orbit), line=0
        oplot, [xmin,xmax],[10,10],line=3
        
        
; xyouts the cal versions
        iday_min = min(day_of_month(orbit))
        iday_max = max(day_of_month(orbit))
        idays = indgen(iday_max-iday_min+1) + iday_min
        FOR jjj=0,n_elements(idays)-1 DO BEGIN
            ncs = norm_axis([.1,.9-.05*jjj])
            ncs(1) = 10^(ncs(1))
            xyouts, ncs(0),ncs(1),'DAY: '+string(idays(jjj),'(i2.2)')+' CAL: '+string(cal_version(idays(jjj)-1),'(f4.2)')
        ENDFOR
        
        
        plot,	time(orbit),averageE(orbit),$
          /ylog,xrange=[xmin,xmax],xstyle=1,$
          ytitle = 'averageE',ystyle=1,min_value = .0001,$
          xtitle='UT',xtickformat = 'xticks',charsize=pltsize
                                ;psym=2, symsize=.5
    endelse
    
    if (!p.multi(0) eq 0 or i eq long(max(orbit_frac))) then time_stamp
    
    if (!p.multi(0) eq 0 and print eq 0) then begin
        print, 'Press any key to see next page'
        res = get_kbrd(1)
    endif
    
    
    
endfor

;===================================================================


end_flag:

if (gif eq 1) then begin
    write_gif,'idl.gif',tvrd(),red,green,blue
endif

if (print EQ 1) then begin
    end_of_prog, /print
endif else if (ps eq 1) then begin
    end_of_prog
endif else if (pcl eq 1) then begin
    device,/close
    spawn, 'lp -d cis_color idl.pcl'
    print, 'spawning lp -d cis_color idl.pcl'
    set_plot,'x'
endif

END 


