;+
; Hydra/DDEIS 3-D Moments Extraction Routine
; FILE: /home/ppq/idl_tools_public/release/extract_moments.pro
; WRITTEN: Version 1.0, 10/1/97, Pamela A. Puhl-Quinn
; This routine extracts the 3-D moments from CDF files
;
; USAGE:
;   extract_moments,960529,0200,6,moments
;
; MANDATORY ARGUMENTS:
;   INPUT: 1st argument(DATE):        YYMMDD or YYYYMMDD
;   INPUT: 2nd argument(START TIME):  'HH:MM:SS','HH:MM',HHMMSS,HHMM,HH.HHH,H
;   INPUT: 3rd argument(# HOURS):     # of hours you want extracted
;   OUTPUT: 4th argument:             Name of returned data structure
;                                        containing the moments data
;
; OPTIONAL KEYWORDS:
;  /ion                          INPUT  (electrons is the default)
;  indir='cdf_directory_path'    INPUT  Just the full path, not the filename
;  full_file_name='path'+'filename'  INPUT, Full path AND filename
;  extract_code_version = ecv    OUTPUT, vers. # of this extraction rtn
;  moments_data_version = mdv    OUTPUT, vers. # of the moments data extracted
;  /nofilter                     INPUT, No filtering of 'bad' points. 
;                                       Default:  the data filter
;                                       defined in the moments CDF
;                                       file is applied.
;-
;--------------------------------------------------------------------------

PRO extract_moments,date_in,time_in,hours_in,mom,indir=indir,extract_code_version=extract_code_version, moments_data_version=moments_data_version,ion=ion,ele=ele,full_file_name=full_file_name,nofilter=nofilter

   IF (n_params() EQ 0) THEN BEGIN
       doc_library, 'extract_moments'
       return
   ENDIF

; Define the version number for this extraction routine:
   extract_code_version = 1.0

; Convert the date and time string using Jeremy's tool----------------------
   hydra_time_io,date_in,time_in,iyear,imonth,iday,ihour,iminute,isecond
   date_str =  strmid(date_in,0,8)  
   hour = string(ihour,'(i2.2)')
   min = string(iminute,'(i2.2)')
   second = string(isecond,'(i2.2)')
   start_time = ihour*3600. + iminute*60. + isecond
   end_time =  start_time + hours_in*3600.

   IF (keyword_set(full_file_name)) THEN BEGIN
      f = full_file_name
      message, 'You specifically chose the file: '+f, /cont
      moments_data_version = strmid( f, strlen(f)-8, 4 )
      GOTO, skip_findfile
   ENDIF 

   IF (keyword_set(indir)) THEN BEGIN
      
      IF (keyword_set(ion)) THEN begin
         f = hydra_findfile( date_str, /moments_ion, path=indir,$
                             version=version)
         moments_data_version = version
      ENDIF ELSE BEGIN
         f = hydra_findfile( date_str, /moments_all, path=indir,$
                             version=version)
         moments_data_version = version
      ENDELSE

   ENDIF ELSE BEGIN

      IF (keyword_set(ion)) THEN BEGIN 
         f = hydra_findfile( date_str, /moments_ion, version=version)
         moments_data_version = version
      ENDIF ELSE BEGIN
         f = hydra_findfile( date_str, /moments_all, version=version)
         moments_data_version = version
      ENDELSE


   ENDELSE

   skip_findfile:

   if (f NE '') then begin
      message, 'Opening '+f+'...', /cont
      cdf = cdf_open(f)
   endif else begin
      return
   ENDELSE

; Extract the variables

   cdf_control, cdf, variable='time', get_var_info=r
   IF r.maxrec EQ -1 THEN BEGIN
      message, 'No data found in CDF', /cont
      return
   ENDIF

   hyd_cdf_varget, cdf, 'time', time, rec_count=r.maxrec+1
   rec_start= ( where( time+6.9 ge start_time ) )(0)
   if rec_start eq -1 then begin
      message, 'No data found after start_time, resetting ', /cont
      rec_start = 0
   endif 
   rec_count= n_elements( where( time(rec_start:*) lt end_time $
                                 and time(rec_start:*) gt start_time, $
                                 count ))
   
   if ( count ) eq 0 then begin
      message, 'No data found in time interval specified', /cont
      return
   endif      
   
   data_values = { mom_dum, $
                   species:0,$
                   epoch:0.d0,$
                   id_number:0,$
                   time:0.d0, $
                   temp:dblarr(3,3),$
                   version:0.0,$
                   status:long(0),$
                   sc_potential:0.0,$
                   variance_sc_potential:0.0,$
                   sc_potential_status:0,$
                   background:fltarr(12),$
                   background_status:0.0,$
                   background_version:0.0,$
                   box:0,$
                   inter_box_gain:intarr(2),$
                   inter_box_gain_status:0,$
                   relative_gain_status:0,$
                   mnf_quality_dd:0,$
                   mnf_quality_b:0,$
                   hk_status:0,$
                   psi_status:0,$
                   b_status:0,$
                   mfe_b_avg:fltarr(3),$
                   mfe_b_avg_sigma:fltarr(3),$
                   bfield_corr_status:0,$
                   n:0.0,$
                   dn:0.0,$
                   n_black:0.0,$
                   n_patch:0.0,$
                   v:fltarr(3),$
                   dv:fltarr(3),$
                   v_black:fltarr(3),$
                   v_patch:fltarr(3),$
                   pdiag:fltarr(3),$
                   dpdiag:fltarr(3),$
                   pd_black:fltarr(3),$
                   pd_patch:fltarr(3),$
                   poff:fltarr(3),$
                   dpoff:fltarr(3),$
                   po_black:fltarr(3),$
                   po_patch:fltarr(3),$
                   q:fltarr(3),$
                   dq:fltarr(3),$
                   q_black:fltarr(3),$
                   q_patch:fltarr(3),$
                   t:0.0,$
                   b_proxy:fltarr(3),$
                   tparl:0.0,$
                   tperp:0.0,$
                   gyrotropy:0.0,$
                   angle_v_tparl:0.0,$
                   angle_q_tparl:0.0,$
                   angle_tparl_bave:0.0,$
                   patch_chisq:0.0,$
                   b:fltarr(10),$
                   return_current:0.0,$
                   return_sec_current:0.0,$
                   return_current_error:0.0,$
                   return_sec_current_error:0.0,$
                   return_current_difference:0.0,$
                   skew:0.0,$
                   itop:0,$
                   mfe_status:0,$
                   mfe_gain_version:0,$
                   mfe_offset_version:0,$
                   mfe_matrix_version:0,$
                   bias_status:0,$
                   bias_voltage_level_1:0,$
                   bias_voltage_level_2:0,$
                   patch_hem_val:0,$
                   patch_version:0.0,$
                   gse_r_ascension:0.0,$
                   gse_declination:0.0,$
                   date_time:lonarr(2),$
                   favg_imin:fltarr(2),$
                   energy_imin:fltarr(2), $
                   fifth_digit_of_status:0, $
                   rc_patch:0.0 $
                 }   
   
   mom = replicate( data_values, rec_count )
   
   hyd_cdf_varget, cdf, 'species',species, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'epoch',epoch, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'id_number',id_number, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'time',time, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'temp',temp, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'version',version, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'status',status, $
     rec_start=rec_start, rec_count=rec_count
   
   fifth_digit_of_status = long(status/10000.) MOD 10

   hyd_cdf_varget, cdf, 'sc_potential',sc_potential, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'variance_sc_potential',variance_sc_potential, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'sc_potential_status',sc_potential_status, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'background',background, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'background_status' ,background_status, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'background_version',background_version, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'box',box, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'inter_box_gain',inter_box_gain, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'inter_box_gain_status',inter_box_gain_status, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'relative_gain_status',relative_gain_status, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'mnf_quality_dd',mnf_quality_dd, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'mnf_quality_b',mnf_quality_b, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'hk_status',hk_status, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'psi_status',psi_status, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'b_status',b_status, $
    rec_start=rec_start, rec_count=rec_count

   isthere = cdf_varexist(cdf,'mfe_b_avg')
   IF (isthere) THEN BEGIN
      hyd_cdf_varget, cdf, 'mfe_b_avg',mfe_b_avg, $
       rec_start=rec_start, rec_count=rec_count
      mom.mfe_b_avg = mfe_b_avg
   ENDIF 

   isthere = cdf_varexist(cdf,'mfe_b_avg_sigma')
   IF (isthere) THEN BEGIN
      hyd_cdf_varget, cdf, 'mfe_b_avg_sigma',mfe_b_avg_sigma, $
       rec_start=rec_start, rec_count=rec_count
      mom.mfe_b_avg_sigma = mfe_b_avg_sigma
   ENDIF 

   hyd_cdf_varget, cdf, 'bfield_corr_status',bfield_corr_status, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'n',n, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'dn',dn, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'n_black',n_black, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'n_patch',n_patch, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'v',v, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'dv',dv, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'v_black',v_black, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'v_patch',v_patch, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'pdiag',pdiag, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'dpdiag',dpdiag, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'pd_black',pd_black, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'pd_patch',pd_patch, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'poff',poff, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'dpoff',dpoff, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'po_black',po_black, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'po_patch',po_patch, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'q',q, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'dq',dq, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'q_black',q_black, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'q_patch',q_patch, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 't',t, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'b_proxy',b_proxy, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'tparl',tparl, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'tperp',tperp, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'gyrotropy',gyrotropy, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'angle_v_tparl',angle_v_tparl, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'angle_q_tparl',angle_q_tparl, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'angle_tparl_bave',angle_tparl_bave, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'patch_chisq',patch_chisq, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'b',b, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'return_current',return_current, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'return_current_error',return_current_error, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'return_sec_current',return_sec_current, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'return_sec_current_error',return_sec_current_error, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'return_current_difference',return_current_difference, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'skew',skew, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'itop',itop, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'mfe_status',mfe_status, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'mfe_gain_version',mfe_gain_version, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'mfe_offset_version',mfe_offset_version, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'mfe_matrix_version',mfe_matrix_version, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'bias_status',bias_status, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'bias_voltage_level_1',bias_voltage_level_1, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'bias_voltage_level_2',bias_voltage_level_2, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'patch_hem_val',patch_hem_val, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'patch_version',patch_version, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'gse_r_ascension',gse_r_ascension, $
    rec_start=rec_start, rec_count=rec_count
 
  hyd_cdf_varget, cdf, 'gse_declination',gse_declination, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'date_time',date_time, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'favg_imin',favg_imin, $
    rec_start=rec_start, rec_count=rec_count

   hyd_cdf_varget, cdf, 'energy_imin',energy_imin, $
    rec_start=rec_start, rec_count=rec_count

   isthere = cdf_varexist(cdf,'rc_patch')
   IF (isthere) THEN BEGIN
       hyd_cdf_varget, cdf, 'rc_patch',rc_patch, $
         rec_start=rec_start, rec_count=rec_count
       mom.rc_patch = reform(rc_patch)
   ENDIF

;   isthere = cdf_varexist(cdf,'b_hr_avg')
;   IF (isthere) THEN BEGIN
;      hyd_cdf_varget, cdf, 'b_hr_avg',b_hr_avg, $
;       rec_start=rec_start, rec_count=rec_count
;      mom.b_hr_avg =  b_hr_avg
;   ENDIF
   
;   isthere = cdf_varexist(cdf,'b_hr_sigma')
;   IF (isthere) THEN BEGIN
;      hyd_cdf_varget, cdf, 'b_hr_sigma',b_hr_sigma, $
;       rec_start=rec_start, rec_count=rec_count
;      mom.b_hr_sigma = b_hr_sigma
;   ENDIF
   
;   isthere = cdf_varexist(cdf,'b_hr_avg_sigma')
;   IF (isthere) THEN BEGIN 
;      hyd_cdf_varget, cdf, 'b_hr_avg_sigma',b_hr_avg_sigma, $
;       rec_start=rec_start, rec_count=rec_count
;      mom.b_hr_avg_sigma =  reform(b_hr_avg_sigma)
;   ENDIF

   cdf_close,cdf
   
; One-dimensional arrays
   mom.species =  reform(species)
   mom.epoch =  reform(epoch)
   mom.id_number = reform(id_number)
   mom.time = reform(time)
   mom.version = reform(version)
   mom.status = reform(status)
   mom.fifth_digit_of_status = reform(fifth_digit_of_status)
   mom.sc_potential = reform(sc_potential)
   mom.variance_sc_potential = reform(variance_sc_potential)
   mom.sc_potential_status = reform(sc_potential_status)
   mom.background_status = reform(background_status)
   mom.background_version = reform(background_version)
   mom.box = reform(box)
   mom.inter_box_gain_status = reform(inter_box_gain_status)
   mom.relative_gain_status = reform(relative_gain_status)
   mom.mnf_quality_dd = reform(mnf_quality_dd)
   mom.mnf_quality_b = reform(mnf_quality_b)
   mom.hk_status = reform(hk_status)
   mom.psi_status = reform(psi_status)
   mom.b_status = reform(b_status)
   mom.bfield_corr_status = reform(bfield_corr_status)
   mom.n = reform(n)
   mom.dn = reform(dn)
   mom.n_black = reform(n_black)
   mom.n_patch = reform(n_patch)
   mom.t = reform(t)
   mom.tparl = reform(tparl)
   mom.tperp = reform(tperp)
   mom.gyrotropy = reform(gyrotropy)
   mom.angle_v_tparl = reform(angle_v_tparl)
   mom.angle_q_tparl = reform(angle_q_tparl)
   mom.angle_tparl_bave = reform(angle_tparl_bave)
   mom.patch_chisq = reform(patch_chisq)
   mom.return_current = reform(return_current)
   mom.return_current_error = reform(return_current_error)
   mom.return_sec_current = reform(return_sec_current)
   mom.return_sec_current_error = reform(return_sec_current_error)
   mom.return_current_difference = reform(return_current_difference)
   mom.skew = reform(skew)
   mom.itop = reform(itop)
   mom.mfe_status = reform(mfe_status)
   mom.mfe_gain_version = reform(mfe_gain_version)
   mom.mfe_offset_version = reform(mfe_offset_version)
   mom.mfe_matrix_version = reform(mfe_matrix_version)
   mom.bias_status = reform(bias_status)
   mom.bias_voltage_level_1 = reform(bias_voltage_level_1)
   mom.bias_voltage_level_2 = reform(bias_voltage_level_2)
   mom.patch_hem_val = reform(patch_hem_val)
   mom.patch_version = reform(patch_version)
   mom.gse_r_ascension = reform(gse_r_ascension)
   mom.gse_declination = reform(gse_declination)

; Arrays of more than one dimension
   mom.favg_imin =  favg_imin
   mom.energy_imin =  energy_imin
   mom.temp =  temp
   mom.background = background
   mom.inter_box_gain = inter_box_gain
   mom.v = v
   mom.dv = dv
   mom.v_black = v_black
   mom.v_patch = v_patch
   mom.pdiag = pdiag
   mom.dpdiag = dpdiag
   mom.pd_black = pd_black
   mom.pd_patch = pd_patch
   mom.poff = poff
   mom.dpoff = dpoff
   mom.po_black = po_black
   mom.po_patch = po_patch
   mom.q = q
   mom.dq = dq
   mom.q_black = q_black
   mom.q_patch = q_patch
   mom.b_proxy = b_proxy
   mom.b = b
   mom.date_time = date_time

; Filter out the bad moments points!!!

   if (not keyword_set(nofilter)) then begin 
       message, 'Filtering out the bad moments points',/cont
       patch_flag = (mom.status/10000) MOD 10 ; indicates the fifth digit of mom.status
       gd = where(mom.status GT 0 AND patch_flag NE 3)
                                ; if the fifth digit is equal to 3, it
                                ;means no patch, discards this block
       IF (gd(0) NE -1) THEN begin
           mom = mom(gd)
           message, 'Number of points filtered out: '+string(n_elements(gd),'(i5)'),/cont
       endif else begin
           message, 'Number of points filtered out: 0'
       endelse
   endif else begin
       message, 'You have chosen not to filter ANY bad points...', /cont
   endelse



END

