; Hydra/DDEIS 3-D Moments Extraction Routine
; FILE: /home/ppq/idl_tools_public/release/extract_rc_moments.pro
; WRITTEN: 10/10/97, Pamela A. Puhl-Quinn
; This routine extracts the 3-D moments from the rcmom CDF files
;
; USAGE:
;   extract_rc_moments,960529,0200,6,moments
;
; MANDATORY ARGUMENTS:
;	1st argument(DATE):        YYMMDD or YYYYMMDD
;	2nd argument(START TIME):  'HH:MM:SS','HH:MM',HHMMSS,HHMM,HH.HHH,H
;	3rd argument(# HOURS):     # of hours you want extracted
;       4th argument:              Name of returned data structure
;
; OPTIONAL KEYWORDS:
;  /ion    (default is electrons)
;  indir='cdf_directory_path'
;---------------------------------------------------------------------------

PRO extract_rc_moments,date_in,time_in,hours_in,mom,indir=indir,ion=ion

   IF (n_params() EQ 0) THEN BEGIN
      print, ' '
      print, '---------------------------------------------
      print, 'USAGE:'
      print, 'extract_rc_moments,960529,0200,6,moments'
      print, ' '
      print, 'MANDATORY ARGUMENTS:'
      print, '1st argument(DATE):        YYMMDD or YYYYMMDD'
      print, '2nd argument(START TIME):  "HH:MM:SS","HH:MM",HHMMSS,HHMM,HH.HHH,H'
      print, '3rd argument(# HOURS):     # of hours you want extracted'
      print, '4th argument:              Name of returned data structure'
      print, ' '
      print, 'OPTIONAL KEYWORDS:'
      print, '/ion (default is electrons)'
      print, 'indir="cdf_directory_path"'
      print, '---------------------------------------------
      print, ' '
      return
   ENDIF

; Convert the date and time string using Jeremy's tool----------------------
   hydra_time_io,date_in,time_in,iyear,imonth,iday,ihour,iminute,isecond
   date_str =  strmid(date_in,0,8)  
   hour = string(ihour,'(i2.2)')
   min = string(iminute,'(i2.2)')
   second = string(isecond,'(i2.2)')
   start_time = ihour*3600. + iminute*60. + isecond
   end_time =  start_time + hours_in*3600.

   myname= 'extract_rc_moments.pro'

   IF (keyword_set(indir)) THEN BEGIN
      IF (keyword_set(ion)) THEN BEGIN
         f = hydra_findfile( date_str, /moments_rc_ion, path=indir)
      ENDIF ELSE BEGIN         
         f = hydra_findfile( date_str, /moments_rc_ele, path=indir)
      ENDELSE


   ENDIF ELSE BEGIN
      IF (keyword_set(ion)) THEN BEGIN
         f = hydra_findfile( date_str, /moments_rc_ion)
      ENDIF ELSE BEGIN
         f = hydra_findfile( date_str, /moments_rc_ele)
      ENDELSE

   ENDELSE

   if (f NE '') then begin
      print, myname+': opening '+f+'...'
      cdf = cdf_open(f)
      n= strlen(f)
      data_version= float( strmid( f, n-8, 4 ))
   endif else begin
      return
   ENDELSE

; Extract the variables

   cdf_control, cdf, variable='time', get_var_info=r
   IF r.maxrec EQ -1 THEN BEGIN
      print, myname+': No data found in CDF'
      stop
   ENDIF

   cdf_varget, cdf, 'time', time, rec_count=r.maxrec+1
   rec_start= ( where( time ge start_time ) )(0)
   if rec_start eq -1 then begin
      print, myname+': No data found after start_time, resetting '
      rec_start = 0
   endif
   rec_count= n_elements( where( time(rec_start:*) lt end_time $
                                 and time(rec_start:*) gt start_time, $
                                 count ))
   
   if ( count ) eq 0 then begin
      print, myname+': No data found in time interval specified'
      return
   endif      
   
   data_values = { momrc_dum, $
                   time:0.d0, $
                   radius:0.0, $
                   lshell:0.0, $
                   sc_potential:0.0, $
                   background:0.0, $
                   return_current_sec_residual:fltarr(12), $
                   return_current_residual:fltarr(12), $   
                   return_current:0.0, $
                   rc_patch:0.0, $
                   rc_black:0.0, $
                   return_current_sec:0.0, $
                   energy_average:0.0, $
                   energy_imin:fltarr(2), $
                   n:0.0, $
                   den_patch:0.0, $
                   patch_chisq:0.0,$
                   potential_cushion:0.0, $
                   return_current_error:0.0, $
                   return_current_sec_error:0.0, $
                   n_error:0.0 $
                 }
   
   mom = replicate( data_values, rec_count )
   
   isthere = cdf_varexist(cdf,'time')
   IF (isthere) THEN BEGIN 
      cdf_varget, cdf, 'time',time, $
       rec_start=rec_start, rec_count=rec_count
      mom.time = reform(time)
   ENDIF
   
   isthere = cdf_varexist(cdf,'return_current_error')
   IF (isthere) THEN BEGIN
      cdf_varget, cdf, 'return_current_error',return_current_error, $
       rec_start=rec_start, rec_count=rec_count
      mom.return_current_error = reform(return_current_error)
   ENDIF
   
   isthere = cdf_varexist(cdf,'return_current_sec_error')
   IF (isthere) THEN BEGIN
      cdf_varget, cdf, 'return_current_sec_error',return_current_sec_error, $
       rec_start=rec_start, rec_count=rec_count
      mom.return_current_sec_error = reform(return_current_sec_error)
   ENDIF
   
   isthere = cdf_varexist(cdf,'n_error')
   IF (isthere) THEN BEGIN 
      cdf_varget, cdf, 'n_error',n_error, $
       rec_start=rec_start, rec_count=rec_count
      mom.n_error = reform(n_error)
   ENDIF
   
   isthere = cdf_varexist(cdf,'radius')
   IF (isthere) THEN BEGIN
      cdf_varget, cdf, 'radius',radius, $
       rec_start=rec_start, rec_count=rec_count
      mom.radius = reform(radius)
   ENDIF
   
   isthere = cdf_varexist(cdf,'lshell')
   IF (isthere) THEN BEGIN 
      cdf_varget, cdf, 'lshell',lshell, $
       rec_start=rec_start, rec_count=rec_count
      mom.lshell = reform(lshell)
   ENDIF
   
   isthere = cdf_varexist(cdf,'sc_potential')
   IF (isthere) THEN BEGIN 
      cdf_varget, cdf, 'sc_potential',sc_potential, $
       rec_start=rec_start, rec_count=rec_count
      mom.sc_potential = reform(sc_potential)
   ENDIF
   
   isthere = cdf_varexist(cdf,'background')
   IF (isthere) THEN BEGIN 
      cdf_varget, cdf, 'background',background, $
       rec_start=rec_start, rec_count=rec_count
      mom.background = reform(background)
   ENDIF
   
   isthere = cdf_varexist(cdf, 'return_current_residual')
   IF (isthere) THEN BEGIN
      cdf_varget, cdf, 'return_current_residual',return_current_residual, $
       rec_start=rec_start, rec_count=rec_count
      mom.return_current_residual =  return_current_residual
   ENDIF
   
   isthere = cdf_varexist(cdf, 'return_current_sec_residual')
   IF (isthere) THEN BEGIN 
      cdf_varget,cdf,'return_current_sec_residual',return_current_sec_residual,$
       rec_start=rec_start, rec_count=rec_count
      mom.return_current_sec_residual =  return_current_sec_residual
   ENDIF
   
   isthere = cdf_varexist(cdf, 'return_current')
   IF (isthere) THEN BEGIN
      cdf_varget, cdf, 'return_current',return_current, $
       rec_start=rec_start, rec_count=rec_count
      mom.return_current = reform(return_current)
   ENDIF
   
   isthere = cdf_varexist(cdf, 'rc_patch')
   IF (isthere) THEN BEGIN 
      cdf_varget, cdf, 'rc_patch',rc_patch, $
       rec_start=rec_start, rec_count=rec_count
      mom.rc_patch = reform(rc_patch)
   ENDIF
   
   isthere = cdf_varexist(cdf, 'rc_black')
   IF (isthere) THEN BEGIN 
      cdf_varget, cdf, 'rc_black',rc_black, $
       rec_start=rec_start, rec_count=rec_count
      mom.rc_black = reform(rc_black)
   ENDIF
   
   isthere = cdf_varexist(cdf, 'return_current_sec')
   IF (isthere) THEN BEGIN
      cdf_varget, cdf, 'return_current_sec',return_current_sec, $
       rec_start=rec_start, rec_count=rec_count
      mom.return_current_sec = reform(return_current_sec)
   ENDIF
   
   isthere = cdf_varexist(cdf, 'energy_average')
   IF (isthere) THEN BEGIN 
      cdf_varget, cdf, 'energy_average',energy_average, $
       rec_start=rec_start, rec_count=rec_count
      mom.energy_average = reform(energy_average)
   ENDIF
   
   isthere = cdf_varexist(cdf, 'energy_imin')
   IF (isthere) THEN BEGIN
      cdf_varget, cdf, 'energy_imin',energy_imin, $
       rec_start=rec_start, rec_count=rec_count
      mom.energy_imin =  energy_imin
   ENDIF
   
   isthere = cdf_varexist(cdf, 'n')
   IF (isthere) THEN BEGIN
      cdf_varget, cdf, 'n',n, $
       rec_start=rec_start, rec_count=rec_count
      mom.n = reform(n)
   ENDIF
   
   isthere = cdf_varexist(cdf, 'den_patch')
   IF (isthere) THEN BEGIN
      cdf_varget, cdf, 'den_patch',den_patch, $
       rec_start=rec_start, rec_count=rec_count
      mom.den_patch = reform(den_patch)
   ENDIF
   
   isthere = cdf_varexist(cdf, 'patch_chisq')
   IF (isthere) THEN BEGIN
      cdf_varget, cdf, 'patch_chisq',patch_chisq, $
       rec_start=rec_start, rec_count=rec_count
      mom.patch_chisq = reform(patch_chisq)
   ENDIF
   
   isthere = cdf_varexist(cdf, 'potential_cushion')
   IF (isthere) THEN BEGIN
      cdf_varget, cdf, 'potential_cushion',potential_cushion, $
       rec_start=rec_start, rec_count=1
      mom.potential_cushion =  potential_cushion
   ENDIF
   
   cdf_close,cdf
   
   
END

